package org.codeberg.quecomet.oshi.data.room

import androidx.compose.runtime.Immutable
import androidx.room.ColumnInfo
import androidx.room.Embedded
import androidx.room.Entity
import androidx.room.ForeignKey
import androidx.room.Index
import androidx.room.PrimaryKey
import androidx.room.Relation
import java.util.Date

@Immutable
@Entity(
    indices = [Index("createdAt"), Index("oshiInstanceId")],
    foreignKeys =
        [
            ForeignKey(
                entity = OshiInstance::class,
                childColumns = ["oshiInstanceId"],
                parentColumns = ["id"],
                onDelete = ForeignKey.CASCADE,
                onUpdate = ForeignKey.CASCADE,
            ),
        ],
)
data class UploadWork(
    @PrimaryKey val workUUID: String,
    val filename: String,
    val fileUri: String,
    val oshiInstanceId: Int,
    @ColumnInfo(defaultValue = CURRENT_TIMESTAMP_EXPRESSION) val createdAt: Date = Date(),
    @ColumnInfo(defaultValue = "1440") val expireInMinutes: Int = 1440,
    @ColumnInfo(defaultValue = "0") val destroyAfterDl: Boolean = false,
    @ColumnInfo(defaultValue = "1") val randomizeFilename: Boolean = true,
    @ColumnInfo(defaultValue = "1") val shortenUrl: Boolean = true,
    @ColumnInfo(defaultValue = "1") val canRetry: Boolean = true,
    // the oshi instance ids we tried so far separated by comma (,)
    @ColumnInfo(defaultValue = "") val retriedWithOshiInstanceIds: String,
)

@Immutable
data class UploadWorkWithRetry(
    @PrimaryKey val workUUID: String,
    val canRetry: Boolean,
)

@Immutable
data class PreparedUploadWork(
    val oshiInstanceId: Int,
    val filename: String,
    val fileUri: String,
    val expireInMinutes: Int = 1440,
    val destroyAfterDl: Boolean = false,
    val randomizeFilename: Boolean = true,
    val shortenUrl: Boolean = true,
)

fun UploadWork.toPreparedUploadWork(): PreparedUploadWork {
  return PreparedUploadWork(
      filename = this.filename,
      fileUri = this.fileUri,
      oshiInstanceId = this.oshiInstanceId,
      expireInMinutes = this.expireInMinutes,
      destroyAfterDl = this.destroyAfterDl,
      randomizeFilename = this.randomizeFilename,
      shortenUrl = this.shortenUrl,
  )
}

fun PreparedUploadWork.toUploadWork(workUUID: String): UploadWork {
  return UploadWork(
      workUUID = workUUID,
      filename = this.filename,
      fileUri = this.fileUri,
      oshiInstanceId = this.oshiInstanceId,
      expireInMinutes = this.expireInMinutes,
      destroyAfterDl = this.destroyAfterDl,
      randomizeFilename = this.randomizeFilename,
      shortenUrl = this.shortenUrl,
      retriedWithOshiInstanceIds = "${this.oshiInstanceId}")
}

@Immutable
data class UploadWorkAndOshiInstance(
    @Embedded val uploadWork: UploadWork,
    @Relation(parentColumn = "oshiInstanceId", entityColumn = "id") val oshiInstance: OshiInstance
)
