package org.codeberg.quecomet.oshi.data

import android.util.Log
import androidx.compose.runtime.Stable
import androidx.datastore.core.DataStore
import dagger.Binds
import dagger.Module
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.first
import org.codeberg.quecomet.oshi.PreferredProxyType
import org.codeberg.quecomet.oshi.SSLCheckMode
import org.codeberg.quecomet.oshi.UserSettings
import java.io.IOException
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
@Stable
class UserSettingsRepositoryImpl
@Inject
constructor(
    private val userSettingsStore: DataStore<UserSettings>,
) : UserSettingsRepository {

  companion object {
    const val TAG = "UserSettingsRepository"
  }

  override val userSettingsFlow: Flow<UserSettings> =
      userSettingsStore.data.catch { exception ->
        if (exception is IOException) {
          Log.e(TAG, "Error reading user settings.", exception)
          emit(UserSettings.getDefaultInstance())
        } else {
          throw exception
        }
      }

  override suspend fun fetchSettings() = userSettingsFlow.first()

  override suspend fun saveSettings(userSettings: UserSettings) {
    userSettingsStore.updateData { it.toBuilder().clear().mergeFrom(userSettings).build() }
  }

  override suspend fun setFirstInitDone(value: Boolean) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setFirstInitDone(value).build()
    }
  }

  override suspend fun setIsProxyEnabled(value: Boolean) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setIsProxyEnabled(value).build()
    }
  }

  override suspend fun setProxyHost(value: String) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setProxyHost(value).build()
    }
  }

  override suspend fun setProxyPort(value: String) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setProxyPort(value).build()
    }
  }

  override suspend fun setPreferredProxyType(value: PreferredProxyType) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setPreferredProxyType(value).build()
    }
  }

  override suspend fun setSelectedOshiInstance(value: Int) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setSelectedOshiInstanceId(value).build()
    }
  }

  override suspend fun setSSLCheckMode(value: SSLCheckMode) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setSslCheckMode(value).build()
    }
  }

  override suspend fun setUploadExpiration(value: Int) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setUploadExpiration(value).build()
    }
  }

  override suspend fun setUploadDestroyAfterDL(value: Boolean) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setUploadDestroyAfterDl(value).build()
    }
  }

  override suspend fun setUploadRandomizeName(value: Boolean) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setUploadRandomizeName(value).build()
    }
  }

  override suspend fun setUploadShortenUrl(value: Boolean) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setUploadShortenUrl(value).build()
    }
  }

  override suspend fun setHasShownNotificationPermissionRequest(value: Boolean) {
    userSettingsStore.updateData { currentSettings ->
      currentSettings.toBuilder().setHasShownNotificationPermissionRequest(value).build()
    }
  }
}

@Module
@InstallIn(SingletonComponent::class)
abstract class UserSettingsRepositoryModule {
  @Binds
  @Singleton
  abstract fun provideUserSettingsRepository(
      userSettingsRepository: UserSettingsRepositoryImpl
  ): UserSettingsRepository
}
