package org.codeberg.quecomet.oshi.data

import androidx.compose.runtime.Stable
import androidx.paging.Pager
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import androidx.work.WorkInfo
import dagger.Binds
import dagger.Module
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import kotlinx.coroutines.flow.Flow
import org.codeberg.quecomet.oshi.OshiWorkManager
import org.codeberg.quecomet.oshi.data.room.OshiDatabase
import org.codeberg.quecomet.oshi.data.room.PreparedUploadWork
import org.codeberg.quecomet.oshi.data.room.UploadWork
import org.codeberg.quecomet.oshi.data.room.UploadWorkAndOshiInstance
import org.codeberg.quecomet.oshi.data.room.UploadWorkWithRetry
import org.codeberg.quecomet.oshi.data.room.toUploadWork
import java.util.UUID
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
@Stable
class UploadWorkRepositoryImpl
@Inject
constructor(
    private val oshiDatabase: OshiDatabase,
    private val oshiWorkManager: OshiWorkManager,
) : UploadWorkRepository {
  override suspend fun createWork(preparedUploadWork: PreparedUploadWork) {
    val uploadWork =
        preparedUploadWork.toUploadWork(
            workUUID = UUID.randomUUID().toString(),
        )
    oshiDatabase.uploadWorkDao().insert(uploadWork)
    oshiWorkManager.createUploadWork(uploadWork.workUUID)
  }

  override suspend fun getUploadWork(workUUID: String): UploadWork? {
    return oshiDatabase.uploadWorkDao().uploadWorkByUUID(workUUID)
  }

  override suspend fun getUploadWorkAndOshiInstance(workUUID: String): UploadWorkAndOshiInstance? {
    return oshiDatabase.uploadWorkDao().uploadWorkAndOshiInstanceByUUID(workUUID)
  }

  override fun getUploadWorkAndOshiInstanceList(
      pageSize: Int
  ): Flow<PagingData<UploadWorkAndOshiInstance>> {
    return Pager(
            config = PagingConfig(pageSize = pageSize),
            pagingSourceFactory = {
              oshiDatabase.uploadWorkDao().uploadWorkAndOshiInstanceSortedByCreatedAtPagingSource()
            },
        )
        .flow
  }

  override fun getUploadWorkWorkInfo(workUUID: String?): Flow<WorkInfo?> {
    return oshiWorkManager.getUploadWorkWorkInfo(workUUID)
  }

  override fun retry(workUUID: String) {
    oshiWorkManager.createUploadWork(workUUID)
  }

  override suspend fun retryDifferentInstance(workUUID: String) {
    oshiDatabase.uploadWorkDao().uploadWorkByUUID(workUUID)?.let {
      val idsUsedBefore =
          it.retriedWithOshiInstanceIds.split(',').mapNotNull { numStr -> numStr.toIntOrNull() }
      val newOshiInstanceId =
          oshiDatabase.oshiInstancesDao().getLatestUsedOshiInstanceIdExcept(idsUsedBefore)
      if (newOshiInstanceId != null) {
        oshiDatabase
            .uploadWorkDao()
            .update(
                it.copy(
                    oshiInstanceId = newOshiInstanceId,
                    retriedWithOshiInstanceIds =
                        it.retriedWithOshiInstanceIds + ",$newOshiInstanceId",
                ),
            )
      } else {
        oshiDatabase.oshiInstancesDao().getFirstInstance()?.let { oshiInstance ->
          oshiDatabase
              .uploadWorkDao()
              .update(
                  it.copy(
                      oshiInstanceId = oshiInstance.id,
                      retriedWithOshiInstanceIds = oshiInstance.id.toString(),
                  ),
              )
        }
      }
      this.retry(workUUID)
    }
  }

  override suspend fun changeInstance(workUUID: String, newOshiInstanceId: Int) {
    oshiDatabase.uploadWorkDao().uploadWorkByUUID(workUUID)?.let {
      oshiDatabase.uploadWorkDao().update(it.copy(oshiInstanceId = newOshiInstanceId))
    }
  }

  override suspend fun delete(workUUID: String) {
    oshiWorkManager.cancelWork(workUUID)
    oshiDatabase.uploadWorkDao().delete(workUUID)
  }

  override fun cancel(workUUID: String) {
    oshiWorkManager.cancelWork(workUUID)
  }

  override suspend fun setCanRetry(workUUID: String, canRetry: Boolean) {
    oshiDatabase
        .uploadWorkDao()
        .setCanRetry(UploadWorkWithRetry(workUUID = workUUID, canRetry = canRetry))
  }
}

@Module
@InstallIn(SingletonComponent::class)
abstract class UploadWorkRepositoryModule {
  @Binds
  @Singleton
  abstract fun provideUploadWorkRepositoryModule(
      uploadWorkRepositoryImpl: UploadWorkRepositoryImpl
  ): UploadWorkRepository
}
