package org.codeberg.quecomet.oshi

import android.content.Intent
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.material3.windowsizeclass.ExperimentalMaterial3WindowSizeClassApi
import androidx.compose.material3.windowsizeclass.calculateWindowSizeClass
import androidx.core.splashscreen.SplashScreen.Companion.installSplashScreen
import androidx.lifecycle.lifecycleScope
import dagger.hilt.android.AndroidEntryPoint
import io.ktor.http.URLProtocol
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.data.OshiInstanceRepository
import org.codeberg.quecomet.oshi.data.UserSettingsRepository
import org.codeberg.quecomet.oshi.data.room.OshiInstance
import org.codeberg.quecomet.oshi.ui.components.ProvideSnackbarHostState
import org.codeberg.quecomet.oshi.ui.components.SnackbarMessageHandler
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import javax.inject.Inject

@AndroidEntryPoint
class MainActivity : ComponentActivity() {

  @Inject lateinit var userSettingsRepository: UserSettingsRepository
  @Inject lateinit var oshiInstanceRepository: OshiInstanceRepository

  @Inject lateinit var deepLinkHandler: DeepLinkHandler
  @Inject lateinit var shareIntentHandler: ShareIntentHandler

  @OptIn(ExperimentalMaterial3WindowSizeClassApi::class)
  override fun onCreate(savedInstanceState: Bundle?) {
    // Handle the splash screen transition.
    // must be called before super.onCreate()
    installSplashScreen()

    super.onCreate(savedInstanceState)

    // to preload settings asynchronously, to runBlocking later in di/HttpClientModule
    lifecycleScope.launch {
      val settings = userSettingsRepository.fetchSettings()

      // setup initial settings
      if (!settings.firstInitDone) {
        userSettingsRepository.saveSettings(
            settings.copy {
              firstInitDone = true
              proxyHost = "127.0.0.1"
              proxyPort = "9050"
              uploadShortenUrl = true
              uploadRandomizeName = true
              uploadExpiration = 1440
            },
        )
        oshiInstanceRepository.addInstance(
            OshiInstance(
                protocol = URLProtocol.HTTPS,
                host = "oshi.at",
                label = "oshi.at",
                alternateHost = "hszyoqwrcp7cxlxnqmovp6vjvmnwj33g4wviuxqzq47emieaxjaperyd.onion",
                alternateProtocol = URLProtocol.HTTPS,
            ))
        oshiInstanceRepository.addInstance(
            OshiInstance(
                protocol = URLProtocol.HTTPS,
                host = "hszyoqwrcp7cxlxnqmovp6vjvmnwj33g4wviuxqzq47emieaxjaperyd.onion",
                label = "oshi.at (onion)",
                alternateHost = "oshi.at",
                alternateProtocol = URLProtocol.HTTPS,
            ))

        oshiInstanceRepository.getFirstInstance()?.let {
          userSettingsRepository.setSelectedOshiInstance(it.id)
        }
      }
    }

    setContent {
      OshiTheme {
        val windowSize = calculateWindowSizeClass(this)

        SnackbarMessageHandler()

        ProvideSnackbarHostState() { MainCompose(windowSize, deepLinkHandler) }
      }
    }
  }

  override fun onStart() {
    super.onStart()
    if (intent?.action == Intent.ACTION_SEND || intent?.action == Intent.ACTION_SEND_MULTIPLE) {
      shareIntentHandler.handleShareIntent(intent)
    } else {
      deepLinkHandler.handleDeepLink(intent)
    }
    // consume the deeplink
    intent = null
  }

  override fun onNewIntent(intent: Intent?) {
    super.onNewIntent(intent)
    if (intent?.action == Intent.ACTION_SEND || intent?.action == Intent.ACTION_SEND_MULTIPLE) {
      shareIntentHandler.handleShareIntent(intent)
    } else {
      deepLinkHandler.handleDeepLink(intent)
    }
  }
}
