import {
	Component,
	createRef,
	h,
	render,
} from "./node_modules/preact/dist/preact.module.js";
import {
	BasicInstrument,
	BasicInstrumentZone,
	BasicSample,
	BasicSoundBank,
	BasicPreset,
	BasicPresetZone,
	generatorTypes,
	loadSoundFont,
} from "./node_modules/spessasynth_core/index.js";

/** @type {{ arrayToVector(inputArray: any): any; KeyExtractor(audio: any, averageDetuningCorrection: boolean, frameSize: number, hopSize: number, hpcpSize: number, maxFrequency: number, maximumSpectralPeaks: number, minFrequency: number, pcpThreshold: number, profileType: string, sampleRate: number, spectralPeaksThreshold: number, tuningFrequency: number, weightType: string, windowType: string): { key: string, scale: string, strength: number }; PercivalBpmEstimator(signal: any, frameSize: number, frameSizeOSS: number, hopSize: number, hopSizeOSS: number, maxBPM: number, minBPM: number, sampleRate: number): { bpm: number }; } | undefined} */
let essentia;
// @ts-ignore
EssentiaWASM().then((wasmModule) => {
	essentia = new wasmModule.EssentiaJS(false);
	if (essentia) {
		essentia.arrayToVector = wasmModule.arrayToVector;
	}
});

const audio = new AudioContext({ latencyHint: "playback", sampleRate: 44100 });
const gain = audio.createGain();
gain.gain.value = 1;
const highpass = audio.createBiquadFilter();
highpass.type = "highpass";
highpass.frequency.value = 0;
const lowpass = audio.createBiquadFilter();
lowpass.type = "lowpass";
lowpass.frequency.value = 22050;

const coarse = matchMedia("(pointer: coarse)").matches;

const edgeToEdge =
	// @ts-ignore
	(window.Android && typeof window.Android.sdkInt == "function"
		? // @ts-ignore
			window.Android.sdkInt()
		: 0) >= 35;

/** @template T @param {T | undefined | null} x @returns {T} */
function alwaysTruthy(x) {
	// @ts-ignore
	return x;
}

/** @param {any} x @param {string} name */
function pick(x, name) {
	return x[name];
}

/** @param {{ length: number, value: string }} props */
function ascii({ length, value }) {
	return value.replace(/[^\x00-\x7F]/g, "?").slice(0, length) || "untitled";
}

/** @param {string} path */
function basename(path) {
	return (
		path
			.split("/")
			.at(-1)
			?.replace(/^\.+/, "")
			.replace(/^\d+\. /, "")
			.split(".")
			.at(0) || ""
	);
}

/** @param {number[]} xs */
const sum = (xs) => xs.reduce((prev, x) => prev + x, 0);

/** @param {{ label: string, value: number }} props */
function octave({ label, value }) {
	return [
		{ label: `C${label}`, value: value++ },
		{ label: `C#${label}`, value: value++ },
		{ label: `D${label}`, value: value++ },
		{ label: `D#${label}`, value: value++ },
		{ label: `E${label}`, value: value++ },
		{ label: `F${label}`, value: value++ },
		{ label: `F#${label}`, value: value++ },
		{ label: `G${label}`, value: value++ },
		{ label: `G#${label}`, value: value++ },
		{ label: `A${label}`, value: value++ },
		{ label: `A#${label}`, value: value++ },
		{ label: `B${label}`, value },
	];
}

const notes = [
	...octave({ label: "-2", value: 0 }),
	...octave({ label: "-1", value: 12 }),
	...octave({ label: "0", value: 24 }).filter(({ value }) => value < 35),
	{ label: "B0 Acoustic Bass Drum", value: 35 },
	{ label: "C1 Bass Drum", value: 36 },
	{ label: "C#1 Side Stick", value: 37 },
	{ label: "D1 Acoustic Snare", value: 38 },
	{ label: "D#1 Hand Clap", value: 39 },
	{ label: "E1 Electric Snare", value: 40 },
	{ label: "F1 Low Floor Tom", value: 41 },
	{ label: "F#1 Closed Hi Hat", value: 42 },
	{ label: "G1 High Floor Tom", value: 43 },
	{ label: "G#1 Pedal Hi-Hat", value: 44 },
	{ label: "A1 Low Tom", value: 45 },
	{ label: "A#1 Open Hi-Hat", value: 46 },
	{ label: "B1 Low-Mid Tom", value: 47 },
	{ label: "C2 Hi-Mid Tom", value: 48 },
	{ label: "C#2 Crash Cymbal 1", value: 49 },
	{ label: "D2 High Tom", value: 50 },
	{ label: "D#2 Ride Cymbal 1", value: 51 },
	{ label: "E2 Chinese Cymbal", value: 52 },
	{ label: "F2 Ride Bell", value: 53 },
	{ label: "F#2 Tambourine", value: 54 },
	{ label: "G2 Splash Cymbal", value: 55 },
	{ label: "G#2 Cowbell", value: 56 },
	{ label: "A2 Crash Cymbal 2", value: 57 },
	{ label: "A#2 Vibraslap", value: 58 },
	{ label: "B2 Ride Cymbal 2", value: 59 },
	{ label: "C3 Hi Bongo", value: 60 },
	{ label: "C#3 Low Bongo", value: 61 },
	{ label: "D3 Mute Hi Conga", value: 62 },
	{ label: "D#3 Open Hi Conga", value: 63 },
	{ label: "E3 Low Conga", value: 64 },
	{ label: "F3 Hi Timbale", value: 65 },
	{ label: "F#3 Low Timbale", value: 66 },
	{ label: "G3 Hi Agogo", value: 67 },
	{ label: "G#3 Low Agogo", value: 68 },
	{ label: "A3 Cabasa", value: 69 },
	{ label: "A#3 Maracas", value: 70 },
	{ label: "B3 Short Whistle", value: 71 },
	{ label: "C4 Long Whistle", value: 72 },
	{ label: "C#4 Short Guiro", value: 73 },
	{ label: "D4 Long Guiro", value: 74 },
	{ label: "D#4 Claves", value: 75 },
	{ label: "E4 Hi Wood Block", value: 76 },
	{ label: "F4 Low Wood Block", value: 77 },
	{ label: "F#4 Mute Cuica", value: 78 },
	{ label: "G4 Open Cuica", value: 79 },
	{ label: "G#4 Mute Triangle", value: 80 },
	{ label: "A4 Open Triangle", value: 81 },
	{ label: "A#4", value: 82 },
	{ label: "B4", value: 83 },
	...octave({ label: "5", value: 84 }),
	...octave({ label: "6", value: 96 }),
	...octave({ label: "7", value: 108 }),
	...octave({ label: "8", value: 120 }).filter(({ value }) => value <= 127),
]
	.map((note) => ({
		...note,
		frequency: Math.pow(2, (note.value - 81) / 12) * 440,
		withoutOctave: note.label.replace(/[0-9-].*/, ""),
	}))
	.map((note, index, notes) => ({
		...note,
		major: [
			note,
			notes.at(index + 2),
			notes.at(index + 4),
			notes.at(index + 5),
			notes.at(index + 7),
			notes.at(index + 9),
			notes.at(index + 11),
			notes.at(index + 12),
		],
		minor: [
			note,
			notes.at(index + 2),
			notes.at(index + 3),
			notes.at(index + 5),
			notes.at(index + 7),
			notes.at(index + 8),
			notes.at(index + 10),
			notes.at(index + 12),
		],
	}))
	.map((note) => ({
		...note,
		major: note.major.every(Boolean) ? note.major.map(alwaysTruthy) : undefined,
		minor: note.minor.every(Boolean) ? note.minor.map(alwaysTruthy) : undefined,
	}));

/** @param {{ value: boolean }} props */
function BooleanDebug({ value }) {
	return h("input", { readOnly: true, value: String(value) });
}

/** @param {{ value: number }} props */
function NumberDebug({ value }) {
	return h("input", { readOnly: true, size: 8, value: String(value) });
}

/** @param {{ value: string }} props */
function TextDebug({ value }) {
	return h("input", { readOnly: true, value: String(value) });
}

/** @param {{ value: object }} props */
function Debug({ value }) {
	return h(
		"ul",
		null,
		Object.keys(value).map((key) => h("li", null, key)),
	);
}

/** @param {{ isPercussion?: boolean, makeKeyRangeSubsequent?: () => void, onKeyRangeMaxChange?: (event: { currentTarget: HTMLSelectElement }) => void, onKeyRangeMinChange?: (event: { currentTarget: HTMLSelectElement }) => void, zone: BasicInstrumentZone | BasicPresetZone }} props */
function Zone({
	isPercussion,
	makeKeyRangeSubsequent,
	onKeyRangeMaxChange,
	onKeyRangeMinChange,
	zone,
}) {
	const noteOptions = [
		{ label: "-1", value: -1 },
		...notes.map(
			isPercussion
				? (x) => x
				: ({ label, value }) => ({ label: label.split(" ")[0], value }),
		),
	];
	return h(
		"div",
		null,
		!onKeyRangeMinChange // Global zone.
			? undefined
			: h(
					"div",
					{ className: "form-group" },
					h("label", null, "velRange.min"),
					h(NumberDebug, { value: zone.velRange.min }),
				),
		!onKeyRangeMinChange // Global zone.
			? undefined
			: zone.velRange.min == -1
				? undefined
				: h(
						"div",
						{ className: "form-group" },
						h("label", null, "velRange.max"),
						h(NumberDebug, { value: zone.velRange.max }),
					),
		!onKeyRangeMinChange
			? undefined
			: h(
					"div",
					{ className: "form-group" },
					h("label", null, "keyRange.min"),
					h(
						"select",
						{ onChange: onKeyRangeMinChange },
						noteOptions.map(({ label, value }) =>
							h(
								"option",
								{ selected: zone.keyRange.min == value, value },
								label,
							),
						),
					),
					!makeKeyRangeSubsequent || zone.keyRange.min == -1
						? undefined
						: h(
								"button",
								{ onClick: makeKeyRangeSubsequent, type: "button" },
								"makeKeyRangeSubsequent",
							),
				),
		zone.keyRange.min == -1 || !onKeyRangeMaxChange
			? undefined
			: h(
					"div",
					{ className: "form-group" },
					h("label", null, "keyRange.max"),
					h(
						"select",
						{ onChange: onKeyRangeMaxChange },
						noteOptions.map(({ label, value }) =>
							h(
								"option",
								{ selected: zone.keyRange.max == value, value },
								label,
							),
						),
					),
				),
		Object.keys(generatorTypes)
			.map((name) => ({ name, type: pick(generatorTypes, name) }))
			.filter(
				(summary) =>
					summary.type != generatorTypes.sampleID &&
					summary.type != generatorTypes.instrument,
			)
			.map((summary) => {
				const generator = zone.generators.find(
					({ generatorType }) => generatorType == summary.type,
				);
				return generator ? { generator, summary } : undefined;
			})
			.filter(Boolean)
			.map(alwaysTruthy)
			.map(({ generator, summary }) =>
				h(
					"div",
					{ className: "form-group" },
					h("label", null, summary.name),
					h(NumberDebug, {
						value: generator.generatorValue,
					}),
				),
			),
		zone.generators
			.filter(
				({ generatorType }) =>
					!Object.keys(generatorTypes).some(
						(name) => pick(generatorTypes, name) == generatorType,
					),
			)
			.map((generator) =>
				h(
					"div",
					{ className: "form-group" },
					h(NumberDebug, {
						value: generator.generatorType,
					}),
					h(NumberDebug, {
						value: generator.generatorValue,
					}),
				),
			),
		zone.modulators.map((modulator) =>
			h(
				"div",
				{ className: "form-group" },
				h(
					"label",
					null,
					Object.keys(generatorTypes).find(
						(name) =>
							pick(generatorTypes, name) == modulator.modulatorDestination,
					),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "modulatorDestination"),
					h(NumberDebug, { value: modulator.modulatorDestination }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "transformAmount"),
					h(NumberDebug, { value: modulator.transformAmount }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "transformType"),
					h(NumberDebug, { value: modulator.transformType }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "isEffectModulator"),
					h(BooleanDebug, { value: modulator.isEffectModulator }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "sourceCurveType"),
					h(NumberDebug, { value: modulator.sourceCurveType }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "sourcePolarity"),
					h(NumberDebug, { value: modulator.sourcePolarity }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "sourceDirection"),
					h(NumberDebug, { value: modulator.sourceDirection }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "sourceUsesCC"),
					h(NumberDebug, { value: modulator.sourceUsesCC }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "sourceIndex"),
					h(NumberDebug, { value: modulator.sourceIndex }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "secSrcCurveType"),
					h(NumberDebug, { value: modulator.secSrcCurveType }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "secSrcPolarity"),
					h(NumberDebug, { value: modulator.secSrcPolarity }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "secSrcDirection"),
					h(NumberDebug, { value: modulator.secSrcDirection }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "secSrcUsesCC"),
					h(NumberDebug, { value: modulator.secSrcUsesCC }),
				),
				h(
					"div",
					{ className: "form-group" },
					h("label", null, "secSrcIndex"),
					h(NumberDebug, { value: modulator.secSrcIndex }),
				),
			),
		),
	);
}

class App extends Component {
	/** @type {{ current: HTMLCanvasElement | null }} */
	canvas = createRef();

	/** @type {{ current: HTMLElement | null }} */
	main = createRef();

	/** @type {{ current: MediaRecorder | null }} */
	recorder = createRef();

	/** @type {{ current: Element | null }} */
	resizingZoneEnd = createRef();

	/** @type {{ current: Element | null }} */
	resizingZoneStart = createRef();

	/** @type {{ current: AudioBufferSourceNode | null }} */
	source = createRef();

	state = {
		analyzing: false,
		bpm: 0,
		/** @type {WeakMap<Float32Array, { bpm: number, keyData: { key: string, scale: string } | undefined }>} */
		bpmKeyCache: new WeakMap(),
		currentInstrument: 0,
		currentInstrumentZone: 0,
		currentPreset: 0,
		currentPresetZone: 0,
		currentSample: 0,
		dataPoint: 0,
		detune: 0,
		font: (() => {
			const font = new BasicSoundBank();
			font.soundFontInfo["ifil"] = "2.01";
			font.soundFontInfo["isng"] = "EMU8000";
			return font;
		})(),
		/** @type {{ key: string, scale: string } | undefined} */
		keyData: undefined,
		playbackInterval: 0,
		recommendedGain: 0,
	};

	get detuneRate() {
		return Math.pow(2, this.state.detune / 12);
	}

	async componentDidMount() {
		addEventListener("keydown", this.onKeyDown);
		if (coarse) {
			addEventListener("touchstart", this.onTouchStart);
			addEventListener("touchend", this.onMouseUp);
			addEventListener("touchcancel", this.onMouseUp);
			addEventListener("touchmove", this.onTouchMove);
		} else {
			addEventListener("mousedown", this.onMouseDown);
			addEventListener("mouseup", this.onMouseUp);
			addEventListener("mousemove", this.onMouseMove);
		}

		// Debug option 1: pre-recorded sample.
		// const url = "./slow funk.mp3";
		// this.state.font.soundFontInfo["INAM"] = basename(url);
		// await this.pushSample(
		// 	this.state.font.soundFontInfo["INAM"],
		// 	await (await fetch(url)).arrayBuffer(),
		// );

		// Debug option 2: existing soundfont.
		// this.setState({
		// 	font: loadSoundFont(await (await fetch("./TimGM6mb.sf2")).arrayBuffer()),
		// });
		// this.forceUpdate();
		// setTimeout(() => {
		// 	this.draw();
		// });
	}

	componentWillUnmount() {
		removeEventListener("keydown", this.onKeyDown);
		if (coarse) {
			removeEventListener("touchstart", this.onTouchStart);
			removeEventListener("touchend", this.onMouseUp);
			removeEventListener("touchcancel", this.onMouseUp);
			removeEventListener("touchmove", this.onTouchMove);
		} else {
			removeEventListener("mousedown", this.onMouseDown);
			removeEventListener("mouseup", this.onMouseUp);
			removeEventListener("mousemove", this.onMouseMove);
		}
	}

	/** @param {{ currentTarget: HTMLInputElement }} event */
	onSoundFontInfoInput = (event) => {
		this.state.font.soundFontInfo[event.currentTarget.name] = ascii({
			length: 255,
			value: event.currentTarget.value,
		});
	};

	draw = () => {
		const sample = this.state.font.samples.at(this.state.currentSample);
		if (sample) {
			const data = sample.getAudioData();
			if (this.canvas.current) {
				const context = this.canvas.current.getContext("2d");
				if (context) {
					const { height, width } = this.canvas.current;
					context.clearRect(0, 0, width, height);
					context.fillStyle = "red";
					for (let i = 0; i < width; i++) {
						const min = Math.floor((data.length / width) * i);
						const max = Math.floor((data.length / width) * (i + 1));
						/** @type {Map<number, number>} */
						const occurrences = new Map();
						for (let j = min; j < max; j++) {
							const value = data[Math.min(j, data.length - 1)];
							occurrences.set(value, (occurrences.get(value) || 0) + 1);
						}
						context.fillRect(
							i,
							height / 2,
							1,
							Math.floor(
								([...occurrences.entries()]
									.map(([value, count]) => ({ count, value }))
									.sort((a, b) => b.count - a.count)
									.at(0)?.value || 0) * height,
							),
						);
					}
				}
			}
			this.setRecommendedGain();
			this.setBpmAndKey();
		}
	};

	stop = () => {
		if (this.state.playbackInterval) {
			clearInterval(this.state.playbackInterval);
		}
		if (this.source.current) {
			this.source.current.stop();
		}
		this.source.current = null;
		const zone = this.state.font.instruments
			.at(this.state.currentInstrument)
			?.instrumentZones.at(this.state.currentInstrumentZone);
		this.setState({
			dataPoint: zone
				? zone.getGeneratorValue(generatorTypes.startAddrsCoarseOffset, 0) *
						32768 +
					zone.getGeneratorValue(generatorTypes.startAddrsOffset, 0)
				: this.zones.at(0)?.start || 0,
			playbackInterval: undefined,
		});
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	importFont = (event) => {
		this.stop();
		const reader = new FileReader();
		reader.onload = async () => {
			if (reader.result && typeof reader.result != "string") {
				let font;
				try {
					font = loadSoundFont(reader.result);
				} catch (/** @type {any} */ error) {
					console.error(error);
					alert(error?.message || "error");
					this.setState({ analyzing: false });
				}
				if (font) {
					this.setState({
						bpm: 0,
						currentInstrument: 0,
						currentInstrumentZone: 0,
						currentPreset: 0,
						currentPresetZone: 0,
						currentSample: 0,
						dataPoint: 0,
						font,
						keyData: undefined,
					});
					setTimeout(() => {
						this.draw();
					});
				}
			}
		};
		if (event.currentTarget.files?.length) {
			this.setState({ analyzing: true });
			reader.readAsArrayBuffer(event.currentTarget.files[0]);
		}
	};

	/** @param {string} name @param {ArrayBuffer} arrayBuffer */
	pushSample = async (name, arrayBuffer) => {
		const audioData = await audio.decodeAudioData(arrayBuffer);
		const sample = new BasicSample(
			ascii({ length: 20, value: name }),
			audioData.sampleRate,
			60, // Middle C.
			0, // No pitch correction.
			1, // Mono.
			// Unlooped.
			0,
			0,
		);
		const left = audioData.getChannelData(0);
		if (audioData.numberOfChannels == 2) {
			const right = audioData.getChannelData(1);
			for (let i = 0; i < Math.min(left.length, right.length); i++) {
				left[i] += right[i];
				left[i] /= 2;
			}
		}
		sample.setAudioData(left);
		this.state.font.samples.push(sample);
		if (!this.state.font.soundFontInfo["INAM"]) {
			this.state.font.soundFontInfo["INAM"] = ascii({
				length: 255,
				value: name,
			});
		}
		this.setState({
			currentSample: this.state.font.samples.length - 1,
			// Rewind, assuming you called stop before this method,
			// landing the cursor at the start of the first zone.
			dataPoint: 0,
		});
		setTimeout(() => {
			this.draw();
		});
	};

	record = async () => {
		if (!this.recorder.current) {
			const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
			const recorder = new MediaRecorder(stream);
			this.recorder.current = recorder;
			/** @type {Blob[]} */
			const chunks = [];
			recorder.addEventListener("dataavailable", ({ data }) => {
				chunks.push(data);
			});
			recorder.addEventListener("stop", async () => {
				this.stop();
				this.setState({ analyzing: true });
				await this.pushSample(
					"untitled",
					await new Blob(chunks, { type: recorder.mimeType }).arrayBuffer(),
				);
				chunks.splice(0);
			});
		}
		const recorder = this.recorder.current;
		if (recorder.state == "recording") {
			recorder.stop();
		} else {
			recorder.start();
			this.forceUpdate();
		}
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	importSample = (event) => {
		this.stop();
		const file = event.currentTarget.files?.length
			? event.currentTarget.files[0]
			: undefined;
		if (file) {
			const reader = new FileReader();
			reader.onload = async () => {
				if (reader.result && typeof reader.result != "string") {
					try {
						await this.pushSample(basename(file.name), reader.result);
						if (!this.state.font.soundFontInfo["INAM"]) {
							this.state.font.soundFontInfo["INAM"] = ascii({
								length: 255,
								value: basename(file.name),
							});
						}
					} catch (/** @type {any} */ error) {
						alert(error?.message || "error");
						this.setState({ analyzing: false });
					}
				}
			};
			this.setState({ analyzing: true });
			reader.readAsArrayBuffer(file);
		}
	};

	setRecommendedGain() {
		let max = Number.MIN_VALUE;
		for (const value of this.state.font.samples
			.at(this.state.currentSample)
			?.getAudioData() || []) {
			if (Math.abs(value) > max) {
				max = Math.abs(value);
			}
		}
		this.setState({
			recommendedGain:
				max == Number.MIN_VALUE ? 1 : Math.floor((1 - max + 1) * 10) / 10,
		});
	}

	normalize = () => {
		gain.gain.value = this.state.recommendedGain;
		this.forceUpdate();
		setTimeout(() => {
			if (this.main.current) {
				for (const input of [...this.main.current.querySelectorAll("input")]) {
					if (input.defaultValue) {
						input.value = input.defaultValue;
					}
				}
			}
		});
	};

	/** @param {{ seconds: number }} props */
	getOffline = ({ seconds } = { seconds: -1 }) => {
		let offline;
		let source;
		const sample = this.state.font.samples.at(this.state.currentSample);
		if (sample) {
			let data = sample.getAudioData();
			if (seconds != -1) {
				data = data.slice(0, sample.sampleRate * seconds);
			}
			const buffer = new AudioBuffer({
				length: data.length,
				numberOfChannels: 1,
				sampleRate: sample.sampleRate,
			});
			buffer.copyToChannel(data, 0);
			offline = new OfflineAudioContext({
				length: Math.ceil(data.length / this.detuneRate),
				numberOfChannels: buffer.numberOfChannels,
				sampleRate: sample.sampleRate,
			});
			source = new AudioBufferSourceNode(offline, { buffer });
		}
		return { offline, sample, source };
	};

	setBpmAndKey = () => {
		const sample = this.state.font.samples.at(this.state.currentSample);
		if (sample) {
			const data = sample.getAudioData();
			const cache = this.state.bpmKeyCache.get(data);
			if (cache) {
				this.setState({
					analyzing: false,
					bpm: cache.bpm,
					keyData: cache.keyData,
				});
			} else if (data.length < sample.sampleRate * 4) {
				this.setState({ analyzing: false, bpm: 0, keyData: undefined });
			} else if (essentia) {
				// Downsample to 16kHz for essentia tensorflow models.
				const original = data.slice(0, sample.sampleRate * 60);
				const sampleRate = 16000;
				const sampleRateRatio = sample.sampleRate / sampleRate;
				const downsampled = new Float32Array(
					Math.round(original.length / sampleRateRatio),
				);
				let offsetResult = 0;
				let offsetAudioIn = 0;
				while (offsetResult < downsampled.length) {
					let nextOffsetAudioIn = Math.round(
						(offsetResult + 1) * sampleRateRatio,
					);
					let accum = 0,
						count = 0;
					for (
						let i = offsetAudioIn;
						i < nextOffsetAudioIn && i < original.length;
						i++
					) {
						accum += original[i];
						count++;
					}
					downsampled[offsetResult] = accum / count;
					offsetResult++;
					offsetAudioIn = nextOffsetAudioIn;
				}
				const vectorSignal = essentia.arrayToVector(downsampled);
				const bpm = Math.round(
					essentia.PercivalBpmEstimator(
						vectorSignal,
						1024,
						2048,
						128,
						128,
						210,
						50,
						sampleRate,
					).bpm,
				);
				const keyData = essentia.KeyExtractor(
					vectorSignal,
					true,
					4096,
					4096,
					12,
					3500,
					60,
					25,
					0.2,
					"bgate",
					sampleRate,
					0.0001,
					440,
					"cosine",
					"hann",
				);
				this.state.bpmKeyCache.set(data, { bpm, keyData });
				this.setState({ analyzing: false, bpm, keyData });
			} else {
				this.setState({ analyzing: false });
			}
		}
	};

	play = () => {
		const sample = this.state.font.samples.at(this.state.currentSample);
		if (sample) {
			const data = sample.getAudioData();
			const buffer = new AudioBuffer({
				length: data.length,
				numberOfChannels: 1,
				sampleRate: sample.sampleRate,
			});
			const zone = this.state.font.instruments
				.at(this.state.currentInstrument)
				?.instrumentZones.at(this.state.currentInstrumentZone);
			const offset =
				(zone?.getGeneratorValue(generatorTypes.startAddrsCoarseOffset, 0) ||
					0) *
					32768 +
				(zone?.getGeneratorValue(generatorTypes.startAddrsOffset, 0) || 0);
			const end =
				data.length -
				1 +
				(zone?.getGeneratorValue(generatorTypes.endAddrsCoarseOffset, 0) || 0) *
					32768 +
				(zone?.getGeneratorValue(generatorTypes.endAddrOffset, 0) || 0);
			buffer.copyToChannel(data, 0);
			const source = new AudioBufferSourceNode(audio, { buffer });
			source.detune.value = this.state.detune * 100;
			source.connect(highpass);
			highpass.connect(lowpass);
			lowpass.connect(gain);
			gain.connect(audio.destination);
			source.start(0, offset / sample.sampleRate);
			if (this.state.playbackInterval) {
				clearInterval(this.state.playbackInterval);
			}
			if (this.source.current) {
				this.source.current.stop();
			}
			this.source.current = source;
			this.setState({
				playbackInterval: setInterval(() => {
					const dataPoint = Math.min(
						end,
						this.state.dataPoint +
							Math.floor((sample.sampleRate * this.detuneRate) / 30),
					);
					if (dataPoint >= end) {
						this.stop();
					} else {
						this.setState({ dataPoint });
					}
				}, 1000 / 30),
				dataPoint: offset,
			});
		}
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onSampleChange = (event) => {
		const currentSample = Number(event.currentTarget.value);
		const sample = this.state.font.samples.at(currentSample);
		const instrument = this.state.font.instruments
			.map((instrument, instrumentIndex) => ({
				instrumentIndex,
				zoneIndex: instrument.instrumentZones.findIndex(
					(zone) => zone.sample == sample,
				),
			}))
			.find(({ zoneIndex }) => zoneIndex != -1);
		const currentInstrument = instrument ? instrument.instrumentIndex : 0;
		const currentInstrumentZone = instrument ? instrument.zoneIndex : 0;
		this.setState({ currentInstrument, currentInstrumentZone, currentSample });
		setTimeout(() => {
			this.draw();
			this.play();
		});
	};

	get zones() {
		const sample = this.state.font.samples.at(this.state.currentSample);
		const zones = [];
		if (sample) {
			for (const instrument of this.state.font.instruments) {
				for (const zone of instrument.instrumentZones) {
					if (zone.sample == sample) {
						zones.push({
							end:
								zone.getGeneratorValue(generatorTypes.endAddrsCoarseOffset, 0) *
									32768 +
								zone.getGeneratorValue(generatorTypes.endAddrOffset, 0),
							start:
								zone.getGeneratorValue(
									generatorTypes.startAddrsCoarseOffset,
									0,
								) *
									32768 +
								zone.getGeneratorValue(generatorTypes.startAddrsOffset, 0),
							zone,
						});
					}
				}
			}
			zones.sort((a, b) => a.start - b.start);
		}
		return zones;
	}

	get instrument() {
		return this.state.font.instruments.find(
			(instrument) =>
				!instrument.instrumentZones.length ||
				instrument.instrumentZones.find(
					(zone) =>
						zone.sample == this.state.font.samples.at(this.state.currentSample),
				),
		);
	}

	chop = () => {
		let instrument = this.instrument;
		const shouldPushInstrument = !instrument;
		const sample = this.state.font.samples.at(this.state.currentSample);
		if (!sample) {
			return;
		}
		if (!instrument) {
			instrument = new BasicInstrument();
			instrument.instrumentName = sample.sampleName;
		}
		const shouldPlay = !instrument.instrumentZones.length;
		const zone = instrument.createZone();
		zone.setSample(sample);
		zone.setGenerator(
			generatorTypes.startAddrsCoarseOffset,
			Math.floor(this.state.dataPoint / 32768),
		);
		zone.setGenerator(
			generatorTypes.startAddrsOffset,
			this.state.dataPoint % 32768,
		);
		if (shouldPushInstrument) {
			this.state.font.instruments.push(instrument);
		}
		let zones = this.zones;
		const index = zones.findIndex((x) => x.zone == zone);
		const data = sample.getAudioData();
		const previous = index >= 1 ? zones.at(index - 1) : undefined;
		const previousEnd = previous ? previous.end : undefined;
		if (previous) {
			if (
				!previous.end ||
				data.length - 1 + previous.end >= this.state.dataPoint
			) {
				const end = -(data.length - 1 - (this.state.dataPoint - 1));
				previous.zone.setGenerator(
					generatorTypes.endAddrsCoarseOffset,
					Math.ceil(end / 32768),
				);
				previous.zone.setGenerator(generatorTypes.endAddrOffset, end % 32768);
			}
		}
		if (index < zones.length - 1) {
			const next = zones[index + 1];
			const end =
				previousEnd && this.state.dataPoint < data.length - 1 + previousEnd
					? previousEnd
					: -(data.length - 1 - (next.start - 1));
			zone.setGenerator(
				generatorTypes.endAddrsCoarseOffset,
				Math.ceil(end / 32768),
			);
			zone.setGenerator(generatorTypes.endAddrOffset, end % 32768);
		}
		let key;
		if (zones.length == 1) {
			key = 36;
		} else if (
			previous &&
			previous.zone.keyRange.min != 1 &&
			previous.zone.keyRange.min == previous.zone.keyRange.max
		) {
			key = previous.zone.keyRange.min;
		}
		const otherZones = zones.filter((x) => x.zone != zone);
		for (const { zone } of otherZones) {
			if (zone.keyRange.min == -1 || zone.keyRange.max == -1) {
				key = undefined;
			} else if (key) {
				while (zone.keyRange.min >= key && zone.keyRange.max <= key) {
					key++;
				}
			}
		}
		if (key && key <= 127) {
			zone.setGenerator(generatorTypes.overridingRootKey, key);
			zone.keyRange.min = key;
			zone.keyRange.max = key;
		}
		this.setState({
			currentInstrument: this.state.font.instruments.indexOf(instrument),
			currentInstrumentZone: instrument.instrumentZones.indexOf(zone),
		});
		let preset = this.state.font.presets.find(
			(preset) =>
				!preset.presetZones.length ||
				preset.presetZones.find((zone) => zone.instrument == instrument),
		);
		const shouldPushPreset = !preset;
		if (!preset) {
			preset = new BasicPreset(this.state.font);
			preset.bank = 128;
			for (const index of [...Array.from({ length: 128 })].map(
				(_, index) => index,
			)) {
				if (
					!this.state.font.presets.find(
						({ bank, program }) => bank == preset?.bank && program == index,
					)
				) {
					preset.program = index;
					break;
				}
			}
			preset.presetName = instrument.instrumentName;
		}
		if (!preset.presetZones.length) {
			const zone = preset.createZone();
			zone.setInstrument(instrument);
		}
		if (shouldPushPreset) {
			this.state.font.presets.push(preset);
		}
		this.setState({
			currentPreset: this.state.font.presets.indexOf(preset),
		});
		if (shouldPlay) {
			this.play();
		}
	};

	/** @param {{ preventDefault: () => void, stopPropagation: () => void }} event */
	onCreateZoneTouchStart = (event) => {
		event.preventDefault();
		event.stopPropagation();
		this.chop();
	};

	/** @param {KeyboardEvent} event */
	onKeyDown = (event) => {
		let value;
		switch (event.key) {
			case "q":
				this.stop();
				return;
			case "e":
				this.trimEnd();
				return;
			case "a":
				value = 36;
				break;
			case "s":
				value = 37;
				break;
			case "d":
				value = 38;
				break;
			case "f":
				value = 39;
				break;
			case "g":
				value = 40;
				break;
			case "h":
				value = 41;
				break;
			case "j":
				value = 42;
				break;
			case "k":
				value = 43;
				break;
			case "l":
				value = 44;
				break;
			case ";":
				value = 45;
				break;
			case "'":
				value = 46;
				break;
			case "\\":
				value = 47;
				break;
			case "x":
				this.deleteInstrumentZone();
				return;
			case "b":
				this.trimStart();
				return;
		}
		const sample = this.state.font.samples.at(this.state.currentSample);
		if (
			value &&
			sample &&
			!this.main.current?.querySelector(
				"input:not([type=file]):focus, select:focus",
			)
		) {
			event.preventDefault();
			event.stopPropagation();
			const instrument = this.state.font.instruments.find(
				({ instrumentZones }) =>
					instrumentZones.some((zone) => zone.sample == sample),
			);
			const currentInstrumentZone = instrument?.instrumentZones.findIndex(
				({ keyRange }) => keyRange.min == value && keyRange.max == value,
			);
			if (
				instrument &&
				typeof currentInstrumentZone == "number" &&
				currentInstrumentZone != -1
			) {
				this.setState({
					currentInstrument: this.state.font.instruments.indexOf(instrument),
					currentInstrumentZone,
				});
				setTimeout(() => {
					this.play();
				});
			} else if (
				!instrument ||
				!instrument.instrumentZones.find(
					(zone) =>
						this.state.dataPoint ==
						zone.getGeneratorValue(generatorTypes.startAddrsCoarseOffset, 0) *
							32768 +
							zone.getGeneratorValue(generatorTypes.startAddrsOffset, 0),
				)
			) {
				this.chop();
				setTimeout(() => {
					const zone = this.state.font.instruments
						.at(this.state.currentInstrument)
						?.instrumentZones.at(this.state.currentInstrumentZone);
					if (zone) {
						zone.setGenerator(generatorTypes.overridingRootKey, value);
						zone.keyRange.min = value;
						zone.keyRange.max = value;
						this.forceUpdate();
					}
				});
			}
		}
	};

	/** @param {{ clientX: number, target: EventTarget | null }} event */
	onMouseDown = (event) => {
		if (
			event.target instanceof Element &&
			event.target.className.indexOf("zone-cursor") != -1
		) {
			const cursor = event.target;
			const relative = cursor.parentElement;
			if (relative) {
				const rect = cursor.getBoundingClientRect();
				if (
					event.clientX >= rect.x &&
					event.clientX < rect.x + rect.width / 2
				) {
					this.resizingZoneStart.current = cursor;
				} else {
					this.resizingZoneEnd.current = cursor;
				}
				const index = [...relative.children]
					.filter((x) => x.className.indexOf("zone-cursor") != -1)
					.indexOf(cursor);
				if (index != -1) {
					const current = this.zones.at(index)?.zone;
					if (current) {
						for (const instrument of this.state.font.instruments) {
							for (const zone of instrument.instrumentZones) {
								if (zone == current) {
									this.setState({
										currentInstrument:
											this.state.font.instruments.indexOf(instrument),
										currentInstrumentZone:
											instrument.instrumentZones.indexOf(zone),
									});
									setTimeout(() => {
										this.play();
									});
								}
							}
						}
					}
				}
			}
		} else if (event.target instanceof HTMLCanvasElement) {
			const rect = event.target.getBoundingClientRect();
			if (event.clientX >= rect.x && event.clientX < rect.x + rect.width) {
				const sample = this.state.font.samples.at(this.state.currentSample);
				if (sample) {
					this.stop();
					this.setState({
						dataPoint: Math.floor(
							((event.clientX - rect.x) / rect.width) *
								sample.getAudioData().length,
						),
					});
					setTimeout(() => {
						const createZoneWillNotPlayByItself =
							this.instrument?.instrumentZones.length;
						this.chop();
						if (createZoneWillNotPlayByItself) {
							setTimeout(() => {
								this.play();
							});
						}
					});
				}
			}
		} else if (event.target instanceof HTMLSelectElement) {
			this.stop();
		}
	};

	/** @param {TouchEvent} event */
	onTouchStart = (event) => {
		if (event.touches.length == 1 && event.target instanceof Element) {
			this.onMouseDown({
				clientX: event.touches[0].clientX,
				target: event.target,
			});
		}
	};

	onMouseUp = () => {
		this.resizingZoneStart.current = null;
		this.resizingZoneEnd.current = null;
	};

	/** @param {{ clientX: number }} event */
	onMouseMove = (event) => {
		const cursor =
			this.resizingZoneStart.current || this.resizingZoneEnd.current;
		if (
			cursor &&
			!(this.resizingZoneStart.current && this.resizingZoneEnd.current)
		) {
			const relative = cursor.parentElement;
			if (relative) {
				const relativeRect = relative.getBoundingClientRect();
				if ((event.clientX < 0 || event.clientX >= outerWidth) && !coarse) {
					this.onMouseUp();
				} else if (
					event.clientX >= relativeRect.x &&
					event.clientX < relativeRect.x + relativeRect.width
				) {
					const rate = (event.clientX - relativeRect.x) / relativeRect.width;
					const sample = this.state.font.samples.at(this.state.currentSample);
					if (sample) {
						const data = sample.getAudioData();
						const currentInstrumentZone = [...relative.children]
							.filter((x) => x.className.indexOf("zone-cursor") != -1)
							.indexOf(cursor);
						const zones = this.zones;
						const current = zones.at(currentInstrumentZone);
						if (currentInstrumentZone != -1 && current) {
							const prev =
								currentInstrumentZone >= 1
									? zones.at(currentInstrumentZone - 1)
									: undefined;
							const next = zones.at(currentInstrumentZone + 1);
							const minimumLength = data.length / 32;
							if (this.resizingZoneStart.current) {
								const start = Math.max(
									prev ? data.length + prev.end + 1 : 0,
									Math.floor(rate * data.length),
								);
								if (data.length + current.end - start >= minimumLength) {
									current.zone.setGenerator(
										generatorTypes.startAddrsCoarseOffset,
										Math.floor(start / 32768),
									);
									current.zone.setGenerator(
										generatorTypes.startAddrsOffset,
										start % 32768,
									);
									this.forceUpdate();
								}
							} else if (this.resizingZoneEnd.current) {
								const end = Math.min(
									-Math.floor((1 - rate) * data.length),
									next ? -(data.length - next.start - 1) : 0,
								);
								if (data.length + end - current.start >= minimumLength) {
									current.zone.setGenerator(
										generatorTypes.endAddrsCoarseOffset,
										Math.ceil(end / 32768),
									);
									current.zone.setGenerator(
										generatorTypes.endAddrOffset,
										end % 32768,
									);
									this.forceUpdate();
								}
							}
						}
					}
				}
			}
		}
	};

	/** @param {TouchEvent} event */
	onTouchMove = (event) => {
		if (event.touches.length == 1) {
			this.onMouseMove({ clientX: event.touches[0].clientX });
		}
	};

	trimStart = () => {
		const zones = this.zones;
		const start = zones.at(0);
		if (start) {
			const offset = start.start;
			/** @type {BasicSample} */
			const sample = start.zone.sample;
			sample.setAudioData(sample.getAudioData().slice(start.start));
			for (const zone of zones) {
				zone.start -= offset;
				zone.zone.setGenerator(
					generatorTypes.startAddrsCoarseOffset,
					Math.floor(zone.start / 32768),
				);
				zone.zone.setGenerator(
					generatorTypes.startAddrsOffset,
					zone.start % 32768,
				);
			}
			this.stop();
			if (sample.getAudioData().length >= sample.sampleRate * 4) {
				this.setState({ analyzing: true });
			}
			setTimeout(() => {
				this.draw();
			});
		}
	};

	trimEnd = () => {
		const zones = this.zones;
		const end = zones.at(-1);
		if (end?.end) {
			/** @type {BasicSample} */
			const sample = end.zone.sample;
			const data = sample.getAudioData();
			const offset = end.end;
			for (const zone of zones) {
				zone.end = Math.min(zone.end - offset, 0);
				zone.zone.setGenerator(
					generatorTypes.endAddrsCoarseOffset,
					Math.ceil(zone.end / 32768),
				);
				zone.zone.setGenerator(generatorTypes.endAddrOffset, zone.end % 32768);
			}
			sample.setAudioData(data.slice(0, offset || data.length));
			this.stop();
			if (data.length >= sample.sampleRate * 4) {
				this.setState({ analyzing: true });
			}
			setTimeout(() => {
				this.draw();
			});
		}
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	onSampleNameInput = (event) => {
		const sample = this.state.font.samples.at(this.state.currentSample);
		if (sample) {
			sample.sampleName = ascii({
				length: 20,
				value: event.currentTarget.value,
			});
			this.forceUpdate();
		}
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	onDetuneInput = (event) => {
		if (event.currentTarget.validity.valid) {
			const detune = Number(event.currentTarget.value);
			this.setState({ detune });
			if (this.source.current) {
				this.source.current.detune.value = detune * 100;
			}
		}
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	onHighpassInput = (event) => {
		if (event.currentTarget.validity.valid) {
			highpass.frequency.value = Number(event.currentTarget.value);
			this.forceUpdate();
		}
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	onLowpassInput = (event) => {
		if (event.currentTarget.validity.valid) {
			lowpass.frequency.value = Number(event.currentTarget.value);
			this.forceUpdate();
		}
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	onGainInput = (event) => {
		if (event.currentTarget.validity.valid) {
			gain.gain.value = Number(event.currentTarget.value.replace(",", "."));
			this.forceUpdate();
		}
	};

	applyFilter = async () => {
		this.setState({ analyzing: true });
		this.stop();
		const { offline, sample, source } = this.getOffline();
		if (offline && sample && source) {
			source.detune.value = this.state.detune * 100;
			const offlineGain = offline.createGain();
			offlineGain.gain.value = gain.gain.value;
			const offlineHighpass = offline.createBiquadFilter();
			offlineHighpass.type = "highpass";
			offlineHighpass.frequency.value = highpass.frequency.value;
			const offlineLowpass = offline.createBiquadFilter();
			offlineLowpass.type = "lowpass";
			offlineLowpass.frequency.value = lowpass.frequency.value;
			source.connect(offlineHighpass);
			offlineHighpass.connect(offlineLowpass);
			offlineLowpass.connect(offlineGain);
			offlineGain.connect(offline.destination);
			source.start();
			const rendered = await offline.startRendering();
			sample.setAudioData(rendered.getChannelData(0));
			for (const zone of this.zones) {
				if (zone.start) {
					zone.start = Math.max(0, Math.floor(zone.start / this.detuneRate));
					zone.zone.setGenerator(
						generatorTypes.startAddrsCoarseOffset,
						Math.floor(zone.start / 32768),
					);
					zone.zone.setGenerator(
						generatorTypes.startAddrsOffset,
						zone.start % 32768,
					);
				}
				if (zone.end) {
					zone.end = Math.min(Math.ceil(zone.end / this.detuneRate), 0);
					zone.zone.setGenerator(
						generatorTypes.endAddrsCoarseOffset,
						Math.ceil(zone.end / 32768),
					);
					zone.zone.setGenerator(
						generatorTypes.endAddrOffset,
						zone.end % 32768,
					);
				}
			}
			highpass.frequency.value = 0;
			lowpass.frequency.value = 22050;
			gain.gain.value = 1;
			this.setState({
				dataPoint: Math.floor(this.state.dataPoint / this.detuneRate),
				detune: 0,
			});
			setTimeout(() => {
				this.draw();
				if (this.main.current) {
					for (const input of [
						...this.main.current.querySelectorAll("input"),
					]) {
						if (input.defaultValue) {
							input.value = input.defaultValue;
						}
					}
				}
			});
		}
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onInstrumentChange = (event) => {
		const currentInstrument = Number(event.currentTarget.value);
		const currentInstrumentZone = 0;
		const zone = this.state.font.instruments
			.at(currentInstrument)
			?.instrumentZones.at(currentInstrumentZone);
		const currentSample = zone
			? this.state.font.samples.indexOf(zone.sample)
			: 0;
		this.setState({
			currentInstrument,
			currentInstrumentZone,
			currentSample: currentSample == -1 ? 0 : currentSample,
		});
		setTimeout(() => {
			this.draw();
			this.play();
		});
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	onInstrumentNameInput = (event) => {
		const instrument = this.state.font.instruments.at(
			this.state.currentInstrument,
		);
		if (instrument) {
			instrument.instrumentName = ascii({
				length: 20,
				value: event.currentTarget.value,
			});
			this.forceUpdate();
		}
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onInstrumentZoneChange = (event) => {
		const currentInstrumentZone = Number(event.currentTarget.value);
		const zone = this.state.font.instruments
			.at(this.state.currentInstrument)
			?.instrumentZones.at(currentInstrumentZone);
		const previousSample = this.state.currentSample;
		let currentSample = zone ? this.state.font.samples.indexOf(zone.sample) : 0;
		if (currentSample == -1) {
			currentSample = 0;
		}
		this.setState({ currentInstrumentZone, currentSample });
		setTimeout(() => {
			if (previousSample != currentSample) {
				this.draw();
			}
			this.play();
		});
	};

	deleteInstrumentZone = () => {
		const instrument = this.state.font.instruments.at(
			this.state.currentInstrument,
		);
		if (instrument) {
			const zone = instrument.instrumentZones.at(
				this.state.currentInstrumentZone,
			);
			if (zone) {
				zone.deleteZone();
				instrument.instrumentZones.splice(this.state.currentInstrumentZone, 1);
			}
			if (
				instrument.instrumentZones.length >= this.state.currentInstrumentZone
			) {
				this.setState({
					currentInstrumentZone: instrument.instrumentZones.length - 1,
				});
			}
			const zones = this.zones;
			if (zones.length && zones[0].start > this.state.dataPoint) {
				this.setState({ dataPoint: zones[0].start });
			} else if (!zones.length) {
				this.setState({ dataPoint: 0 });
			}
		}
		this.stop();
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onInstrumentKeyRangeMaxChange = (event) => {
		const zone = this.state.font.instruments
			.at(this.state.currentInstrument)
			?.instrumentZones.at(this.state.currentInstrumentZone);
		if (zone) {
			const value = Number(event.currentTarget.value);
			zone.keyRange.max = value;
			if (zone.keyRange.min > value) {
				zone.keyRange.min = value;
			}
			if (zone.keyRange.min == zone.keyRange.max) {
				zone.setGenerator(generatorTypes.overridingRootKey, zone.keyRange.min);
			}
			this.forceUpdate();
		}
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onInstrumentKeyRangeMinChange = (event) => {
		const zone = this.state.font.instruments
			.at(this.state.currentInstrument)
			?.instrumentZones.at(this.state.currentInstrumentZone);
		if (zone) {
			const value = Number(event.currentTarget.value);
			const singleNote =
				zone.keyRange.min == -1 || zone.keyRange.min == zone.keyRange.max;
			zone.keyRange.min = value;
			if (value == -1) {
				zone.keyRange.max = 127;
			} else if (singleNote || zone.keyRange.max < value) {
				zone.keyRange.max = value;
			}
			if (zone.keyRange.min == zone.keyRange.max) {
				zone.setGenerator(generatorTypes.overridingRootKey, zone.keyRange.min);
			}
			this.forceUpdate();
		}
	};

	makeKeyRangeSubsequent = () => {
		const zones = this.zones;
		const start = zones.at(0);
		let key = start?.zone.keyRange.min || -1;
		if (start && key != -1 && 127 - key > zones.length - 1) {
			for (const { zone } of zones) {
				zone.setGenerator(generatorTypes.overridingRootKey, key);
				zone.keyRange.max = key;
				zone.keyRange.min = key++;
			}
		}
		this.forceUpdate();
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onPresetChange = (event) => {
		this.setState({ currentPreset: +event.currentTarget.value });
	};

	/** @param {{ currentTarget: HTMLInputElement }} event */
	onPresetNameInput = (event) => {
		const preset = this.state.font.presets.at(this.state.currentPreset);
		if (preset) {
			preset.presetName = ascii({
				length: 20,
				value: event.currentTarget.value,
			});
			this.forceUpdate();
		}
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onPresetZoneChange = (event) => {
		this.setState({ currentPresetZone: +event.currentTarget.value });
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onPresetKeyRangeMaxChange = (event) => {
		const zone = this.state.font.presets
			.at(this.state.currentPreset)
			?.presetZones.at(this.state.currentPresetZone);
		if (zone) {
			const value = Number(event.currentTarget.value);
			zone.keyRange.max = value;
			if (zone.keyRange.min > value) {
				zone.keyRange.min = value;
			}
			this.forceUpdate();
		}
	};

	/** @param {{ currentTarget: HTMLSelectElement }} event */
	onPresetKeyRangeMinChange = (event) => {
		const zone = this.state.font.presets
			.at(this.state.currentPreset)
			?.presetZones.at(this.state.currentPresetZone);
		if (zone) {
			const value = Number(event.currentTarget.value);
			const singleNote =
				zone.keyRange.min == -1 || zone.keyRange.min == zone.keyRange.max;
			zone.keyRange.min = value;
			if (value == -1) {
				zone.keyRange.max = 127;
			} else if (singleNote || zone.keyRange.max < value) {
				zone.keyRange.max = value;
			}
			this.forceUpdate();
		}
	};

	export = async () => {
		const a = document.createElement("a");
		a.download = this.state.font.soundFontInfo["INAM"] + ".sf2";
		a.href =
			URL.createObjectURL(
				new Blob(
					[
						await this.state.font.write({
							compress: false,
							compressionFunction: undefined,
							decompress: false,
							progressFunction: undefined,
							writeDefaultModulators: false,
							writeExtendedLimits: true,
						}),
					],
					{ type: "application/octet-stream" },
				),
			) +
			"#" +
			a.download;
		document.body.appendChild(a);
		a.click();
		document.body.removeChild(a);
	};

	render() {
		return h(
			"main",
			{ ref: this.main },
			h(
				"style",
				null,
				[
					"[readonly] { background-color: transparent; border-color: transparent; box-shadow: none; }",
					".analyzing { background-color: rgba(255, 255, 255, 0.9); }",
					"@media (prefers-color-scheme: dark) { .analyzing { background-color: rgba(0, 0, 0, 0.8); } }",
					"canvas { height: auto; max-width: 100%; }",
					".cursor { border-left: 1px solid currentColor; height: 100%; position: absolute; top: 0; }",
					".form-group { margin-left: 1.5em }",
				].join("\n"),
			),

			edgeToEdge
				? h("div", {
						style: {
							borderTop: "28px solid black",
							left: 0,
							position: "fixed",
							right: 0,
							top: 0,
							zIndex: 1,
						},
					})
				: undefined,
			edgeToEdge ? h("div", { style: { height: "20px" } }) : undefined,

			h("h1", null, "Sapfir"),
			h(
				"p",
				null,
				"Create ",
				h("a", { href: "https://www.synthfont.com/sfspec24.pdf" }, ".sf2"),
				" from recorded sample. Meant for chopping breaks and vocals to make boom bap with the ",
				h(
					"a",
					{ href: "https://f-droid.org/packages/fm.helio/" },
					"Helio sequencer",
				),
				" on Android or the ",
				h("a", { href: "https://ardour.org/" }, "Ardour"),
				" DAW on GNU/Linux when external plugins are unavailable. For serious SoundFont editing, consider ",
				h(
					"a",
					{ href: "https://github.com/spessasus/SpessaFont" },
					"SpessaFont",
				),
				".",
			),

			h(
				"p",
				null,
				"Import existing sf2 to edit: ",
				h("input", {
					accept: ".sf2,audio/x-sfbk",
					onChange: this.importFont,
					type: "file",
				}),
			),

			h("h2", null, "soundFontInfo"),
			Object.keys(this.state.font.soundFontInfo).map((name) =>
				h(
					"div",
					{ className: "form-group" },
					h("label", null, name),
					h("input", {
						defaultValue: this.state.font.soundFontInfo[name],
						name,
						onInput: this.onSoundFontInfoInput,
					}),
				),
			),

			this.state.font.customDefaultModulators
				? h(
						"div",
						{ className: "form-group" },
						h("label", null, "customDefaultModulators"),
						h(BooleanDebug, { value: this.state.font.customDefaultModulators }),
					)
				: undefined,

			this.state.font.isXGBank
				? h(
						"div",
						{ className: "form-group" },
						h("label", null, "isXGBank"),
						h(BooleanDebug, { value: this.state.font.isXGBank }),
					)
				: undefined,

			h("h2", null, "samples"),
			h(
				"p",
				null,
				h(
					"button",
					{ onClick: this.record, type: "button" },
					this.recorder.current?.state == "recording" ? "stop" : "record",
				),
				" or ",
				h("input", {
					accept: "audio/*",
					onChange: this.importSample,
					type: "file",
				}),
			),
			this.state.font.samples.length
				? h(
						"select",
						{ onChange: this.onSampleChange },
						this.state.font.samples.map(({ sampleName }, index) =>
							h(
								"option",
								{
									selected: this.state.currentSample == index,
									value: index,
								},
								sampleName,
							),
						),
					)
				: undefined,
			this.zones.find((zone) => zone.start <= this.state.dataPoint)
				? h("button", { onClick: this.play, type: "button" }, "play")
				: undefined,
			!this.zones.find((zone) => zone.start == this.state.dataPoint) &&
				this.state.font.samples.length
				? h(
						"button",
						{
							onClick: this.chop,
							onTouchStart: this.onCreateZoneTouchStart,
							type: "button",
						},
						"chop",
					)
				: undefined,
			this.zones.at(0)?.start
				? h("button", { onClick: this.trimStart, type: "button" }, "trimStart")
				: undefined,
			this.zones.at(-1)?.end
				? h("button", { onClick: this.trimEnd, type: "button" }, "trimEnd")
				: undefined,
			this.state.playbackInterval
				? h("button", { onClick: this.stop, type: "button" }, "stop")
				: undefined,
			[this.state.font.samples.at(this.state.currentSample)]
				.filter(Boolean)
				.map(alwaysTruthy)
				.map((sample) =>
					h(
						"div",
						null,
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "sampleName"),
							h("input", {
								defaultValue: sample.sampleName,
								maxLength: 20,
								onInput: this.onSampleNameInput,
							}),
						),
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "sampleRate"),
							h(NumberDebug, { value: sample.sampleRate }),
						),
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "samplePitch"),
							h(NumberDebug, { value: sample.samplePitch }),
						),
						sample.samplePitchCorrection
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "samplePitchCorrection"),
									h(NumberDebug, { value: sample.samplePitchCorrection }),
								)
							: undefined,
						sample.sampleType != 1
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "sampleType"),
									h(NumberDebug, { value: sample.sampleType }),
								)
							: undefined,
						sample.sampleLoopStartIndex
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "sampleLoopStartIndex"),
									h(NumberDebug, { value: sample.sampleLoopStartIndex }),
								)
							: undefined,
						sample.sampleLoopEndIndex
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "sampleLoopEndIndex"),
									h(NumberDebug, { value: sample.sampleLoopEndIndex }),
								)
							: undefined,
						sample.isCompressed
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "isCompressed"),
									h(BooleanDebug, { value: sample.isCompressed }),
								)
							: undefined,
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "detune"),
							h("input", {
								defaultValue: this.state.detune,
								min: -24,
								max: 24,
								onInput: this.onDetuneInput,
								step: 1,
								type: "number",
							}),
							this.state.detune &&
								this.state.keyData &&
								["major", "minor"].indexOf(this.state.keyData.scale) != -1
								? notes.at(
										60 +
											notes.findIndex(
												({ withoutOctave }) =>
													withoutOctave == this.state.keyData?.key,
											) +
											this.state.detune,
									)?.[
										this.state.keyData.scale == "major" ? "major" : "minor"
									]?.[0].withoutOctave +
										" " +
										this.state.keyData.scale
								: undefined,
						),
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "highpass"),
							h("input", {
								defaultValue: highpass.frequency.value,
								max: 22050,
								min: 0,
								onInput: this.onHighpassInput,
								step: 10,
								type: "number",
							}),
						),
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "lowpass"),
							h("input", {
								defaultValue: lowpass.frequency.value,
								max: 22050,
								min: 0,
								onInput: this.onLowpassInput,
								step: 100,
								type: "number",
							}),
						),
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "gain"),
							h("input", {
								defaultValue: gain.gain.value,
								max: 2,
								min: 0,
								onInput: this.onGainInput,
								step: 0.1,
								type: "number",
							}),
							Math.abs(1 - this.state.recommendedGain) > 0.1
								? h(
										"button",
										{ onClick: this.normalize, type: "button" },
										"normalize",
									)
								: undefined,
						),
						this.state.detune ||
							highpass.frequency.value ||
							lowpass.frequency.value != 22050 ||
							gain.gain.value != 1
							? h(
									"div",
									{ className: "form-group" },
									h(
										"button",
										{ onClick: this.applyFilter, type: "button" },
										"applyFilter",
									),
								)
							: undefined,
						this.state.bpm
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "bpm"),
									h(NumberDebug, { value: this.state.bpm }),
								)
							: undefined,
						this.state.keyData
							? h(
									"div",
									{ className: "form-group" },
									h(
										"label",
										null,
										"key",
										h(TextDebug, {
											value:
												this.state.keyData.key + " " + this.state.keyData.scale,
										}),
									),
								)
							: undefined,
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "dataPoint"),
							h(
								"div",
								{ style: { display: "table", position: "relative" } },
								h("canvas", { height: 480, ref: this.canvas, width: 640 }),
								h("div", {
									className: "cursor",
									style: {
										left:
											(this.state.dataPoint /
												(this.state.font.samples
													.at(this.state.currentSample)
													?.getAudioData().length || 1)) *
												100 +
											"%",
									},
								}),
								this.zones.map(({ zone }) =>
									h("div", {
										className: "cursor zone-cursor",
										style: {
											backgroundColor:
												"color-mix(in srgb, currentColor " +
												(zone ==
												this.state.font.instruments
													.at(this.state.currentInstrument)
													?.instrumentZones.at(this.state.currentInstrumentZone)
													? "25%"
													: "12.5%") +
												", transparent)",
											borderLeft: "1px solid blue",
											left:
												((zone.getGeneratorValue(
													generatorTypes.startAddrsCoarseOffset,
													0,
												) *
													32768 +
													zone.getGeneratorValue(
														generatorTypes.startAddrsOffset,
														0,
													)) /
													(this.state.font.samples
														.at(this.state.currentSample)
														?.getAudioData().length || 1)) *
													100 +
												"%",
											right:
												-(
													(zone.getGeneratorValue(
														generatorTypes.endAddrsCoarseOffset,
														0,
													) *
														32768 +
														zone.getGeneratorValue(
															generatorTypes.endAddrOffset,
															0,
														)) /
													(this.state.font.samples
														.at(this.state.currentSample)
														?.getAudioData().length || 1)
												) *
													100 +
												"%",
										},
									}),
								),
							),
						),
					),
				),

			this.state.font.instruments.length
				? h("h2", null, "instruments")
				: undefined,
			this.state.font.instruments.length
				? h(
						"select",
						{ onChange: this.onInstrumentChange },
						this.state.font.instruments.map(({ instrumentName }, index) =>
							h(
								"option",
								{
									selected: this.state.currentInstrument == index,
									value: index,
								},
								instrumentName,
							),
						),
					)
				: undefined,
			[this.state.font.instruments.at(this.state.currentInstrument)]
				.filter(Boolean)
				.map(alwaysTruthy)
				.map((instrument) =>
					h(
						"div",
						null,
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "instrumentName"),
							h("input", {
								defaultValue: instrument.instrumentName,
								maxLength: 20,
								onInput: this.onInstrumentNameInput,
							}),
						),
						instrument.instrumentZones.length
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "instrumentZones"),
									h(
										"select",
										{ onChange: this.onInstrumentZoneChange },
										instrument.instrumentZones.map((zone, index) =>
											h(
												"option",
												{
													selected: this.state.currentInstrumentZone == index,
													value: index,
												},
												zone.keyRange.min == -1
													? ""
													: [...new Set([zone.keyRange.min, zone.keyRange.max])]
															.map(
																(value) =>
																	notes
																		.find((note) => note.value == value)
																		?.label.split(" ")[0] || value,
															)
															.join(" - ") + ": ",
												zone.sample?.sampleName,
											),
										),
									),
									[
										instrument.instrumentZones.at(
											this.state.currentInstrumentZone,
										),
									]
										.filter(Boolean)
										.map(alwaysTruthy)
										.map((zone) =>
											h(
												"div",
												null,
												h(
													"div",
													{ className: "form-group" },
													h(
														"button",
														{
															onClick: this.deleteInstrumentZone,
															type: "button",
														},
														"deleteInstrumentZone",
													),
												),
												h(Zone, {
													isPercussion:
														this.state.font.presets.find((preset) =>
															preset.presetZones.some(
																(presetZone) =>
																	presetZone.instrument == instrument,
															),
														)?.bank == 128,
													makeKeyRangeSubsequent: (() => {
														const start = this.zones.at(0)?.zone;
														return (
															start &&
															start ==
																instrument.instrumentZones.at(
																	this.state.currentInstrumentZone,
																) &&
															start.keyRange.min != -1 &&
															127 - start.keyRange.min >
																instrument.instrumentZones.length - 1 &&
															!this.zones.every(
																({ zone }, index, zones) =>
																	zone.keyRange.min == zone.keyRange.max &&
																	zone.keyRange.min &&
																	(!index ||
																		zones[index - 1].zone.keyRange.min ==
																			zone.keyRange.min - 1),
															)
														);
													})()
														? this.makeKeyRangeSubsequent
														: undefined,
													onKeyRangeMaxChange:
														this.onInstrumentKeyRangeMaxChange,
													onKeyRangeMinChange:
														this.onInstrumentKeyRangeMinChange,
													zone,
												}),
											),
										),
								)
							: undefined,
						instrument.globalZone.generators.length ||
							instrument.globalZone.modulators.length
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "globalZone"),
									h(Zone, { zone: instrument.globalZone }),
								)
							: undefined,
					),
				),

			this.state.font.presets.length ? h("h2", null, "presets") : undefined,
			this.state.font.presets.length
				? h(
						"select",
						{ onChange: this.onPresetChange },
						this.state.font.presets.map(({ presetName }, index) =>
							h(
								"option",
								{
									selected: this.state.currentPreset == index,
									value: index,
								},
								presetName,
							),
						),
					)
				: undefined,
			[this.state.font.presets.at(this.state.currentPreset)]
				.filter(Boolean)
				.map(alwaysTruthy)
				.map((preset) =>
					h(
						"div",
						null,
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "presetName"),
							h("input", {
								defaultValue: preset.presetName,
								maxLength: 20,
								onInput: this.onPresetNameInput,
							}),
						),
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "program"),
							h(NumberDebug, { value: preset.program }),
						),
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "bank"),
							h(NumberDebug, { value: preset.bank }),
						),
						h(
							"div",
							{ className: "form-group" },
							h("label", null, "presetZones"),
							h(
								"select",
								{ onChange: this.onPresetZoneChange },
								preset.presetZones.map((zone, index) =>
									h(
										"option",
										{
											selected: this.state.currentPresetZone == index,
											value: index,
										},
										zone.instrument?.instrumentName,
									),
								),
							),
							[preset.presetZones.at(this.state.currentPresetZone)]
								.filter(Boolean)
								.map(alwaysTruthy)
								.map((zone) =>
									h(Zone, {
										onKeyRangeMaxChange: this.onPresetKeyRangeMaxChange,
										onKeyRangeMinChange: this.onPresetKeyRangeMinChange,
										zone,
									}),
								),
						),
						preset.globalZone.generators.length ||
							preset.globalZone.modulators.length
							? h(
									"div",
									{ className: "form-group" },
									h("label", null, "globalZone"),
									h(Zone, { zone: preset.globalZone }),
								)
							: undefined,
					),
				),
			this.state.font.presets.length
				? h(
						"p",
						null,
						h(
							"button",
							{ onClick: this.export, type: "button" },
							"export .sf2",
						),
					)
				: undefined,

			h(
				"footer",
				null,
				h("hr"),
				h(
					"p",
					null,
					"License: ",
					h(
						"a",
						{
							href: "https://codeberg.org/nykula/sapfir/raw/branch/main/COPYING",
						},
						"AGPL-3.0-or-later",
					),
				),
				h(
					"p",
					null,
					"Source: ",
					h("a", { href: "https://codeberg.org/nykula/sapfir" }, "sapfir"),
					", ",
					h(
						"a",
						{ href: "https://github.com/spessasus/spessasynth_core" },
						"spessasynth_core",
					),
					", ",
					h("a", { href: "https://mtg.github.io/essentia.js/" }, "essentia.js"),
					", ",
					h("a", { href: "https://preactjs.com/" }, "Preact"),
				),
			),

			this.state.analyzing
				? h(
						"div",
						{
							className: "analyzing",
							style: {
								alignItems: "center",
								bottom: "-4rem",
								display: "flex",
								justifyContent: "center",
								left: 0,
								position: "fixed",
								right: 0,
								top: "-4rem",
							},
						},
						"Analyzing...",
					)
				: undefined,
		);
	}
}

render(h(App), document.body);
