package org.codeberg.genericpers0n.soundrecorderplus;

import android.content.Context;
import android.database.sqlite.SQLiteException;

import java.io.File;
import java.io.IOException;

public class RecordingsManager {
    private String mRecordingsPath;
    private final DBHelper mDatabase;

    public RecordingsManager(Context context)
    {
        mRecordingsPath = context.getExternalFilesDir(null).getAbsolutePath();
        mDatabase = new DBHelper(context);
    }

    /** Sets the path to work from.
     * <p>
     * TODO: rescan all recordings and add new ones in the database. Maybe
     * delete old recordings in the database too?
     *
     * @param path The path recordings are stored in.
     */
    public void setRecordingsPath(String path)
    {
        mRecordingsPath = path;
    }

    public String getRecordingsPath()
    {
        return mRecordingsPath;
    }

    public int getNumberOfRecordings()
    {
        return mDatabase.getCount();
    }

    /** Adds a recording to the recordings database.
     * <p>
     * If adding the recording fails, this method throws a SQLiteException.
     *
     * @param name The name of the recording. It is assumed that the recording
     * will be in the directory as returned by getRecordingsPath().
     * @param time The duration of the recording.
     */
    public void addRecording(String name, long time) throws SQLiteException
    {
        long id = mDatabase.addRecording(name, mRecordingsPath, time);
        if (id == -1)
        {
            throw new SQLiteException("Failed to add recording " + mRecordingsPath + " to database");
        }
    }

    /** Returns the database item of the recording at a given index.
     *
     * @param index The index of the recording item.
     * @return The information of a recording at a given index.
     */
    public RecordingItem getRecording(int index)
    {
        return mDatabase.getItemAt(index);
    }

    /** Removes a recording from the database and deletes it from storage.
     * <p>
     * If deleting the recording fails, this method throw an IOException. It is
     * possible that the recording might have been removed from the database.
     * <p>
     * TODO: Better handle database errors. These are not handled very well in
     * DBHelper.
     *
     * @param index The index of the recording
     */
    public void deleteRecording(int index) throws IOException //, SQLiteException
    {
        RecordingItem recording = mDatabase.getItemAt(index);
        File recordingFile = new File(mRecordingsPath, recording.getName());

        // Remove the file
        mDatabase.removeItemWithId(recording.getId());
        if (!recordingFile.delete())
        {
            throw new IOException("Failed to delete recording " + recording.getName());
        }
    }

    /** Renames a recording.
     * <p>
     * If the rename fails, this method throws an IOException.
     *
     * @param index The index of the recording.
     * @param newName The new name of the recording.
     */
    public void renameRecording(int index, String newName) throws IOException
    {
        RecordingItem recording = mDatabase.getItemAt(index);
        File recordingFile = new File(mRecordingsPath, recording.getName());
        File newRecordingFile = new File(mRecordingsPath, newName);

        mDatabase.renameItem(recording, newName, mRecordingsPath);
        if (!recordingFile.renameTo(newRecordingFile))
        {
            throw new IOException("Failed to rename recording " + recording.getName() + " to " + newName);
        }
    }
}
