package org.chorem.android.saymytexts;

/*
 * #%L
 * Say My Texts
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import android.bluetooth.BluetoothClass;
import android.bluetooth.BluetoothDevice;
import android.content.BroadcastReceiver;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.provider.BaseColumns;
import android.provider.ContactsContract;
import android.telephony.SmsMessage;
import android.util.Log;

/**
 * Receives the SMSs and if the headset is plugged, start the service to say it out loud.
 *
 * @author Kevin Morin (Code Lutin)
 * @since 1.0
 */
public class NewTextBroadcastReceiver extends BroadcastReceiver {

    private static final String TAG = "NewTextBroadcastReceiver";

    @Override
    public void onReceive(Context context, Intent intent) {
        Intent serviceIntent = new Intent(context, SayMyTextService.class);
        String action = intent.getAction();
        if ("android.provider.Telephony.SMS_RECEIVED".equals(action)) {

            // get the SMS message passed in
            Bundle bundle = intent.getExtras();
            if (bundle != null) {
                SmsMessage[] msgs = null;
                String messageReceived = "";

                // retrieve the SMS message received
                Object[] pdus = (Object[]) bundle.get("pdus");
                msgs = new SmsMessage[pdus.length];
                for (int i = 0; i < msgs.length; i++) {
                    msgs[i] = SmsMessage.createFromPdu((byte[]) pdus[i]);
                    messageReceived += msgs[i].getDisplayMessageBody() + " ";
                }

                // check if the message is '=' to avoid restart the phone
                if ("=".equals(messageReceived.trim())) {
                    SharedPreferences sharedPref = PreferenceManager.getDefaultSharedPreferences(context);
                    String key = context.getString(R.string.preference_block_equals_sms_key);
                    boolean blockEqualsSms = sharedPref.getBoolean(key, false);

                    if (blockEqualsSms) {
                        abortBroadcast();
                        return;
                    }
                }

                // Get the Sender Phone Number
                String senderPhoneNumber = msgs[0].getDisplayOriginatingAddress();
                String senderName = getContactDisplayNameByNumber(context, senderPhoneNumber);
                SMS sms = new SMS(senderPhoneNumber, senderName, messageReceived);

                Log.d(TAG, messageReceived);
                // start the service to say it out loud
                serviceIntent.putExtra(SayMyTextService.INTENT_EXTRA_SMS, sms);
                serviceIntent.setAction(SayMyTextService.ACTION_READ_SMS);
                context.startService(serviceIntent);
            }

        } else {
            BluetoothDevice device = intent.getParcelableExtra(BluetoothDevice.EXTRA_DEVICE);
            serviceIntent.setAction(SayMyTextService.ACTION_MANAGE_BT_DEVICE);

            if (BluetoothDevice.ACTION_ACL_CONNECTED.equals(action)) {
                BluetoothClass bluetoothClass = device.getBluetoothClass();
                if (bluetoothClass != null) {
                    int majorDeviceClass = bluetoothClass.getMajorDeviceClass();
                    if (majorDeviceClass == BluetoothClass.Device.Major.AUDIO_VIDEO) {
                        serviceIntent.putExtra(SayMyTextService.INTENT_EXTRA_BT_DEVICE, device);
                        serviceIntent.putExtra(SayMyTextService.INTENT_EXTRA_ADD_BT_DEVICE, true);
                        context.startService(serviceIntent);
                    }

                } else {
                    //error recognizing the bt class
                }

            } else {
                serviceIntent.putExtra(SayMyTextService.INTENT_EXTRA_BT_DEVICE, device);
                serviceIntent.putExtra(SayMyTextService.INTENT_EXTRA_ADD_BT_DEVICE, false);
                context.startService(serviceIntent);
            }
        }
    }

    /**
     * Finds the contact name in the contact book
     * @param number the number of the contact
     * @return the name if the contact is known, the number otherwise
     */
    protected String getContactDisplayNameByNumber(Context context, String number) {
        Uri uri = Uri.withAppendedPath(ContactsContract.PhoneLookup.CONTENT_FILTER_URI, Uri.encode(number));
        String name = number;

        ContentResolver contentResolver = context.getContentResolver();
        Cursor contactLookup = contentResolver.query(uri,
                                                     new String[] { BaseColumns._ID, ContactsContract.PhoneLookup.DISPLAY_NAME },
                                                     null, null, null);

        try {
            if (contactLookup != null && contactLookup.getCount() > 0) {
                contactLookup.moveToNext();
                name = contactLookup.getString(contactLookup.getColumnIndex(ContactsContract.Data.DISPLAY_NAME));
            }
        } finally {
            if (contactLookup != null) {
                contactLookup.close();
            }
        }

        return name;
    }
}
