package api

import area.AreaJson
import area.toAreasJson
import element.ElementJson
import element.toElementsJson
import element_comment.ElementCommentJson
import element_comment.toElementCommentsJson
import event.EventJson
import event.toEventsJson
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.HttpUrl
import okhttp3.HttpUrl.Companion.toHttpUrl
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.brotli.BrotliInterceptor
import okhttp3.coroutines.executeAsync
import reports.ReportJson
import reports.toReportsJson
import user.UserJson
import user.toUsersJson
import java.time.ZonedDateTime
import java.time.format.DateTimeFormatter

private val BASE_URL = "https://api.btcmap.org".toHttpUrl()

interface Api {
    suspend fun getAreas(updatedSince: ZonedDateTime?, limit: Long): List<AreaJson>

    suspend fun getElementComments(
        updatedSince: ZonedDateTime?,
        limit: Long
    ): List<ElementCommentJson>

    suspend fun getElements(updatedSince: ZonedDateTime?, limit: Long): List<ElementJson>

    suspend fun getEvents(updatedSince: ZonedDateTime?, limit: Long): List<EventJson>

    suspend fun getReports(updatedSince: ZonedDateTime?, limit: Long): List<ReportJson>

    suspend fun getUsers(updatedSince: ZonedDateTime?, limit: Long): List<UserJson>
}

class ApiImpl(
    private val baseUrl: HttpUrl = BASE_URL,
    private val httpClient: OkHttpClient = apiHttpClient(),
) : Api {
    override suspend fun getAreas(updatedSince: ZonedDateTime?, limit: Long): List<AreaJson> {
        val url = baseUrl.newBuilder().apply {
            addPathSegment("v3")
            addPathSegment("areas")
            addQueryParameter("updated_since", updatedSince.apiFormat())
            addQueryParameter("limit", "$limit")
        }.build()

        val res = httpClient.newCall(Request.Builder().url(url).build()).executeAsync()

        if (!res.isSuccessful) {
            throw Exception("Unexpected HTTP response code: ${res.code}")
        }

        return withContext(Dispatchers.IO) {
            res.body.byteStream().use { it.toAreasJson() }
        }
    }

    override suspend fun getElementComments(
        updatedSince: ZonedDateTime?,
        limit: Long
    ): List<ElementCommentJson> {
        val url = baseUrl.newBuilder().apply {
            addPathSegment("v3")
            addPathSegment("element-comments")
            addQueryParameter("updated_since", updatedSince.apiFormat())
            addQueryParameter("limit", "$limit")
        }.build()

        val res = httpClient.newCall(Request.Builder().url(url).build()).executeAsync()

        if (!res.isSuccessful) {
            throw Exception("Unexpected HTTP response code: ${res.code}")
        }

        return withContext(Dispatchers.IO) {
            res.body.byteStream().use { it.toElementCommentsJson() }
        }
    }

    override suspend fun getElements(updatedSince: ZonedDateTime?, limit: Long): List<ElementJson> {
        val url = baseUrl.newBuilder().apply {
            addPathSegment("v3")
            addPathSegment("elements")
            addQueryParameter("updated_since", updatedSince.apiFormat())
            addQueryParameter("limit", "$limit")
        }.build()

        val res = httpClient.newCall(Request.Builder().url(url).build()).executeAsync()

        if (!res.isSuccessful) {
            throw Exception("Unexpected HTTP response code: ${res.code}")
        }

        return withContext(Dispatchers.IO) {
            res.body.byteStream().use { it.toElementsJson() }
        }
    }

    override suspend fun getEvents(updatedSince: ZonedDateTime?, limit: Long): List<EventJson> {
        val url = baseUrl.newBuilder().apply {
            addPathSegment("v3")
            addPathSegment("events")
            addQueryParameter("updated_since", updatedSince.apiFormat())
            addQueryParameter("limit", "$limit")
        }.build()

        val res = httpClient.newCall(Request.Builder().url(url).build()).executeAsync()

        if (!res.isSuccessful) {
            throw Exception("Unexpected HTTP response code: ${res.code}")
        }

        return withContext(Dispatchers.IO) {
            res.body.byteStream().use { it.toEventsJson() }
        }
    }

    override suspend fun getReports(updatedSince: ZonedDateTime?, limit: Long): List<ReportJson> {
        val url = baseUrl.newBuilder().apply {
            addPathSegment("v3")
            addPathSegment("reports")
            addQueryParameter("updated_since", updatedSince.apiFormat())
            addQueryParameter("limit", "$limit")
        }.build()

        val res = httpClient.newCall(Request.Builder().url(url).build()).executeAsync()

        if (!res.isSuccessful) {
            throw Exception("Unexpected HTTP response code: ${res.code}")
        }

        return withContext(Dispatchers.IO) {
            res.body.byteStream().use { it.toReportsJson() }
        }
    }

    override suspend fun getUsers(updatedSince: ZonedDateTime?, limit: Long): List<UserJson> {
        val url = baseUrl.newBuilder().apply {
            addPathSegment("v3")
            addPathSegment("users")
            addQueryParameter("updated_since", updatedSince.apiFormat())
            addQueryParameter("limit", "$limit")
        }.build()

        val res = httpClient.newCall(Request.Builder().url(url).build()).executeAsync()

        if (!res.isSuccessful) {
            throw Exception("Unexpected HTTP response code: ${res.code}")
        }

        return withContext(Dispatchers.IO) {
            res.body.byteStream().use { it.toUsersJson() }
        }
    }
}

private fun apiHttpClient(): OkHttpClient {
    return OkHttpClient.Builder()
        .addInterceptor(BrotliInterceptor)
        .addInterceptor {
            var res = it.proceed(it.request())

            var retryAttempts = 0

            while (res.code == 429 && retryAttempts < 10) {
                res.close()
                Thread.sleep(retryAttempts * 1000 + (Math.random() * 1000.0).toLong())
                res = it.proceed(it.request())
                retryAttempts++
            }

            res
        }.build()
}

private fun ZonedDateTime?.apiFormat(): String {
    return this?.format(DateTimeFormatter.ISO_OFFSET_DATE_TIME) ?: "2020-01-01T00:00:00Z"
}