package com.fsck.k9.ui.settings.general

import androidx.preference.PreferenceDataStore
import app.k9mail.feature.telemetry.api.TelemetryManager
import com.fsck.k9.K9
import com.fsck.k9.K9.PostMarkAsUnreadNavigation
import com.fsck.k9.job.K9JobManager
import com.fsck.k9.ui.base.AppLanguageManager
import net.thunderbird.core.common.action.SwipeAction
import net.thunderbird.core.common.action.SwipeActions
import net.thunderbird.core.preference.AppTheme
import net.thunderbird.core.preference.BackgroundOps
import net.thunderbird.core.preference.BodyContentType
import net.thunderbird.core.preference.GeneralSettingsManager
import net.thunderbird.core.preference.NotificationQuickDelete
import net.thunderbird.core.preference.SplitViewMode
import net.thunderbird.core.preference.SubTheme
import net.thunderbird.core.preference.display.visualSettings.message.list.UiDensity
import net.thunderbird.core.preference.update

@Suppress("LargeClass")
class GeneralSettingsDataStore(
    private val jobManager: K9JobManager,
    private val appLanguageManager: AppLanguageManager,
    private val generalSettingsManager: GeneralSettingsManager,
    private val telemetryManager: TelemetryManager,
) : PreferenceDataStore() {

    private var skipSaveSettings = false

    @Suppress("CyclomaticComplexMethod")
    override fun getBoolean(key: String, defValue: Boolean): Boolean {
        val config = generalSettingsManager.getConfig()
        val coreSettings = config.display.coreSettings
        val debuggingSettings = config.debugging
        val inboxSettings = config.display.inboxSettings
        val interactionSettings = config.interaction
        val notificationSettings = config.notification
        val privacySettings = config.privacy
        val visualSettings = config.display.visualSettings
        val messageListSettings = visualSettings.messageListSettings
        return when (key) {
            "fixed_message_view_theme" -> coreSettings.fixedMessageViewTheme
            "animations" -> visualSettings.isShowAnimations
            "show_unified_inbox" -> inboxSettings.isShowUnifiedInbox
            "show_starred_count" -> inboxSettings.isShowStarredCount
            "messagelist_stars" -> inboxSettings.isShowMessageListStars
            "messagelist_show_correspondent_names" -> messageListSettings.isShowCorrespondentNames
            "messagelist_sender_above_subject" -> inboxSettings.isMessageListSenderAboveSubject
            "messagelist_show_contact_name" -> messageListSettings.isShowContactName
            "messagelist_change_contact_name_color" -> messageListSettings.isChangeContactNameColor
            "messagelist_show_contact_picture" -> messageListSettings.isShowContactPicture
            "messagelist_colorize_missing_contact_pictures" -> messageListSettings.isColorizeMissingContactPictures
            "messagelist_background_as_unread_indicator" -> messageListSettings.isUseBackgroundAsUnreadIndicator
            "show_compose_button" -> inboxSettings.isShowComposeButtonOnMessageList
            "threaded_view" -> inboxSettings.isThreadedViewEnabled
            "messageview_fixedwidth_font" -> visualSettings.isUseMessageViewFixedWidthFont
            "messageview_autofit_width" -> visualSettings.isAutoFitWidth
            "drawerExpandAllFolder" -> visualSettings.drawerExpandAllFolder
            "quiet_time_enabled" -> notificationSettings.isQuietTimeEnabled
            "disable_notifications_during_quiet_time" -> !notificationSettings.isNotificationDuringQuietTimeEnabled
            "privacy_hide_useragent" -> privacySettings.isHideUserAgent
            "privacy_hide_timezone" -> privacySettings.isHideTimeZone
            "debug_logging" -> debuggingSettings.isDebugLoggingEnabled
            "sync_debug_logging" -> debuggingSettings.isSyncLoggingEnabled
            "sensitive_logging" -> debuggingSettings.isSensitiveLoggingEnabled
            "volume_navigation" -> interactionSettings.useVolumeKeysForNavigation
            "enable_telemetry" -> K9.isTelemetryEnabled
            else -> defValue
        }
    }

    override fun putBoolean(key: String, value: Boolean) {
        when (key) {
            "fixed_message_view_theme" -> setFixedMessageViewTheme(value)
            "animations" -> setIsShowAnimations(isShowAnimations = value)
            "drawerExpandAllFolder" -> setDrawerExpandAllFolder(drawerExpandAllFolder = value)
            "show_unified_inbox" -> setIsShowUnifiedInbox(value)
            "show_starred_count" -> setIsShowStarredCount(isShowStarredCount = value)
            "messagelist_stars" -> setIsShowMessageListStars(isShowMessageListStars = value)
            "messagelist_show_correspondent_names" -> setIsShowCorrespondentNames(isShowCorrespondentNames = value)
            "messagelist_sender_above_subject" -> setIsMessageListSenderAboveSubject(
                isMessageListSenderAboveSubject = value,
            )

            "messagelist_show_contact_name" -> setIsShowContactName(isShowContactName = value)
            "messagelist_change_contact_name_color" -> setIsChangeContactNameColor(isChangeContactNameColor = value)
            "messagelist_show_contact_picture" -> setIsShowContactPicture(isShowContactPicture = value)
            "messagelist_colorize_missing_contact_pictures" -> setIsColorizeMissingContactPictures(
                isColorizeMissingContactPictures = value,
            )

            "messagelist_background_as_unread_indicator" -> setIsUseBackgroundAsUnreadIndicator(
                isUseBackgroundAsUnreadIndicator = value,
            )

            "show_compose_button" -> setIsShowComposeButtonOnMessageList(isShowComposeButtonOnMessageList = value)
            "threaded_view" -> setIsThreadedViewEnabled(isThreadedViewEnabled = value)
            "messageview_fixedwidth_font" -> setIsUseMessageViewFixedWidthFont(isUseMessageViewFixedWidthFont = value)
            "messageview_autofit_width" -> setIsAutoFitWidth(isAutoFitWidth = value)
            "quiet_time_enabled" -> setIsQuietTimeEnabled(isQuietTimeEnabled = value)
            "disable_notifications_during_quiet_time" -> setIsNotificationDuringQuietTimeEnabled(!value)
            "privacy_hide_useragent" -> setIsHideUserAgent(isHideUserAgent = value)
            "privacy_hide_timezone" -> setIsHideTimeZone(isHideTimeZone = value)
            "debug_logging" -> setIsDebugLoggingEnabled(isDebugLoggingEnabled = value)
            "sync_debug_logging" -> setIsSyncLoggingEnabled(isSyncLoggingEnabled = value)
            "sensitive_logging" -> setIsSensitiveLoggingEnabled(isSensitiveLoggingEnabled = value)
            "volume_navigation" -> setUseVolumeKeysForNavigation(value)
            "enable_telemetry" -> setTelemetryEnabled(value)
            else -> return
        }

        saveSettings()
    }

    override fun getInt(key: String?, defValue: Int): Int {
        return when (key) {
            "messagelist_contact_name_color" ->
                generalSettingsManager
                    .getConfig().display.visualSettings.messageListSettings.contactNameColor

            "message_view_content_font_slider" -> K9.fontSizes.messageViewContentAsPercent
            else -> defValue
        }
    }

    override fun putInt(key: String?, value: Int) {
        when (key) {
            "messagelist_contact_name_color" -> setContactNameColor(value)
            "message_view_content_font_slider" -> K9.fontSizes.messageViewContentAsPercent = value
            else -> return
        }

        saveSettings()
    }

    override fun getString(key: String, defValue: String?): String? {
        val config = generalSettingsManager.getConfig()
        val coreSettings = config.display.coreSettings
        val interactionSettings = config.interaction
        val notificationSettings = config.notification
        val networkSettings = config.network
        val visualSettings = config.display.visualSettings
        val messageListSettings = visualSettings.messageListSettings
        return when (key) {
            "language" -> appLanguageManager.getAppLanguage()
            "theme" -> appThemeToString(coreSettings.appTheme)
            "message_compose_theme" -> subThemeToString(coreSettings.messageComposeTheme)
            "messageViewTheme" -> subThemeToString(coreSettings.messageViewTheme)
            "messagelist_preview_lines" -> messageListSettings.previewLines.toString()
            "splitview_mode" -> coreSettings.splitViewMode.name
            "notification_quick_delete" -> notificationSettings.notificationQuickDeleteBehaviour.name
            "lock_screen_notification_visibility" -> K9.lockScreenNotificationVisibility.name
            "background_ops" -> networkSettings.backgroundOps.name
            "quiet_time_starts" -> notificationSettings.quietTimeStarts
            "quiet_time_ends" -> notificationSettings.quietTimeEnds
            "messageview_body_content_type" -> visualSettings.bodyContentType.name
            "message_list_subject_font" -> K9.fontSizes.messageListSubject.toString()
            "message_list_sender_font" -> K9.fontSizes.messageListSender.toString()
            "message_list_date_font" -> K9.fontSizes.messageListDate.toString()
            "message_list_preview_font" -> K9.fontSizes.messageListPreview.toString()
            "message_view_account_name_font" -> K9.fontSizes.messageViewAccountName.toString()
            "message_view_sender_font" -> K9.fontSizes.messageViewSender.toString()
            "message_view_recipients_font" -> K9.fontSizes.messageViewRecipients.toString()
            "message_view_subject_font" -> K9.fontSizes.messageViewSubject.toString()
            "message_view_date_font" -> K9.fontSizes.messageViewDate.toString()
            "message_compose_input_font" -> K9.fontSizes.messageComposeInput.toString()
            "swipe_action_right" -> swipeActionToString(interactionSettings.swipeActions.rightAction)
            "swipe_action_left" -> swipeActionToString(interactionSettings.swipeActions.leftAction)
            "message_list_density" -> messageListSettings.uiDensity.toString()
            "post_remove_navigation" -> interactionSettings.messageViewPostRemoveNavigation
            "post_mark_as_unread_navigation" -> K9.messageViewPostMarkAsUnreadNavigation.name
            else -> defValue
        }
    }

    override fun putString(key: String, value: String?) {
        if (value == null) return

        when (key) {
            "language" -> {
                skipSaveSettings = true
                appLanguageManager.setAppLanguage(value)
            }

            "theme" -> setTheme(value)
            "message_compose_theme" -> setMessageComposeTheme(value)
            "messageViewTheme" -> setMessageViewTheme(value)
            "messagelist_preview_lines" -> setMessageListPreviewLines(value.toInt())
            "splitview_mode" -> setSplitViewModel(SplitViewMode.valueOf(value.uppercase()))
            "notification_quick_delete" -> {
                setNotificationQuickDeleteBehaviour(
                    behaviour = NotificationQuickDelete.valueOf(value),
                )
            }

            "lock_screen_notification_visibility" -> {
                K9.lockScreenNotificationVisibility = K9.LockScreenNotificationVisibility.valueOf(value)
            }

            "background_ops" -> setBackgroundOps(value)
            "messageview_body_content_type" -> setBodyContentType(value)
            "quiet_time_starts" -> setQuietTimeStarts(quietTimeStarts = value)
            "quiet_time_ends" -> setQuietTimeEnds(quietTimeEnds = value)
            "message_list_subject_font" -> K9.fontSizes.messageListSubject = value.toInt()
            "message_list_sender_font" -> K9.fontSizes.messageListSender = value.toInt()
            "message_list_date_font" -> K9.fontSizes.messageListDate = value.toInt()
            "message_list_preview_font" -> K9.fontSizes.messageListPreview = value.toInt()
            "message_view_account_name_font" -> K9.fontSizes.messageViewAccountName = value.toInt()
            "message_view_sender_font" -> K9.fontSizes.messageViewSender = value.toInt()
            "message_view_recipients_font" -> K9.fontSizes.messageViewRecipients = value.toInt()
            "message_view_subject_font" -> K9.fontSizes.messageViewSubject = value.toInt()
            "message_view_date_font" -> K9.fontSizes.messageViewDate = value.toInt()
            "message_compose_input_font" -> K9.fontSizes.messageComposeInput = value.toInt()
            "swipe_action_right" -> updateSwipeAction(value) { swipeAction -> copy(rightAction = swipeAction) }
            "swipe_action_left" -> updateSwipeAction(value) { swipeAction -> copy(leftAction = swipeAction) }
            "message_list_density" -> updateMessageListDensity(value)
            "post_remove_navigation" -> setMessageViewPostRemoveNavigation(value)
            "post_mark_as_unread_navigation" -> {
                K9.messageViewPostMarkAsUnreadNavigation = PostMarkAsUnreadNavigation.valueOf(value)
            }

            else -> return
        }

        saveSettings()
    }

    override fun getStringSet(key: String, defValues: Set<String>?): Set<String>? {
        return when (key) {
            "confirm_actions" -> {
                mutableSetOf<String>().apply {
                    val interactionSettings = generalSettingsManager.getConfig().interaction
                    if (interactionSettings.isConfirmDelete) add("delete")
                    if (interactionSettings.isConfirmDeleteStarred) add("delete_starred")
                    if (interactionSettings.isConfirmDeleteFromNotification) add("delete_notif")
                    if (interactionSettings.isConfirmSpam) add("spam")
                    if (interactionSettings.isConfirmDiscardMessage) add("discard")
                    if (interactionSettings.isConfirmMarkAllRead) add("mark_all_read")
                }
            }

            "messageview_visible_refile_actions" -> {
                mutableSetOf<String>().apply {
                    if (K9.isMessageViewDeleteActionVisible) add("delete")
                    if (K9.isMessageViewArchiveActionVisible) add("archive")
                    if (K9.isMessageViewMoveActionVisible) add("move")
                    if (K9.isMessageViewCopyActionVisible) add("copy")
                    if (K9.isMessageViewSpamActionVisible) add("spam")
                }
            }

            else -> defValues
        }
    }

    override fun putStringSet(key: String, values: MutableSet<String>?) {
        val checkedValues = values ?: emptySet<String>()
        when (key) {
            "confirm_actions" -> {
                skipSaveSettings = true
                generalSettingsManager.update { settings ->
                    settings.copy(
                        interaction = settings.interaction.copy(
                            isConfirmDelete = "delete" in checkedValues,
                            isConfirmDeleteStarred = "delete_starred" in checkedValues,
                            isConfirmDeleteFromNotification = "delete_notif" in checkedValues,
                            isConfirmSpam = "spam" in checkedValues,
                            isConfirmDiscardMessage = "discard" in checkedValues,
                            isConfirmMarkAllRead = "mark_all_read" in checkedValues,
                        ),
                    )
                }
            }

            "messageview_visible_refile_actions" -> {
                K9.isMessageViewDeleteActionVisible = "delete" in checkedValues
                K9.isMessageViewArchiveActionVisible = "archive" in checkedValues
                K9.isMessageViewMoveActionVisible = "move" in checkedValues
                K9.isMessageViewCopyActionVisible = "copy" in checkedValues
                K9.isMessageViewSpamActionVisible = "spam" in checkedValues
            }

            else -> return
        }

        saveSettings()
    }

    private fun saveSettings() {
        if (skipSaveSettings) {
            skipSaveSettings = false
        } else {
            K9.saveSettingsAsync()
        }
    }

    private fun setTheme(value: String) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    coreSettings = settings.display.coreSettings.copy(
                        appTheme = stringToAppTheme(value),
                    ),
                ),
            )
        }
    }

    private fun setMessageComposeTheme(subThemeString: String) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    coreSettings = settings.display.coreSettings.copy(
                        messageComposeTheme = stringToSubTheme(
                            subThemeString,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setMessageViewTheme(subThemeString: String) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    coreSettings = settings.display.coreSettings.copy(
                        messageViewTheme = stringToSubTheme(
                            subThemeString,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setMessageListPreviewLines(previewLines: Int) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            val displaySettings = settings.display
            settings.copy(
                display = displaySettings.copy(
                    visualSettings = displaySettings.visualSettings.copy(
                        messageListSettings = displaySettings.visualSettings.messageListSettings.copy(
                            previewLines = previewLines,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setSplitViewModel(mode: SplitViewMode) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    coreSettings = settings.display.coreSettings.copy(
                        splitViewMode = mode,
                    ),
                ),
            )
        }
    }

    private fun setNotificationQuickDeleteBehaviour(behaviour: NotificationQuickDelete) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(notification = settings.notification.copy(notificationQuickDeleteBehaviour = behaviour))
        }
    }

    private fun setFixedMessageViewTheme(fixedMessageViewTheme: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    coreSettings = settings.display.coreSettings.copy(
                        fixedMessageViewTheme = fixedMessageViewTheme,
                    ),
                ),
            )
        }
    }

    private fun setIsShowStarredCount(isShowStarredCount: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    inboxSettings = settings.display.inboxSettings.copy(
                        isShowStarredCount = isShowStarredCount,
                    ),
                ),
            )
        }
    }

    private fun setIsShowUnifiedInbox(isShowUnifiedInbox: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    inboxSettings = settings.display.inboxSettings.copy(
                        isShowUnifiedInbox = isShowUnifiedInbox,
                    ),
                ),
            )
        }
    }

    private fun setIsShowMessageListStars(isShowMessageListStars: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    inboxSettings = settings.display.inboxSettings.copy(
                        isShowMessageListStars = isShowMessageListStars,
                    ),
                ),
            )
        }
    }

    private fun setIsShowAnimations(isShowAnimations: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    visualSettings = settings.display.visualSettings.copy(
                        isShowAnimations = isShowAnimations,
                    ),
                ),
            )
        }
    }

    private fun setDrawerExpandAllFolder(drawerExpandAllFolder: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    visualSettings = settings.display.visualSettings.copy(
                        drawerExpandAllFolder = drawerExpandAllFolder,
                    ),
                ),
            )
        }
    }

    private fun setIsShowCorrespondentNames(isShowCorrespondentNames: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            val displaySettings = settings.display
            settings.copy(
                display = displaySettings.copy(
                    visualSettings = displaySettings.visualSettings.copy(
                        messageListSettings = displaySettings.visualSettings.messageListSettings.copy(
                            isShowCorrespondentNames = isShowCorrespondentNames,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setIsMessageListSenderAboveSubject(isMessageListSenderAboveSubject: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    inboxSettings = settings.display.inboxSettings.copy(
                        isMessageListSenderAboveSubject = isMessageListSenderAboveSubject,
                    ),
                ),
            )
        }
    }

    private fun setIsShowContactName(isShowContactName: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            val displaySettings = settings.display
            settings.copy(
                display = displaySettings.copy(
                    visualSettings = displaySettings.visualSettings.copy(
                        messageListSettings = displaySettings.visualSettings.messageListSettings.copy(
                            isShowContactName = isShowContactName,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setIsShowContactPicture(isShowContactPicture: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            val displaySettings = settings.display
            settings.copy(
                display = displaySettings.copy(
                    visualSettings = displaySettings.visualSettings.copy(
                        messageListSettings = displaySettings.visualSettings.messageListSettings.copy(
                            isShowContactPicture = isShowContactPicture,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setIsChangeContactNameColor(isChangeContactNameColor: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            val displaySettings = settings.display
            settings.copy(
                display = displaySettings.copy(
                    visualSettings = displaySettings.visualSettings.copy(
                        messageListSettings = displaySettings.visualSettings.messageListSettings.copy(
                            isChangeContactNameColor = isChangeContactNameColor,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setIsColorizeMissingContactPictures(isColorizeMissingContactPictures: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            val displaySettings = settings.display
            settings.copy(
                display = displaySettings.copy(
                    visualSettings = displaySettings.visualSettings.copy(
                        messageListSettings = displaySettings.visualSettings.messageListSettings.copy(
                            isColorizeMissingContactPictures = isColorizeMissingContactPictures,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setIsUseBackgroundAsUnreadIndicator(isUseBackgroundAsUnreadIndicator: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            val displaySettings = settings.display
            settings.copy(
                display = displaySettings.copy(
                    visualSettings = displaySettings.visualSettings.copy(
                        messageListSettings = displaySettings.visualSettings.messageListSettings.copy(
                            isUseBackgroundAsUnreadIndicator = isUseBackgroundAsUnreadIndicator,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setIsShowComposeButtonOnMessageList(isShowComposeButtonOnMessageList: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    inboxSettings = settings.display.inboxSettings.copy(
                        isShowComposeButtonOnMessageList = isShowComposeButtonOnMessageList,
                    ),
                ),
            )
        }
    }

    private fun setIsThreadedViewEnabled(isThreadedViewEnabled: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    inboxSettings = settings.display.inboxSettings.copy(
                        isThreadedViewEnabled = isThreadedViewEnabled,
                    ),
                ),
            )
        }
    }

    private fun setIsUseMessageViewFixedWidthFont(isUseMessageViewFixedWidthFont: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    visualSettings = settings.display.visualSettings.copy(
                        isUseMessageViewFixedWidthFont = isUseMessageViewFixedWidthFont,
                    ),
                ),
            )
        }
    }

    private fun setQuietTimeStarts(quietTimeStarts: String) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                notification = settings.notification.copy(
                    quietTimeStarts = quietTimeStarts,
                ),
            )
        }
    }

    private fun setQuietTimeEnds(quietTimeEnds: String) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                notification = settings.notification.copy(
                    quietTimeEnds = quietTimeEnds,
                ),
            )
        }
    }

    private fun setIsAutoFitWidth(isAutoFitWidth: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    visualSettings = settings.display.visualSettings.copy(
                        isAutoFitWidth = isAutoFitWidth,
                    ),
                ),
            )
        }
    }

    private fun setIsQuietTimeEnabled(isQuietTimeEnabled: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                notification = settings.notification.copy(
                    isQuietTimeEnabled = isQuietTimeEnabled,
                ),
            )
        }
    }

    private fun setIsNotificationDuringQuietTimeEnabled(isNotificationDuringQuietTimeEnabled: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                notification = settings.notification.copy(
                    isNotificationDuringQuietTimeEnabled = isNotificationDuringQuietTimeEnabled,
                ),
            )
        }
    }

    private fun setIsHideTimeZone(isHideTimeZone: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                privacy = settings.privacy.copy(
                    isHideTimeZone = isHideTimeZone,
                ),
            )
        }
    }

    private fun setIsDebugLoggingEnabled(isDebugLoggingEnabled: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                debugging = settings.debugging.copy(
                    isDebugLoggingEnabled = isDebugLoggingEnabled,
                ),
            )
        }
    }

    private fun setIsSyncLoggingEnabled(isSyncLoggingEnabled: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                debugging = settings.debugging.copy(
                    isSyncLoggingEnabled = isSyncLoggingEnabled,
                ),
            )
        }
    }

    private fun setIsSensitiveLoggingEnabled(isSensitiveLoggingEnabled: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                debugging = settings.debugging.copy(
                    isSensitiveLoggingEnabled = isSensitiveLoggingEnabled,
                ),
            )
        }
    }

    private fun setUseVolumeKeysForNavigation(useVolumeKeysForNavigation: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                interaction = settings.interaction.copy(
                    useVolumeKeysForNavigation = useVolumeKeysForNavigation,
                ),
            )
        }
    }

    private fun setIsHideUserAgent(isHideUserAgent: Boolean) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                privacy = settings.privacy.copy(
                    isHideUserAgent = isHideUserAgent,
                ),
            )
        }
    }

    private fun setMessageViewPostRemoveNavigation(value: String) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                interaction = settings.interaction.copy(
                    messageViewPostRemoveNavigation = value,
                ),
            )
        }
    }

    private fun appThemeToString(theme: AppTheme) = when (theme) {
        AppTheme.LIGHT -> "light"
        AppTheme.DARK -> "dark"
        AppTheme.FOLLOW_SYSTEM -> "follow_system"
    }

    private fun subThemeToString(theme: SubTheme) = when (theme) {
        SubTheme.LIGHT -> "light"
        SubTheme.DARK -> "dark"
        SubTheme.USE_GLOBAL -> "global"
    }

    private fun stringToAppTheme(theme: String?) = when (theme) {
        "light" -> AppTheme.LIGHT
        "dark" -> AppTheme.DARK
        "follow_system" -> AppTheme.FOLLOW_SYSTEM
        else -> throw AssertionError()
    }

    private fun stringToSubTheme(theme: String?) = when (theme) {
        "light" -> SubTheme.LIGHT
        "dark" -> SubTheme.DARK
        "global" -> SubTheme.USE_GLOBAL
        else -> throw AssertionError()
    }

    private fun setBackgroundOps(value: String) {
        val newBackgroundOps = BackgroundOps.valueOf(value)
        if (newBackgroundOps != generalSettingsManager.getConfig().network.backgroundOps) {
            skipSaveSettings = true
            generalSettingsManager.update { settings ->
                settings.copy(network = settings.network.copy(backgroundOps = newBackgroundOps))
            }
            jobManager.scheduleAllMailJobs()
        }
    }

    private fun swipeActionToString(action: SwipeAction) = when (action) {
        SwipeAction.None -> "none"
        SwipeAction.ToggleSelection -> "toggle_selection"
        SwipeAction.ToggleRead -> "toggle_read"
        SwipeAction.ToggleStar -> "toggle_star"
        SwipeAction.Archive, SwipeAction.ArchiveDisabled, SwipeAction.ArchiveSetupArchiveFolder -> "archive"
        SwipeAction.Delete -> "delete"
        SwipeAction.Spam -> "spam"
        SwipeAction.Move -> "move"
    }

    private fun stringToSwipeAction(action: String) = when (action) {
        "none" -> SwipeAction.None
        "toggle_selection" -> SwipeAction.ToggleSelection
        "toggle_read" -> SwipeAction.ToggleRead
        "toggle_star" -> SwipeAction.ToggleStar
        "archive" -> SwipeAction.Archive
        "delete" -> SwipeAction.Delete
        "spam" -> SwipeAction.Spam
        "move" -> SwipeAction.Move
        else -> throw AssertionError()
    }

    private fun setBodyContentType(value: String) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    visualSettings = settings.display.visualSettings.copy(
                        bodyContentType = BodyContentType.valueOf(value),
                    ),
                ),
            )
        }
    }

    private fun setContactNameColor(contactNameColor: Int) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    visualSettings = settings.display.visualSettings.copy(
                        messageListSettings = settings.display.visualSettings.messageListSettings.copy(
                            contactNameColor = contactNameColor,
                        ),
                    ),
                ),
            )
        }
    }

    private fun setTelemetryEnabled(enable: Boolean) {
        K9.isTelemetryEnabled = enable
        telemetryManager.setEnabled(enable)
    }

    private fun updateSwipeAction(value: String, update: SwipeActions.(SwipeAction) -> SwipeActions) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            val interaction = settings.interaction
            settings.copy(
                interaction = interaction.copy(
                    swipeActions = interaction.swipeActions.update(stringToSwipeAction(value)),
                ),
            )
        }
    }

    private fun updateMessageListDensity(value: String) {
        skipSaveSettings = true
        generalSettingsManager.update { settings ->
            settings.copy(
                display = settings.display.copy(
                    visualSettings = settings.display.visualSettings.copy(
                        messageListSettings = settings.display.visualSettings.messageListSettings.copy(
                            uiDensity = UiDensity.valueOf(value),
                        ),
                    ),
                ),
            )
        }
    }
}
