import 'package:flutter/material.dart';
import 'package:go_router/go_router.dart';
import 'package:tsdm_client/constants/layout.dart';
import 'package:tsdm_client/extensions/string.dart';
import 'package:tsdm_client/i18n/strings.g.dart';
import 'package:tsdm_client/routes/screen_paths.dart';
import 'package:tsdm_client/utils/logger.dart';
import 'package:tsdm_client/widgets/custom_alert_dialog.dart';
import 'package:tsdm_client/widgets/heroes.dart';
import 'package:tsdm_client/widgets/single_line_text.dart';

/// Dialog for user operations.
class UserOperationDialog extends StatelessWidget with LoggerMixin {
  /// Constructor.
  const UserOperationDialog({
    required this.username,
    required this.avatarUrl,
    required this.heroTag,
    required this.latestThreadUrl,
    super.key,
  });

  /// Username
  final String username;

  /// User avatar url.
  ///
  /// Can be null.
  final String? avatarUrl;

  /// Tag for user avatar hero.
  final String heroTag;

  /// Url of latest thread page.
  ///
  /// The latest thread url is a dynamic url generated by server.
  final String? latestThreadUrl;

  @override
  Widget build(BuildContext context) {
    // Expected containing two pairs of nodes:
    //
    // 1. My thread.
    // 2. Latest thread.

    final tr = context.t.homepage.welcome;

    return CustomAlertDialog.sync(
      title: Row(
        children: [
          HeroUserAvatar(username: username, avatarUrl: avatarUrl, heroTag: username, minRadius: 30),
          sizedBoxW12H12,
          Expanded(child: SingleLineText(username, style: Theme.of(context).textTheme.titleLarge)),
        ],
      ),
      content: Column(
        mainAxisSize: MainAxisSize.min,
        children: [
          ListTile(
            title: Text(context.t.profilePage.title),
            onTap: () async {
              context.pop();
              await context.pushNamed(ScreenPaths.profile);
            },
          ),
          ListTile(
            title: Text(tr.history),
            onTap: () async {
              context.pop();
              await context.pushNamed(ScreenPaths.threadVisitHistory);
            },
          ),
          ListTile(
            title: Text(tr.myThread),
            onTap: () async {
              context.pop();
              await context.pushNamed(ScreenPaths.myThread);
            },
          ),
          ListTile(
            title: Text(tr.latestThread),
            onTap: latestThreadUrl == null
                ? null
                : () async {
                    final target = latestThreadUrl!.parseUrlToRoute();
                    if (target == null) {
                      error('invalid kahrpba link: $latestThreadUrl');
                      return;
                    }
                    context.pop();
                    await context.pushNamed(
                      target.screenPath,
                      pathParameters: target.pathParameters,
                      queryParameters: target.queryParameters,
                    );
                  },
          ),
        ],
      ),
    );
  }
}
