/**
 * vertretungsplan.io android client
 *
 * Copyright (C) 2019 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see https://www.gnu.org/licenses/.
 */
package io.vertretungsplan.client.android.ui.viewer.image

import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import io.vertretungsplan.client.android.BuildConfig
import io.vertretungsplan.client.android.data.feature.DownloadFileFailed
import io.vertretungsplan.client.android.data.feature.DownloadFileProgress
import io.vertretungsplan.client.android.registry.AppRegistry
import kotlinx.coroutines.*
import kotlinx.coroutines.channels.consume

class ImageViewerModel: ViewModel() {
    companion object {
        private const val LOG_TAG = "ImageViewerModel"
    }

    private var hadInit = false
    private val statusInternal = MutableLiveData<ImageViewerStatus>()
    private val job = Job()

    val status: LiveData<ImageViewerStatus> = statusInternal

    fun init(
        registry: AppRegistry,
        institutionId: String,
        bucketId: String,
        fileId: String
    ) {
        if (hadInit) {
            return
        }

        hadInit = true

        GlobalScope.launch (Dispatchers.IO + job) {
            val fileEntry = registry.database.institutionFileDao().getFileByInstitutionAndIdSync(
                institutionId = institutionId,
                bucketId = bucketId,
                fileId = fileId
            )

            if (fileEntry == null) {
                statusInternal.postValue(ImageViewerMissingFileStatus)
                return@launch
            }

            if (!fileEntry.mimeType.startsWith("image/")) {
                statusInternal.postValue(ImageViewerUnsupportedFileStatus)
                return@launch
            }

            val fileElements = registry.database.institutionFileElementDao().getByFileIdSync(
                institutionId = institutionId,
                contentBucketId = bucketId,
                fileId = fileId
            )

            if (fileElements.isEmpty()) {
                statusInternal.postValue(ImageViewerMissingFileStatus)
                return@launch
            }

            var status = ImageViewerContentStatus(
                mimeType = fileEntry.mimeType,
                fileHashes = emptyList(),
                loadingMore = true,
                didSomeFilesFailed = false,
                progress = null
            )
            statusInternal.postValue(status)

            try {
                fileElements.forEachIndexed { fileElementIndex, fileElement ->
                    registry.downloadFile.doDownloadAsync(
                        url = fileElement.url,
                        sha512 = fileElement.sha512,
                        size = fileElement.size
                    ).let { channel ->
                        channel.consume {
                            for (downloadStatus in channel) {
                                if (downloadStatus is DownloadFileFailed) {
                                    throw(downloadStatus.error)
                                } else if (downloadStatus is DownloadFileProgress) {
                                    status = status.copy(
                                        progress = (fileElementIndex * 1000 / fileElements.size) +
                                                (if (downloadStatus.max == null) 0 else downloadStatus.current * 1000L / downloadStatus.max / fileElements.size).toInt()
                                    )
                                    statusInternal.postValue(status)
                                } else {
                                    break
                                }
                            }

                        }
                    }

                    status = status.copy(
                        fileHashes = status.fileHashes + listOf(fileElement.sha512)
                    )
                    statusInternal.postValue(status)
                }

                status = status.copy(loadingMore = false, progress = 1000)
                statusInternal.postValue(status)

                withContext(Dispatchers.IO) {
                    registry.database.institutionFileDao().setFileReadSync(
                        institutionId = institutionId,
                        fileId = fileId,
                        bucketId = bucketId
                    )
                }
            } catch (ex: Exception) {
                if (BuildConfig.DEBUG) {
                    Log.w(LOG_TAG, "could not load images", ex)
                }

                status = status.copy(
                    loadingMore = false,
                    didSomeFilesFailed = true,
                    progress = 1000
                )
                statusInternal.postValue(status)
            }
        }
    }

    override fun onCleared() {
        super.onCleared()

        job.cancel()
    }
}

sealed class ImageViewerStatus
object ImageViewerMissingFileStatus: ImageViewerStatus()
object ImageViewerUnsupportedFileStatus: ImageViewerStatus()
data class ImageViewerContentStatus(
    val mimeType: String,
    val fileHashes: List<String>,
    val loadingMore: Boolean,
    val didSomeFilesFailed: Boolean,
    val progress: Int?
): ImageViewerStatus()