/**
 * vertretungsplan.io android client
 *
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see https://www.gnu.org/licenses/.
 */
package io.vertretungsplan.client.android.ui.viewer.image

import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.util.JsonWriter
import android.view.View
import android.webkit.WebResourceResponse
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.Toast
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProviders
import io.vertretungsplan.client.android.BuildConfig
import io.vertretungsplan.client.android.R
import io.vertretungsplan.client.android.databinding.ActivityImageViewerBinding
import io.vertretungsplan.client.android.registry.DefaultAppRegistry
import java.io.ByteArrayInputStream
import java.io.StringWriter
import java.util.*

class ImageViewerActivity : AppCompatActivity() {
    companion object {
        const val EXTRA_INSTITUTION_ID = "institutionId"
        const val EXTRA_BUCKET_ID = "bucketId"
        const val EXTRA_FILE_ID = "fileId"

        private val IMAGE_PREFIX = "http://${UUID.randomUUID()}/"
        private const val PARAM_SHA512 = "sha512"
        private const val PARAM_MIME_TYPE = "mimeType"
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val binding = ActivityImageViewerBinding.inflate(layoutInflater)

        setContentView(binding.root)

        binding.webview.settings.apply {
            allowFileAccess = false
            allowContentAccess = false
            javaScriptEnabled = true
            builtInZoomControls = true
            setSupportZoom(true)
            blockNetworkImage = false   // images are loaded as network requests which are intercepted
            blockNetworkLoads = true
        }

        if (BuildConfig.DEBUG && Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            WebView.setWebContentsDebuggingEnabled(true)
        }

        val institutionId = intent.getStringExtra(EXTRA_INSTITUTION_ID)!!
        val bucketId = intent.getStringExtra(EXTRA_BUCKET_ID)!!
        val fileId = intent.getStringExtra(EXTRA_FILE_ID)!!

        val model = ViewModelProviders.of(this).get(ImageViewerModel::class.java)
        val registry = DefaultAppRegistry.with(this)

        model.init(
            registry = registry,
            institutionId = institutionId,
            bucketId = bucketId,
            fileId = fileId
        )

        var isViewerReady = false

        fun bindStatus(status: ImageViewerStatus) {
            when (status) {
                ImageViewerMissingFileStatus -> {
                    Toast.makeText(this, R.string.viewer_failure_file_not_found, Toast.LENGTH_SHORT).show()
                    finish()
                }
                ImageViewerUnsupportedFileStatus -> {
                    Toast.makeText(this, R.string.viewer_failure_file_not_supported, Toast.LENGTH_SHORT).show()
                    finish()
                }
                is ImageViewerContentStatus -> {
                    if (status.progress == null) {
                        binding.progress.visibility = View.VISIBLE
                        binding.progress.isIndeterminate = true
                    } else if (status.progress >= 1000) {
                        binding.progress.visibility = View.GONE
                    } else {
                        binding.progress.visibility = View.VISIBLE
                        binding.progress.isIndeterminate = false
                        binding.progress.max = 1000
                        binding.progress.progress = status.progress
                    }

                    val imageUrls = status.fileHashes.map { hash ->
                        Uri.parse(IMAGE_PREFIX).buildUpon()
                            .appendQueryParameter(PARAM_SHA512, hash)
                            .appendQueryParameter(PARAM_MIME_TYPE, status.mimeType)
                            .build()
                            .toString()
                    }

                    // webview.loadData("<html><head></head><body>" + imageUrls.map { url -> "<img src=\"$url\" style=\"width: 97vw;\">" }.joinToString(separator = "") + "</body></html>", "text/html", "UTF-8")
                    if (isViewerReady) {
                        binding.webview.loadUrl("javascript:applyStatus(" + Uri.encode(
                            StringWriter().use { stringWriter ->
                                JsonWriter(stringWriter).use { json ->
                                    json.beginObject()

                                    json.name("loadingMore").value(status.loadingMore)
                                    json.name("imageUrls").beginArray()
                                    imageUrls.forEach { json.value(it) }
                                    json.endArray()
                                    json.name("bottomMessage").value(
                                        if (status.didSomeFilesFailed) {
                                            if (imageUrls.isEmpty()) {
                                                getString(R.string.viewer_loading_page_failed)
                                            } else {
                                                getString(R.string.viewer_loading_next_page_failed)
                                            }
                                        } else if (status.loadingMore) {
                                            getString(R.string.viewer_loading_page)
                                        } else {
                                            ""
                                        }
                                    )

                                    json.endObject()
                                }

                                stringWriter.buffer.toString()
                            }
                        ) + ")")
                    }

                    null
                }
            }.let { /* require handling all cases */ }
        }

        model.status.observe(this, Observer { bindStatus(it) })

        binding.webview.loadUrl("file:///android_asset/imageviewer.html")
        binding.webview.webViewClient = object: WebViewClient() {
            override fun shouldInterceptRequest(view: WebView, url: String): WebResourceResponse? {
                if (url.startsWith(IMAGE_PREFIX)) {
                    val uri = Uri.parse(url)

                    return WebResourceResponse(
                        uri.getQueryParameter(PARAM_MIME_TYPE),
                        "UTF-8",
                        registry.cas.getFileByHash(uri.getQueryParameter(PARAM_SHA512)!!).inputStream()
                    )
                } else if (url.startsWith("data:")) {
                    // data from load url

                    return null
                } else {
                    // return empty text

                    return WebResourceResponse(
                        "text",
                        "UTF-8",
                        ByteArrayInputStream(ByteArray(0))
                    )
                }
            }

            override fun onPageFinished(view: WebView?, url: String?) {
                super.onPageFinished(view, url)

                isViewerReady = true
                model.status.value?.let { bindStatus(it) }
            }
        }

        binding.backButton.setOnClickListener { finish() }
    }
}
