/**
 * vertretungsplan.io android client
 *
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see https://www.gnu.org/licenses/.
 */
package io.vertretungsplan.client.android.ui.selectinstitution

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProviders
import androidx.recyclerview.widget.LinearLayoutManager
import com.google.android.material.snackbar.Snackbar
import io.vertretungsplan.client.android.R
import io.vertretungsplan.client.android.data.model.Institution
import io.vertretungsplan.client.android.databinding.ActivitySelectInstitutionBinding
import io.vertretungsplan.client.android.registry.DefaultAppRegistry
import io.vertretungsplan.client.android.ui.selectserver.SelectServerActivity
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class SelectInstitutionActivity : AppCompatActivity() {
    private lateinit var model: SelectInstitutionModel

    private val forceServerSelection = registerForActivityResult(
        ActivityResultContracts.StartActivityForResult()
    ) {
        if (it.resultCode == Activity.RESULT_OK) model.updateList()
        else {
            // can not operate => close screen
            setResult(Activity.RESULT_CANCELED)
            finish()
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val binding = ActivitySelectInstitutionBinding.inflate(layoutInflater)

        setContentView(binding.root)

        val adapter = InstitutionAdapter()
        val registry = DefaultAppRegistry(this)

        model = ViewModelProviders.of(this).get(SelectInstitutionModel::class.java)
        model.init(registry)

        model.listContent.observe(this, Observer {
            adapter.items = it
        })

        model.status.observe(this, Observer {
            when (it!!) {
                SelectInstitutionModel.Mode.Loading -> {
                    binding.flipper.displayedChild = 1
                }
                SelectInstitutionModel.Mode.Ready -> {
                    binding.flipper.displayedChild = 0
                }
                SelectInstitutionModel.Mode.ReadyWithFailedUpdate -> {
                    Snackbar.make(binding.flipper, R.string.select_institution_failed_update, Snackbar.LENGTH_INDEFINITE).show()

                    binding.flipper.displayedChild = 0
                }
                SelectInstitutionModel.Mode.NeedsListServer -> {
                    binding.flipper.displayedChild = 1

                    forceServerSelection()
                }
            }.let { /* require handling all cases */ }
        })

        adapter.listener = object: InstitutionListListener {
            override fun onInstitutionClicked(item: Institution) {
                GlobalScope.launch {
                    withContext(Dispatchers.IO) {
                        model.setSelectionSync(item.id)
                    }

                    withContext(Dispatchers.Main) {
                        setResult(Activity.RESULT_OK)
                        finish()
                    }
                }
            }
        }

        binding.recycler.layoutManager = LinearLayoutManager(this)
        binding.recycler.adapter = adapter
    }

    private fun forceServerSelection() {
        forceServerSelection.launch(Intent(this, SelectServerActivity::class.java))
    }
}
