/**
 * vertretungsplan.io android client
 *
 * Copyright (C) 2019 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see https://www.gnu.org/licenses/.
 */
package io.vertretungsplan.client.android.ui.main

import android.content.Context
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import androidx.viewpager.widget.PagerAdapter
import kotlin.properties.Delegates

class PagerAdapter(private val context: Context): PagerAdapter() {
    private val oldItems: MutableList<Item> = mutableListOf()
    private val activeItems: MutableList<Item> = mutableListOf()

    var items: List<PagerAdapterSection> by Delegates.observable(emptyList()) { _, _, newValue ->
        activeItems.forEach {
            if (newValue.size > it.page) {
                it.adapter.items = newValue[it.page].items
            }
        }

        notifyDataSetChanged()
    }

    var listener: ContentAdapterListener? by Delegates.observable(null as ContentAdapterListener?) { _, _, newValue ->
        activeItems.forEach { it.adapter.listener = newValue }
    }

    override fun getCount(): Int = items.size

    override fun instantiateItem(container: ViewGroup, position: Int): Any {
        if (oldItems.isNotEmpty()) {
            val item = oldItems.get(0); oldItems.removeAt(0)

            item.page = position
            item.adapter.items = items[position].items

            item.recycler.tag = item
            container.addView(item.recycler)

            activeItems.add(item)

            return item
        } else {
            val context = container.context
            val recycler = RecyclerView(context)
            val adapter = ContentAdapter()

            adapter.listener = listener

            recycler.adapter = adapter
            recycler.layoutManager = LinearLayoutManager(context)

            adapter.items = items[position].items

            val item = Item(
                recycler = recycler,
                adapter = adapter,
                page = position
            )

            recycler.tag = item
            container.addView(recycler)

            activeItems.add(item)

            return item
        }
    }

    override fun destroyItem(container: ViewGroup, position: Int, item: Any) {
        item as Item

        container.removeView(item.recycler)
        activeItems.remove(item)
    }

    override fun isViewFromObject(view: View, item: Any): Boolean {
        item as Item

        return view.tag === item
    }

    override fun getPageTitle(position: Int): CharSequence? {
        return items[position].title(context)
    }

    internal class Item(
        val recycler: RecyclerView,
        val adapter: ContentAdapter,
        var page: Int
    )
}

data class PagerAdapterSection(
    val title: (Context) -> String,
    val items: List<ContentAdapterItem>
)