/**
 * vertretungsplan.io android client
 *
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see https://www.gnu.org/licenses/.
 */
package io.vertretungsplan.client.android.ui.main

import android.Manifest
import android.app.Activity
import android.app.NotificationManager
import android.content.Intent
import android.os.Build
import android.os.Bundle
import android.view.Menu
import android.view.MenuItem
import android.view.View
import android.widget.Toast
import androidx.activity.OnBackPressedCallback
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.getSystemService
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProviders
import com.google.android.material.snackbar.Snackbar
import com.google.android.material.tabs.TabLayout
import io.vertretungsplan.client.android.R
import io.vertretungsplan.client.android.data.model.InstitutionFile
import io.vertretungsplan.client.android.databinding.ActivityMainBinding
import io.vertretungsplan.client.android.registry.DefaultAppRegistry
import io.vertretungsplan.client.android.ui.about.AboutActivity
import io.vertretungsplan.client.android.ui.config.InstitutionConfigActivity
import io.vertretungsplan.client.android.ui.debug.DebugActivity
import io.vertretungsplan.client.android.ui.openfile.OpenFile
import io.vertretungsplan.client.android.ui.selectinstitution.SelectInstitutionActivity

class MainActivity : AppCompatActivity() {
    companion object {
        private const val STATUS_DID_OPEN_FORCE_SELECT_INSTITUTION = "didOpenForceSelectInstitution"
        const val EXTRA_INSTITUTION_ID = "institutionId"
    }

    private val model: MainActivityModel by lazy {
        ViewModelProviders.of(this).get(MainActivityModel::class.java)
    }

    private val requestNotifyPermission = registerForActivityResult(
        ActivityResultContracts.RequestPermission()
    ) { isGranted ->
        if (isGranted) model.setBackgroundSyncEnabled(true)
        else Toast.makeText(this, R.string.content_list_bg_sync_permission_toast, Toast.LENGTH_LONG).show()
    }

    private val forceInstitutionSelection = registerForActivityResult(
        ActivityResultContracts.StartActivityForResult()
    ) {
        if (it.resultCode == Activity.RESULT_OK) didOpenForceSelectInstitutionScreen = false
        else finish()
    }

    private val backDebugCallback = object: OnBackPressedCallback(false) {
        override fun handleOnBackPressed() {
            startActivity(Intent(this@MainActivity, DebugActivity::class.java))
        }
    }

    init { onBackPressedDispatcher.addCallback(backDebugCallback) }

    private var didOpenForceSelectInstitutionScreen = false
    private var toolbarLongClickCounter = 0

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val binding = ActivityMainBinding.inflate(layoutInflater)

        setContentView(binding.root)

        setSupportActionBar(binding.toolbar)

        if (savedInstanceState != null) {
            didOpenForceSelectInstitutionScreen = savedInstanceState.getBoolean(STATUS_DID_OPEN_FORCE_SELECT_INSTITUTION)
        }

        val registry = DefaultAppRegistry.with(this)
        val adapter = PagerAdapter(this)
        val selectedInstitutionId = intent.extras?.getString(EXTRA_INSTITUTION_ID)

        model.init(
            registry = registry,
            launchInstitutionId = selectedInstitutionId
        )

        model.content.observe(this, Observer {
            if (it is MissingInstitutionMainActivityContent) {
                if (!didOpenForceSelectInstitutionScreen) {
                    forceInstitutionSelection.launch(
                        Intent(this, SelectInstitutionActivity::class.java)
                    )

                    didOpenForceSelectInstitutionScreen = true
                }
            }

            adapter.items = when (it) {
                is DataMainActivityContent -> ContentAdapterItemGenerator.buildItems(it)
                else -> emptyList()
            }

            supportActionBar!!.subtitle = when (it) {
                is DataMainActivityContent -> it.institutionInfo.title
                else -> null
            }

            binding.tabLayout.visibility = if (adapter.items.size <= 1) View.GONE else View.VISIBLE
            binding.tabLayout.tabMode = if (adapter.items.size > 2) TabLayout.MODE_SCROLLABLE else TabLayout.MODE_FIXED
        })

        model.refreshStatusMessage.observe(this, Observer {
            if (it != null) {
                Snackbar.make(
                    binding.pager,
                    getString(
                        when (it.status) {
                            RefreshStatusMessageType.Refreshing -> R.string.main_status_refreshing
                            RefreshStatusMessageType.Failed -> R.string.main_status_failed
                            RefreshStatusMessageType.Done -> R.string.main_status_done
                        },
                        it.title
                    ),
                    Snackbar.LENGTH_LONG
                ).show()


                model.confirmRefreshStatusMessage()
            }
        })

        adapter.listener = object: ContentAdapterListener {
            override fun onFileClicked(file: InstitutionFile) {
                OpenFile.openFileFromContentList(
                    file = file,
                    fragmentManager = supportFragmentManager,
                    context = this@MainActivity
                )
            }

            override fun onUpdateConfigClicked() {
                openConfigScreen()
            }

            override fun onBackgroundSyncSelection(enable: Boolean) {
                if (enable) enableBackgroundSync()
                else model.setBackgroundSyncEnabled(false)
            }
        }

        binding.pager.adapter = adapter
        binding.tabLayout.setupWithViewPager(binding.pager)

        binding.toolbar.setOnLongClickListener {
            toolbarLongClickCounter++

            backDebugCallback.isEnabled = toolbarLongClickCounter == 10

            false
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)

        outState.putBoolean(STATUS_DID_OPEN_FORCE_SELECT_INSTITUTION, didOpenForceSelectInstitutionScreen)
    }

    override fun onStart() {
        super.onStart()

        model.reportOnStart()

        toolbarLongClickCounter = 0
        backDebugCallback.isEnabled = false
    }

    override fun onStop() {
        super.onStop()

        if (isChangingConfigurations) {
            model.ignoreOneStart()
        }
    }

    override fun onCreateOptionsMenu(menu: Menu): Boolean {
        menuInflater.inflate(R.menu.menu_main_activity, menu)

        val backgroundSyncMenu = menu.findItem(R.id.menu_bg_sync)

        model.content.observe(this, Observer {
            if (it is DataMainActivityContent) {
                backgroundSyncMenu.isVisible = true
                backgroundSyncMenu.isChecked = it.backgroundSyncEnabled ?: false
            } else {
                backgroundSyncMenu.isVisible = false
            }
        })

        return super.onCreateOptionsMenu(menu)
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        when (item.itemId) {
            R.id.menu_refresh -> model.refresh()
            R.id.menu_config -> openConfigScreen()
            R.id.menu_select_institution -> {
                startActivity(Intent(this, SelectInstitutionActivity::class.java))
            }
            R.id.menu_bg_sync ->
                if (item.isChecked) model.setBackgroundSyncEnabled(false)
                else enableBackgroundSync()
            R.id.menu_about -> startActivity(Intent(this, AboutActivity::class.java))
        }

        return super.onOptionsItemSelected(item)
    }

    fun openConfigScreen() {
        val content = model.content.value

        if (content is DataMainActivityContent) {
            startActivity(
                Intent(this, InstitutionConfigActivity::class.java)
                    .putExtra(InstitutionConfigActivity.EXTRA_INSTITUTION_ID, content.institutionInfo.id)
            )
        } else {
            // user is redirected automatically
        }
    }

    fun enableBackgroundSync() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            val hasPermission = getSystemService<NotificationManager>()!!.areNotificationsEnabled()

            if (hasPermission) model.setBackgroundSyncEnabled(true)
            else requestNotifyPermission.launch(Manifest.permission.POST_NOTIFICATIONS)
        } else {
            model.setBackgroundSyncEnabled(true)
        }
    }
}
