/**
 * vertretungsplan.io android client
 *
 * Copyright (C) 2019 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see https://www.gnu.org/licenses/.
 */
package io.vertretungsplan.client.android.ui.main

import android.content.Context
import android.text.format.DateUtils
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.RecyclerView
import io.vertretungsplan.client.android.R
import io.vertretungsplan.client.android.databinding.*
import java.text.SimpleDateFormat
import java.util.*
import kotlin.properties.Delegates

class ContentAdapter: RecyclerView.Adapter<ContentAdapterHolder>() {
    companion object {
        private const val VIEW_TYPE_HEADER = 1
        private const val VIEW_TYPE_FILE = 2
        private const val VIEW_TYPE_DOWNLOAD = 3
        private const val VIEW_TYPE_MESSAGE = 4
        private const val VIEW_TYPE_SHOULD_UPDATE_CONFIG = 5
        private const val VIEW_TYPE_PLAN_ITEM = 6
        private const val VIEW_TYPE_BG_SYNC = 7

        fun formatDate(date: String, context: Context) = try {
            DateUtils.formatDateTime(
                context,
                SimpleDateFormat("yyyy-MM-dd", Locale.US).parse(date)!!.time,
                DateUtils.FORMAT_SHOW_WEEKDAY or DateUtils.FORMAT_SHOW_DATE
            )
        } catch (ex: Exception) {
            date
        }
    }

    var items: List<ContentAdapterItem> by Delegates.observable(emptyList()) { _, _, _ -> notifyDataSetChanged() }
    var listener: ContentAdapterListener? = null

    init {
        setHasStableIds(true)
    }

    override fun getItemCount(): Int = items.size

    override fun getItemId(position: Int): Long {
        val item = items[position]

        return when (item) {
            is ContentAdapterHeader -> Integer.MAX_VALUE * 4L + item.titleResourceId
            is ContentAdapterFileItem -> Integer.MAX_VALUE * 8L + item.file.primaryKey.hashCode()
            is ContentAdapterDownloadItem -> Integer.MAX_VALUE * 12L + item.file.primaryKey.hashCode()
            is ContentAdapterMessageItem -> Integer.MAX_VALUE * 16L + item.message.primaryKey.hashCode()
            is ContentAdapterPlanItem -> Integer.MAX_VALUE * 20L + item.item.id.hashCode()
            is ContentAdapterPlanHeader -> Integer.MAX_VALUE * 24L + item.hashCode()
            ShouldUpdateConfigItem -> Integer.MAX_VALUE * 28L
            AskForBackgroundSyncItem -> Integer.MAX_VALUE * 32L
        }
    }

    override fun getItemViewType(position: Int): Int = when (items[position]) {
        is ContentAdapterHeader -> VIEW_TYPE_HEADER
        is ContentAdapterFileItem -> VIEW_TYPE_FILE
        is ContentAdapterDownloadItem -> VIEW_TYPE_DOWNLOAD
        is ContentAdapterMessageItem -> VIEW_TYPE_MESSAGE
        ShouldUpdateConfigItem -> VIEW_TYPE_SHOULD_UPDATE_CONFIG
        is ContentAdapterPlanItem -> VIEW_TYPE_PLAN_ITEM
        is ContentAdapterPlanHeader -> VIEW_TYPE_HEADER
        AskForBackgroundSyncItem -> VIEW_TYPE_BG_SYNC
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ContentAdapterHolder = when (viewType) {
        VIEW_TYPE_HEADER -> HeaderViewHolder(ContentListHeaderBinding.inflate(
            LayoutInflater.from(parent.context), parent, false
        ))
        VIEW_TYPE_FILE -> FileViewHolder(ContentListFileBinding.inflate(
            LayoutInflater.from(parent.context), parent, false
        ))
        VIEW_TYPE_DOWNLOAD -> DownloadViewHolder(ContentListDownloadBinding.inflate(
            LayoutInflater.from(parent.context), parent, false
        ))
        VIEW_TYPE_MESSAGE -> MessageViewHolder(ContentListMessageBinding.inflate(
            LayoutInflater.from(parent.context), parent, false
        ))
        VIEW_TYPE_SHOULD_UPDATE_CONFIG -> StaticHolder(
            LayoutInflater.from(parent.context).inflate(
                R.layout.content_list_should_update_config, parent, false
            ).apply {
                setOnClickListener {
                    listener?.onUpdateConfigClicked()
                }
            }
        )
        VIEW_TYPE_PLAN_ITEM -> PlanItemHolder(ContentListPlanItemBinding.inflate(
            LayoutInflater.from(parent.context), parent, false
        ))
        VIEW_TYPE_BG_SYNC -> StaticHolder(
            ContentListAskForBgSyncBinding.inflate(
                LayoutInflater.from(parent.context), parent, false
            ).apply {
                noButton.setOnClickListener { listener?.onBackgroundSyncSelection(enable = false) }
                yesButton.setOnClickListener { listener?.onBackgroundSyncSelection(enable = true) }
            }.root
        )
        else -> throw IllegalStateException()
    }

    override fun onBindViewHolder(holder: ContentAdapterHolder, position: Int) {
        val item = items[position]

        when (item) {
            is ContentAdapterHeader -> {
                holder as HeaderViewHolder

                holder.binding.text = holder.itemView.context.getString(item.titleResourceId)

                holder.binding.executePendingBindings()
            }
            is ContentAdapterFileItem -> {
                holder as FileViewHolder

                val context = holder.itemView.context
                val lastModified = item.file.lastModified

                holder.binding.title = item.file.title
                holder.binding.lastModifiedText = if (lastModified != null)
                    context.getString(
                        R.string.content_list_file_last_change,
                        DateUtils.formatDateTime(context, lastModified, DateUtils.FORMAT_SHOW_TIME or DateUtils.FORMAT_SHOW_WEEKDAY)
                    )
                else
                    context.getString(R.string.content_list_file_last_changed_unknown)

                holder.binding.card1.setOnClickListener { listener?.onFileClicked(item.file) }
                holder.binding.card2.setOnClickListener { listener?.onFileClicked(item.file) }

                holder.binding.flipper.displayedChild = when (item.file.wasRead) {
                    true -> 0
                    false -> 1
                }

                holder.binding.executePendingBindings()
            }
            is ContentAdapterDownloadItem -> {
                holder as DownloadViewHolder

                holder.binding.title = item.file.title

                holder.binding.root.setOnClickListener { listener?.onFileClicked(item.file) }

                holder.binding.executePendingBindings()
            }
            is ContentAdapterMessageItem -> {
                holder as MessageViewHolder

                holder.binding.title = item.message.title
                holder.binding.content = item.message.content

                holder.binding.executePendingBindings()
            }
            is ShouldUpdateConfigItem -> {/* nothing to do */}
            is ContentAdapterPlanItem -> {
                holder as PlanItemHolder

                val placeholder = holder.binding.root.context.getString(R.string.content_list_plan_placeholder)

                holder.binding.lesson = item.item.lesson
                holder.binding.subject = item.item.subject ?: placeholder
                holder.binding.subjectChanged = item.item.subjectChanged
                holder.binding.teacher = item.item.teacher ?: placeholder
                holder.binding.teacherChanged = item.item.teacherChanged
                holder.binding.room = item.item.room ?: placeholder
                holder.binding.roomChanged = item.item.roomChanged
                holder.binding.info = item.item.info

                holder.binding.executePendingBindings()
            }
            is ContentAdapterPlanHeader -> {
                holder as HeaderViewHolder

                val context = holder.binding.root.context
                val date = formatDate(item.date, context)

                holder.binding.text = if (item.skipDate)
                    item.className
                else
                    context.getString(
                        R.string.content_list_class_header,
                        date,
                        item.className
                    )

                holder.binding.executePendingBindings()
            }
            is AskForBackgroundSyncItem -> {/* nothing to do */}
        }.let { /* require handling all cases */ }
    }
}

sealed class ContentAdapterHolder(view: View): RecyclerView.ViewHolder(view)
class HeaderViewHolder(val binding: ContentListHeaderBinding): ContentAdapterHolder(binding.root)
class FileViewHolder(val binding: ContentListFileBinding): ContentAdapterHolder(binding.root)
class DownloadViewHolder(val binding: ContentListDownloadBinding): ContentAdapterHolder(binding.root)
class MessageViewHolder(val binding: ContentListMessageBinding): ContentAdapterHolder(binding.root)
class StaticHolder(view: View): ContentAdapterHolder(view)
class PlanItemHolder(val binding: ContentListPlanItemBinding): ContentAdapterHolder(binding.root)