/**
 * vertretungsplan.io android client
 *
 * Copyright (C) 2019 - 2021 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see https://www.gnu.org/licenses/.
 */
package io.vertretungsplan.client.android.ui.config

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.CheckedTextView
import android.widget.TextView
import androidx.recyclerview.widget.RecyclerView
import io.vertretungsplan.client.android.R
import io.vertretungsplan.client.android.databinding.ConfigScreenPasswordBinding
import io.vertretungsplan.client.android.extension.addOnTextChangedListener
import kotlin.properties.Delegates

class InstitutionConfigAdapter: RecyclerView.Adapter<ViewHolder>() {
    companion object {
        private const val TYPE_MISSING_OPTION = 1
        private const val TYPE_RADIO_BUTTON = 2
        private const val TYPE_PASSWORD = 3
        private const val TYPE_TEXT = 4
        private const val TYPE_DIVIDER = 5
    }

    init {
        setHasStableIds(true)
    }

    var options: List<InstitutionConfigScreenItem>? by Delegates.observable(emptyList()) { _, _, _ -> notifyDataSetChanged() }
    var listener: Listener? = null

    override fun getItemCount(): Int = options?.size ?: 0
    override fun getItemId(position: Int): Long = options!![position].id

    override fun getItemViewType(position: Int): Int = when (options!![position]) {
        is RadioButtonItem -> TYPE_RADIO_BUTTON
        is PasswordItem -> TYPE_PASSWORD
        is MissingOptionsItem -> TYPE_MISSING_OPTION
        is TextItem -> TYPE_TEXT
        is DividerItem -> TYPE_DIVIDER
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ViewHolder = when (viewType) {
        TYPE_MISSING_OPTION -> MissingOptionHolder(
            LayoutInflater.from(parent.context)
                .inflate(android.R.layout.simple_list_item_1, parent, false) as TextView
        )
        TYPE_RADIO_BUTTON -> RadioButtonHolder(
            LayoutInflater.from(parent.context)
                .inflate(android.R.layout.simple_list_item_single_choice, parent, false) as CheckedTextView
        )
        TYPE_PASSWORD -> PasswordHolder(
            ConfigScreenPasswordBinding.inflate(
                LayoutInflater.from(parent.context), parent, false
            )
        ).apply {
            binding.editText.addOnTextChangedListener {
                val listener = this.textChangeListener

                if (listener != null) {
                    listener()
                }
            }
        }
        TYPE_TEXT -> TextHolder(
            LayoutInflater.from(parent.context)
                .inflate(android.R.layout.simple_list_item_1, parent, false) as TextView
        )
        TYPE_DIVIDER -> DividerHolder(
            LayoutInflater.from(parent.context)
                .inflate(R.layout.divider, parent, false)
        )
        else -> throw IllegalArgumentException()
    }

    override fun onBindViewHolder(holder: ViewHolder, position: Int) {
        val item = options!![position]

        when (item) {
            is RadioButtonItem -> {
                holder as RadioButtonHolder

                holder.view.text = item.label
                holder.view.isChecked = item.isChecked
                holder.view.setOnClickListener {
                    listener?.setValue(item.param, item.value)
                }
            }
            is PasswordItem -> {
                holder as PasswordHolder

                holder.textChangeListener = null

                if (holder.binding.editText.text.toString() != item.value) {
                    holder.binding.editText.setText(item.value)
                }

                holder.binding.textInputLayout.error = if (item.isInvalid) holder.binding.root.context.getString(R.string.config_error_password) else null
                holder.binding.textInputLayout.hint = item.hint

                holder.textChangeListener = {
                    listener?.setValue(
                        key = item.param,
                        value = holder.binding.editText.text.toString()
                    )
                }
            }
            is MissingOptionsItem -> {
                holder as MissingOptionHolder

                holder.view.setText(item.textResource)
            }
            is TextItem -> {
                holder as TextHolder

                holder.view.setText(item.text)
            }
            is DividerItem -> {/* nothing to do */}
            else -> throw IllegalArgumentException()
        }
    }
}

sealed class ViewHolder(view: View): RecyclerView.ViewHolder(view)
class MissingOptionHolder(val view: TextView): ViewHolder(view)
class RadioButtonHolder(val view: CheckedTextView): ViewHolder(view)
class PasswordHolder(val binding: ConfigScreenPasswordBinding, var textChangeListener: (() -> Unit)? = null): ViewHolder(binding.root)
class TextHolder(val view: TextView): ViewHolder(view)
class DividerHolder(val view: View): ViewHolder(view)

interface Listener {
    fun setValue(key: String, value: String)
}