/**
 * vertretungsplan.io android client
 *
 * Copyright (C) 2019 - 2022 Jonas Lochmann
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see https://www.gnu.org/licenses/.
 */
package io.vertretungsplan.client.android.ui.config

import android.app.Activity
import android.content.Context
import android.os.Bundle
import android.view.inputmethod.InputMethodManager
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProviders
import androidx.recyclerview.widget.LinearLayoutManager
import com.google.android.material.snackbar.Snackbar
import io.vertretungsplan.client.android.BuildConfig
import io.vertretungsplan.client.android.R
import io.vertretungsplan.client.android.databinding.ActivityInstitutionConfigBinding
import io.vertretungsplan.client.android.registry.DefaultAppRegistry

class InstitutionConfigActivity : AppCompatActivity() {
    companion object {
        const val EXTRA_INSTITUTION_ID = "institutionId"
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val binding = ActivityInstitutionConfigBinding.inflate(layoutInflater)

        setContentView(binding.root)
        setSupportActionBar(binding.toolbar)

        val inputMethodManager = getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
        val registry = DefaultAppRegistry.with(this)

        val adapter = InstitutionConfigAdapter()

        val institutionId = intent.getStringExtra(EXTRA_INSTITUTION_ID)!!
        val model = ViewModelProviders.of(this).get(InstitutionConfigModel::class.java)
        model.init(
            institutionId = institutionId,
            registry = registry
        )

        model.optionsLive.observe(this, Observer { adapter.options = it })
        model.isBusy.observe(this, Observer {
            binding.content.flipper.displayedChild = if (it) 0 else 1

            if (it) {
                binding.content.confirmBtn.hide()
                inputMethodManager.hideSoftInputFromWindow(binding.content.recycler.windowToken, 0)
            } else {
                binding.content.confirmBtn.show()
            }
        })
        model.isDone.observe(this, Observer {
            if (it) {
                setResult(Activity.RESULT_OK)
                finish()
            }
        })
        model.institution.observe(this, Observer {
            if (it == null) {
                finish()
            } else {
                supportActionBar!!.subtitle = it.title
            }
        })

        var lastSnackbar: Snackbar? = null
        model.snackbarUpdate.observe(this, Observer {
            when (it!!) {
                SnackbarUpdate.None -> {/* nothing to do */}
                SnackbarUpdate.HideLast -> lastSnackbar?.dismiss()
                SnackbarUpdate.ConfigNotComplete -> {
                    lastSnackbar = Snackbar.make(binding.content.recycler, R.string.config_snackbar_incomplete, Snackbar.LENGTH_LONG).apply { show() }
                    model.confirmSnackbarUpdate()
                }
                SnackbarUpdate.InvalidPassword -> {
                    lastSnackbar = Snackbar.make(binding.content.recycler, R.string.config_snackbar_invalid_password, Snackbar.LENGTH_LONG).apply { show() }
                    model.confirmSnackbarUpdate()
                }
                SnackbarUpdate.ValidationFailed -> {
                    lastSnackbar = Snackbar.make(binding.content.recycler, R.string.config_snackbar_validation_failed, Snackbar.LENGTH_LONG).apply { show() }
                    model.confirmSnackbarUpdate()
                }
                SnackbarUpdate.LoadingFailed -> {
                    lastSnackbar = Snackbar.make(binding.content.recycler, R.string.config_snackbar_loading_failed, Snackbar.LENGTH_INDEFINITE).apply {
                        addCallback(object: Snackbar.Callback() {
                            override fun onDismissed(transientBottomBar: Snackbar?, event: Int) {
                                super.onDismissed(transientBottomBar, event)

                                if (event == DISMISS_EVENT_SWIPE) {
                                    finish()
                                }
                            }
                        })

                        show()
                    }
                    // no model.confirmSnackbarUpdate()
                }
            }.let {/* require handling all paths */}
        })

        binding.content.recycler.adapter = adapter
        binding.content.recycler.layoutManager = LinearLayoutManager(this)

        adapter.listener = object: Listener {
            override fun setValue(key: String, value: String) {
                model.updateParam(key, value)
            }
        }

        binding.content.confirmBtn.setOnClickListener { model.sync() }

        if (BuildConfig.DEBUG) {
            binding.content.confirmBtn.setOnLongClickListener { model.saveConfigWithoutValidation(); true }
        }
    }
}
