pub mod fellowship;

use async_stream::stream;
use axum::Router;
use axum::extract::ws;
use axum::routing::get;
use futures::FutureExt as _;
use futures::SinkExt as _;
use futures::Stream;
use futures::TryStreamExt as _;
use std::fmt;
use std::fmt::Debug;
use std::fmt::Display;
use std::time::Duration;
use tokio::time::sleep;
use tracing::warn;
use ts_rs::TS;

use serde::{Deserialize, Serialize};

use crate::character::Character;
use crate::server::backend::AppState;
use crate::server::connection_handler::SafeSocket;
use crate::version::Version;
use fellowship::socket;

pub fn get_router() -> Router<AppState> {
    Router::new().route("/fellowship", get(socket))
}

#[derive(Serialize, Deserialize, TS)]
#[ts(export)]
pub enum Event {
    UpdateMessage,
    UpdateFellowship,
    Version(Version),
    Character(Box<Character>),
}

// TODO cleanup with macro
impl Display for Event {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        let text = match self {
            Self::UpdateMessage => "UpdateMessage",
            Self::UpdateFellowship => "UpdateFellowship",
            Self::Version(_) => "Version",
            Self::Character(_) => "Character",
        };
        write!(f, "{text}")
    }
}

fn non_blocking_stream(stream: SafeSocket) -> impl Stream<Item = Option<ws::Message>> {
    stream! {
        loop {
            {
                let result = stream.lock().await.try_next().now_or_never();
                if let Some(x) = result {
                    yield x.expect("Failed to yield stream");
                }
            }
            sleep(Duration::from_millis(20)).await;
        }
    }
}

async fn close_connection<T: Debug>(stream: SafeSocket, error: T) {
    let error = format!("{error:?}");
    if let Err(err) = stream.lock().await.send(error.into()).await {
        warn!("Failed to send error message {err:?}");
    }
    if let Err(err) = stream.lock().await.close().await {
        warn!("Failed to close connection {err:?}");
    }
}
