use std::sync::Arc;

use anyhow::Context as _;
use anyhow::Result;
use anyhow::anyhow;
use axum::extract::State;
use axum::extract::WebSocketUpgrade;
use axum::extract::ws::Message;
use axum::extract::ws::WebSocket;
use axum::response::IntoResponse;
use futures::{StreamExt as _, pin_mut};
use sea_orm::DatabaseConnection;
use serde::Deserialize;
use tokio::sync::Mutex;
use tracing::{debug, error, info, warn};

use super::Event;
use crate::fellowship::Fellowship;
use crate::server::backend::AppState;
use crate::server::connection_handler::{CONNECTIONS, Connection, ConnectionHandler, SafeSocket};
use crate::server::users::{User, UserInClear};
use crate::version::Version;

use super::{close_connection, non_blocking_stream};

#[derive(Clone)]
struct ConnectionData {
    loremaster: String,
    player: Option<String>,
}

impl ConnectionData {
    pub fn get_key(&self) -> &str {
        &self.loremaster
    }
    pub fn get_name(&self) -> String {
        self.clone().player.unwrap_or(self.clone().loremaster)
    }
}

#[derive(Deserialize, ts_rs::TS)]
#[ts(export)]
pub enum Auth {
    Loremaster(UserInClear),
    Player(Connection),
}

pub async fn socket(state: State<AppState>, ws: WebSocketUpgrade) -> impl IntoResponse {
    ws.on_upgrade(async move |stream| {
        if let Err(x) = handle_socket(&state.db, stream).await {
            error!(?x);
        }
    })
}

#[expect(clippy::cognitive_complexity)]
async fn handle_socket(db: &DatabaseConnection, stream: WebSocket) -> Result<()> {
    // Authenticate the request
    let stream = Arc::new(Mutex::new(stream));
    let conn = match process_handshake(db, stream.clone()).await {
        Err(x) => {
            warn!("{:}", x);
            close_connection(stream.clone(), x).await;
            return Ok(());
        }
        Ok(x) => x,
    };

    info!("Connection is setup");
    let iter = non_blocking_stream(stream.clone());
    pin_mut!(iter);
    while let Some(message) = iter.next().await {
        let message = message.context("Failed to get a message")?;
        debug!("Loremaster - {:?}", message);

        match message {
            // Handle connection closed
            Message::Close(_) => {
                {
                    let mut handler = CONNECTIONS.lock().await;
                    match handler.get_mut(conn.get_key()) {
                        None => panic!("No frontend available when disconnecting"),
                        Some(x) => {
                            let player = conn.get_name();
                            x.remove(&player).await;
                        }
                    }
                }
                ConnectionHandler::cleanup().await;
            }
            Message::Text(x) => {
                if let Err(x) = handle_message(db, &conn, x.to_string()).await {
                    error!("Failed to process socket message {x}");
                }
            }

            // Normal handling
            x => {
                eprintln!("{x:?}");
            }
        }
    }
    Ok(())
}

async fn handle_message(
    db: &DatabaseConnection,
    conn: &ConnectionData,
    message: String,
) -> Result<()> {
    let message: Event = serde_json::from_str(&message)?;

    info!(
        "Request {} between {:?} and {:?}",
        message, conn.loremaster, conn.player
    );
    match message {
        Event::UpdateFellowship => {
            ConnectionHandler::update_everyone(conn.loremaster.clone(), Event::UpdateFellowship)
                .await;
        }
        Event::Character(mut x) => {
            let loremaster = User::get_user(db, &conn.loremaster).await?;
            x.save_external(db, &loremaster).await?;
        }
        _ => Err(anyhow!("Message not implemented"))?,
    }
    Ok(())
}

async fn check_auth(db: &DatabaseConnection, auth: &Auth) -> Result<ConnectionData> {
    match auth {
        Auth::Loremaster(user) => {
            User::check_auth(db, user.clone()).await?;

            Ok(ConnectionData {
                loremaster: user.username.clone(),
                player: None,
            })
        }
        Auth::Player(conn) => {
            info!(
                "User {} connecting to {} from {}",
                conn.player, conn.fellowship, conn.loremaster
            );

            // Check if fellowship exists
            Fellowship::get_fellowship_and_loremaster_from_names(
                db,
                &conn.fellowship,
                &conn.loremaster,
            )
            .await?;

            Ok(ConnectionData {
                loremaster: conn.loremaster.clone(),
                player: Some(conn.player.clone()),
            })
        }
    }
}

async fn process_handshake(db: &DatabaseConnection, socket: SafeSocket) -> Result<ConnectionData> {
    // Get the version
    let message = socket
        .lock()
        .await
        .next()
        .await
        .ok_or(anyhow!("No next stream"))??;

    // Check the major version
    let version: Version = match message {
        Message::Text(x) => serde_json::from_str(&x)?,
        _ => return Err(anyhow!("Type not implemented")),
    };
    let local = Version::new();
    if version.major != local.major {
        Err(anyhow!("Wrong version"))?;
    }

    let message = socket
        .lock()
        .await
        .next()
        .await
        .ok_or(anyhow!("No next stream"))??;
    let auth: Auth = match message {
        Message::Text(x) => serde_json::from_str(&x)?,
        _ => return Err(anyhow!("Type not implemented")),
    };

    let conn = check_auth(db, &auth).await?;

    // Manage connection handler
    let mut handler = CONNECTIONS.lock().await;
    let key: String = conn.get_key().into();
    let name: String = conn.get_name();
    match handler.get_mut(&key) {
        None => {
            info!("Creating a new connection {key}");
            handler.insert(key, ConnectionHandler::new(name, socket));
        }
        Some(x) => {
            x.add(name, socket);
            // Check the minor version
            if version.minor != local.minor {
                x.send_to_single_player(
                    &conn.get_name(),
                    "Minor versions mismatch, risk of errors",
                )
                .await?;
            }
        }
    }

    Ok(conn)
}
