use anyhow::anyhow;
use axum::extract::{Path, Query, State};
use axum::routing::{get, post};
use axum::{Json, Router};
use serde::Deserialize;

use super::LoremasterOrPlayer;
use super::socket::Event;
use crate::error::Result;
use crate::message::Message;
use crate::server::backend::AppState;
use crate::server::connection_handler::ConnectionHandler;

pub fn get_router() -> Router<AppState> {
    Router::new()
        .route("/{fellowship_id}", post(send_message))
        .route("/{fellowship_id}", get(get_message))
}

async fn send_message(
    state: State<AppState>,
    user: LoremasterOrPlayer,
    Path(fellowship_id): Path<i32>,
    Json(message): Json<Message>,
) -> Result<()> {
    let loremaster = user.get_loremaster();

    message
        .save_to_db(&state.db, &loremaster, fellowship_id)
        .await?;

    // Update player
    let player_to_update = if message.get_to_loremaster() {
        loremaster.get_username()
    } else {
        message.get_player()
    };
    ConnectionHandler::update_single(
        loremaster.username.clone(),
        player_to_update,
        Event::UpdateMessage,
    )
    .await;
    Ok(())
}

#[derive(Deserialize)]
struct OptionalString {
    string: Option<String>,
}

async fn get_message(
    state: State<AppState>,
    user: LoremasterOrPlayer,
    Path(fellowship_id): Path<i32>,
    Query(OptionalString { string: player }): Query<OptionalString>,
) -> Result<Json<Vec<Message>>> {
    if let LoremasterOrPlayer::Player(_) = user
        && player.is_none()
    {
        Err(anyhow!("Only the loremaster can access all the messages"))?;
    }
    let loremaster = user.get_loremaster();
    let messages = Message::get(&state.db, &loremaster, fellowship_id, player).await?;

    Ok(Json(messages))
}
