use axum::extract::{Path, State};
use axum::http::StatusCode;
use axum::routing::{get, post};
use axum::{Json, Router};

use crate::error::Result;
use crate::server::backend::AppState;
use crate::server::cache_response::CacheResponse;
use crate::server::users::User;
use crate::version::Version;

#[cfg(all(feature = "tauri", feature = "tauri-updater"))]
use anyhow::anyhow;

#[cfg(feature = "tauri")]
use tokio::sync::Mutex;

#[cfg(feature = "tauri")]
use std::sync::LazyLock;

#[cfg(feature = "tauri")]
pub static HANDLER: LazyLock<Mutex<Option<tauri::AppHandle>>> = LazyLock::new(|| Mutex::new(None));

pub fn get_router() -> Router<AppState> {
    Router::new()
        .route("/health", get(health))
        .route("/requires-authentication", get(requires_authentication))
        .route("/check-auth", get(check_auth))
        .route("/create-user/{user}", post(create_user))
        .route("/version", get(get_version))
        .route("/can-update", get(can_update))
        .route("/update", post(update))
}

async fn health() -> StatusCode {
    StatusCode::OK
}

async fn requires_authentication() -> CacheResponse<&'static str> {
    CacheResponse::new(if cfg!(feature = "cloud") {
        "true"
    } else {
        "false"
    })
}

async fn check_auth(_user: User) -> &'static str {
    "Ok"
}

async fn create_user(state: State<AppState>, Path(user): Path<String>, body: String) -> Result<()> {
    User::create_user(&state.db, &user, &body).await?;
    Ok(())
}

async fn get_version() -> Json<Version> {
    Json(Version::new())
}

pub async fn can_update() -> Result<CacheResponse<bool>> {
    #[cfg(all(feature = "tauri", feature = "tauri-updater"))]
    {
        use tauri_plugin_updater::UpdaterExt;

        let app = HANDLER
            .lock()
            .await
            .clone()
            .ok_or(anyhow!("Cannot find tauri app"))?;

        return Ok(CacheResponse::new(app.updater()?.check().await?.is_some()));
    }
    #[cfg(any(not(feature = "tauri"), target_os = "android"))]
    Ok(CacheResponse::new(false))
}

pub async fn update() -> Result<&'static str> {
    #[cfg(all(feature = "tauri", feature = "tauri-updater"))]
    {
        use tauri_plugin_updater::UpdaterExt;
        use tracing::info;

        let app = HANDLER
            .lock()
            .await
            .clone()
            .ok_or(anyhow!("Cannot find tauri app"))?;

        if let Some(update) = app.updater()?.check().await? {
            let mut downloaded = 0;

            update
                .download_and_install(
                    |chunk_length, content_length| {
                        downloaded += chunk_length;

                        info!("downloaded {downloaded} from {content_length:?}");
                    },
                    || {
                        info!("download finished");
                    },
                )
                .await?;

            info!("update installed");

            app.restart();
        }
    }

    Ok("Done")
}
