use std::collections::HashMap;

use anyhow::anyhow;
use axum::extract::{Path, State};
use axum::routing::{delete, get, post};
use axum::{Json, Router};
use enums::character_type::CharacterType;
use sea_orm::{ColumnTrait as _, EntityTrait as _, QueryFilter as _};
use serde::{Deserialize, Serialize};

use crate::adversary::Adversary;
use crate::character::Character;
use crate::error::Result;
use crate::fellowship::patron::{Patron, PatronEnum};
use crate::fellowship::request::FellowshipRequest;
use crate::server::backend::AppState;
use crate::server::connection_handler::ConnectionHandler;
use crate::server::name_or_id::NameOrId;
use crate::{
    entity::fellowship as fellowship_entity,
    fellowship::{
        Fellowship,
        undertaking::{Undertaking, UndertakingEnum},
    },
    server::cache_response::CacheResponse,
    server::users::User,
};

use super::LoremasterOrPlayer;
use super::socket::Event;

pub fn get_router() -> Router<AppState> {
    axum::Router::new()
        .route("/undertaking/{undertaking}", get(get_undertaking))
        .route("/", get(get_fellowships))
        .route("/patron/official", get(get_official_patrons))
        .route("/patron/personal", get(get_personal_patrons))
        .route("/patron/{id}", get(get_patron))
        .route("/patron", post(set_patron))
        .route("/patron/{id}", delete(delete_patron))
        .route("/{id}", get(get_fellowship))
        .route("/", post(set_fellowship))
        .route("/{id}", delete(delete_fellowship))
        .route("/character", post(add_character))
        .route("/{fellowship_id}/adversary", post(add_adversary))
        .route(
            "/{fellowship_id}/adversary/{adversary_id}",
            delete(delete_adversary),
        )
        .route(
            "/{fellowship_id}/character/{character_id}",
            delete(delete_character),
        )
        .route("/player", get(get_player_fellowship))
        .route(
            "/{fellowship_id}/increase-eye/{increase}",
            post(increase_eye),
        )
        .route("/{fellowship_id}/reset-eye", post(reset_eye))
        .route(
            "/{fellowship_id}/reset-points",
            post(reset_fellowship_points),
        )
        .route(
            "/{fellowship_id}/increase-points/{increase}",
            post(increase_points),
        )
}

async fn get_undertaking(Path(undertaking): Path<UndertakingEnum>) -> CacheResponse<Undertaking> {
    let undertaking = Undertaking::new(undertaking);
    CacheResponse::new(undertaking)
}

async fn get_fellowships(state: State<AppState>, user: User) -> Result<Json<Vec<Fellowship>>> {
    let fellowship_ids = fellowship_entity::Entity::find()
        .filter(fellowship_entity::Column::OwnerId.eq(user.id))
        .all(&state.db)
        .await?;

    let mut fellowships = Vec::with_capacity(fellowship_ids.len());
    for id in fellowship_ids {
        fellowships.push(Fellowship::from_db_model(&state.db, &user, &id).await?);
    }

    Ok(Json(fellowships))
}

async fn get_official_patrons() -> CacheResponse<HashMap<PatronEnum, Patron>> {
    CacheResponse::new(Patron::get_list_from_enums())
}

async fn get_personal_patrons(state: State<AppState>, user: User) -> Result<Json<Vec<Patron>>> {
    let out = Patron::get_list_from_db(&state.db, &user).await?;
    Ok(Json(out))
}

async fn get_patron(
    state: State<AppState>,
    user: User,
    Path(id): Path<NameOrId>,
) -> Result<Json<Patron>> {
    let patron = Patron::get(&state.db, &user, id).await?;
    Ok(Json(patron))
}

async fn set_patron(
    state: State<AppState>,
    user: User,
    Json(patron): Json<Patron>,
) -> Result<Json<i32>> {
    let id = patron.save_to_db(&state.db, &user).await?;
    Ok(Json(id))
}

async fn get_fellowship(
    state: State<AppState>,
    user: User,
    Path(id): Path<i32>,
) -> Result<Json<Fellowship>> {
    Ok(Json(Fellowship::get(&state.db, &user, id).await?))
}

async fn set_fellowship(
    state: State<AppState>,
    user: User,
    Json(fellowship): Json<Fellowship>,
) -> Result<Json<i32>> {
    let id = fellowship.save_to_db(&state.db, &user).await?;

    // Update players
    ConnectionHandler::update_everyone(user.get_username().clone(), Event::UpdateFellowship).await;

    Ok(Json(id))
}

async fn delete_fellowship(state: State<AppState>, user: User, Path(id): Path<i32>) -> Result<()> {
    Fellowship::delete_from_db(&state.db, &user, id).await?;
    Ok(())
}

async fn delete_patron(state: State<AppState>, user: User, Path(id): Path<i32>) -> Result<()> {
    Patron::delete_from_db(&state.db, &user, id).await?;
    Ok(())
}

#[derive(Serialize, Deserialize, PartialEq, ts_rs::TS)]
#[ts(export)]
enum RequestOrigin {
    Local,
    Extern,
}

#[derive(Serialize, Deserialize, ts_rs::TS)]
#[ts(export)]
struct AddToFellowship {
    fellowship: String,
    loremaster: String,
    user: String,
    character: Character,
    origin: RequestOrigin,
}

async fn add_character(state: State<AppState>, Json(data): Json<AddToFellowship>) -> Result<()> {
    let (fellowship, loremaster) = Fellowship::get_fellowship_and_loremaster_from_names(
        &state.db,
        &data.fellowship,
        &data.loremaster,
    )
    .await?;

    // Add to local db if needed
    let (character, owner) = if data.origin == RequestOrigin::Extern {
        let mut character = data.character;
        character.set_player_name(data.user);
        character.save_external(&state.db, &loremaster).await?;
        (character, loremaster.clone())
    } else {
        (data.character, User::get_user(&state.db, &data.user).await?)
    };

    let id = character
        .get_id()
        .ok_or(anyhow!("Character does not seem to exist in the DB"))?;
    fellowship
        .add_character(&state.db, &loremaster, &owner, id)
        .await?;

    // Update players
    ConnectionHandler::update_everyone(loremaster.get_username().clone(), Event::UpdateFellowship)
        .await;

    Ok(())
}

async fn add_adversary(
    state: State<AppState>,
    user: User,
    Path(fellowship_id): Path<i32>,
    Json(adversary): Json<Adversary>,
) -> Result<()> {
    let fellowship = Fellowship::get(&state.db, &user, fellowship_id).await?;
    let id = if let Some(id) = adversary.get_id() {
        id
    } else {
        adversary.save_to_db(&state.db, &user).await?
    };
    fellowship.add_adversary(&state.db, &user, id).await?;

    // Update players
    ConnectionHandler::update_everyone(user.username.clone(), Event::UpdateFellowship).await;

    Ok(())
}

async fn delete_adversary(
    state: State<AppState>,
    user: User,
    Path((fellowship_id, adversary_id)): Path<(i32, i32)>,
) -> Result<()> {
    let fellowship = Fellowship::get(&state.db, &user, fellowship_id).await?;
    fellowship
        .remove_adversary(&state.db, &user, adversary_id)
        .await?;

    // Update players
    ConnectionHandler::update_everyone(user.username.clone(), Event::UpdateFellowship).await;

    Ok(())
}

async fn delete_character(
    state: State<AppState>,
    user: User,
    Path((fellowship_id, character_id)): Path<(i32, i32)>,
) -> Result<()> {
    let fellowship = Fellowship::get(&state.db, &user, fellowship_id).await?;
    fellowship
        .remove_character(&state.db, &user, character_id)
        .await?;

    let character = Character::get(&state.db, &user, character_id).await?;
    if character.get_character_type() == CharacterType::External {
        Character::delete_from_db(&state.db, character_id, &user).await?;
    }

    // Update players
    ConnectionHandler::update_everyone(user.username.clone(), Event::UpdateFellowship).await;

    Ok(())
}

async fn get_player_fellowship(request: FellowshipRequest) -> Json<Fellowship> {
    Json(request.fellowship)
}

async fn increase_eye(
    state: State<AppState>,
    user: User,
    Path((fellowship_id, increase)): Path<(i32, i32)>,
) -> Result<Json<u32>> {
    let mut fellowship = Fellowship::get(&state.db, &user, fellowship_id).await?;
    fellowship.increase_eye_rating(increase)?;
    fellowship.save_to_db(&state.db, &user).await?;

    Ok(Json(fellowship.get_eye()))
}

async fn reset_eye(
    state: State<AppState>,
    user: User,
    Path(fellowship_id): Path<i32>,
) -> Result<Json<u32>> {
    let mut fellowship = Fellowship::get(&state.db, &user, fellowship_id).await?;
    fellowship.reset_eye_rating();
    fellowship.save_to_db(&state.db, &user).await?;
    Ok(Json(fellowship.get_eye()))
}

async fn reset_fellowship_points(
    state: State<AppState>,
    user: User,
    Path(fellowship_id): Path<i32>,
) -> Result<Json<u32>> {
    let mut fellowship = Fellowship::get(&state.db, &user, fellowship_id).await?;
    fellowship.reset_points();
    fellowship.save_to_db(&state.db, &user).await?;

    // Update players
    ConnectionHandler::update_everyone(user.username.clone(), Event::UpdateFellowship).await;
    Ok(Json(fellowship.get_points()))
}

async fn increase_points(
    state: State<AppState>,
    user: LoremasterOrPlayer,
    Path((fellowship_id, increase)): Path<(i32, i32)>,
) -> Result<Json<u32>> {
    if let LoremasterOrPlayer::Player(_x) = &user
        && increase > 0
    {
        Err(anyhow!(
            "Only the loremaster can increase the fellowship points"
        ))?;
    }
    let loremaster = user.clone().get_loremaster();
    let mut fellowship = user.get_fellowship(&state.db, fellowship_id).await?;

    fellowship.increase_points(increase)?;
    fellowship.save_to_db(&state.db, &loremaster).await?;

    // Update players
    ConnectionHandler::update_everyone(loremaster.username.clone(), Event::UpdateFellowship).await;

    Ok(Json(fellowship.get_eye()))
}
