use std::collections::HashMap;

use anyhow::anyhow;
use axum::Router;
use axum::extract::{Path, Query};
use axum::routing::get;
use enums::armor::ArmorEnum;
use enums::calling::CallingEnum;
use enums::combat_proficiency::CombatProficiency;
use enums::combat_skill::CombatSkillEnum;
use enums::culture::CultureEnum;
use enums::region::Region;
use enums::reward::RewardEnum;
use enums::shield::ShieldEnum;
use enums::skill::SkillEnum;
use enums::virtue::VirtueEnum;
use enums::weapon::WeaponEnum;
use serde::Deserialize;
use strum::IntoEnumIterator as _;

use crate::character::advantage::AdvantageTrait as _;
use crate::character::item::armor::Armor;
use crate::character::item::equipment::Equipment;
use crate::character::item::shield::Shield;
use crate::character::item::treasure::StandardOfLiving;
use crate::character::item::weapon::Weapon;
use crate::character::item::{Item, ItemType};
use crate::character::reward::Reward;
use crate::character::secondary_attributes::{SecondaryAttributeAdvantage, SecondaryAttributeEnum};
use crate::character::virtue::Virtue;
use crate::error::Result;
use crate::fellowship::undertaking::UndertakingEnum;
use crate::roll::RollType;
use crate::server::backend::AppState;
use crate::server::cache_response::CacheResponse;
use crate::tools::name_generator::NameGenerators;

pub fn get_router() -> Router<AppState> {
    axum::Router::new()
        .route("/skills", get(get_skills))
        .route("/combat-skills", get(get_combat_skills))
        .route("/combat-proficiencies", get(get_combat_proficiencies))
        .route("/callings", get(get_callings))
        .route("/cultures", get(get_cultures))
        .route("/standard-living", get(get_standard_living))
        .route("/secondary-attributes", get(get_secondary_attributes))
        .route("/undertaking", get(get_undertakings))
        .route("/empty/item/{eq_type}", get(get_empty_item))
        .route("/empty/advantage/{attr}", get(get_advantage))
        .route("/name-generator/races", get(get_name_generator_races))
        .route("/roll-type", get(get_roll_type))
        .route("/virtue", get(get_virtues))
        .route("/reward", get(get_rewards))
        .route("/virtue/{name}", get(get_virtue))
        .route("/reward/{name}", get(get_reward))
        .route("/{eq_type}/{item_name}", get(get_item))
        .route("/item/{eq_type}", get(get_list_items))
        .route("/eye-threshold/{eye}", get(get_eye_threshold))
}

pub async fn get_skills() -> CacheResponse<Vec<SkillEnum>> {
    CacheResponse::new(SkillEnum::iter().collect())
}

pub async fn get_combat_skills() -> CacheResponse<Vec<CombatSkillEnum>> {
    CacheResponse::new(CombatSkillEnum::iter().collect())
}

pub async fn get_combat_proficiencies() -> CacheResponse<Vec<CombatProficiency>> {
    CacheResponse::new(CombatProficiency::iter().collect())
}

pub async fn get_callings() -> CacheResponse<Vec<CallingEnum>> {
    CacheResponse::new(CallingEnum::iter().collect())
}

pub async fn get_cultures() -> CacheResponse<Vec<CultureEnum>> {
    CacheResponse::new(CultureEnum::iter().collect())
}

pub async fn get_standard_living() -> CacheResponse<Vec<StandardOfLiving>> {
    CacheResponse::new(StandardOfLiving::iter().collect())
}

pub async fn get_secondary_attributes() -> CacheResponse<Vec<SecondaryAttributeEnum>> {
    CacheResponse::new(SecondaryAttributeEnum::iter().collect())
}

pub async fn get_undertakings() -> CacheResponse<Vec<UndertakingEnum>> {
    CacheResponse::new(UndertakingEnum::iter().collect())
}

pub async fn get_empty_item(Path(eq_type): Path<ItemType>) -> CacheResponse<Item> {
    CacheResponse::new(match eq_type {
        ItemType::Armor => Item::Armor(Armor::new(String::new(), 0, 0)),
        ItemType::Equipment => Item::Equipment(Equipment::new(String::new(), 0)),
        ItemType::Shield => Item::Shield(Shield::new(String::new(), 0, 0)),
        ItemType::Weapon => Item::Weapon(Weapon::new(
            String::new(),
            0,
            0,
            None,
            0,
            CombatProficiency::Brawling,
        )),
    })
}

pub async fn get_advantage(
    Path(attr): Path<SecondaryAttributeEnum>,
) -> CacheResponse<SecondaryAttributeAdvantage> {
    let answer = match attr {
        SecondaryAttributeEnum::Valor => SecondaryAttributeAdvantage::Reward(Reward::default()),
        SecondaryAttributeEnum::Wisdom => SecondaryAttributeAdvantage::Virtue(Virtue::default()),
    };
    CacheResponse::new(answer)
}

pub async fn get_name_generator_races() -> CacheResponse<Vec<NameGenerators>> {
    CacheResponse::new(NameGenerators::iter().collect())
}

pub async fn get_roll_type() -> CacheResponse<Vec<RollType>> {
    CacheResponse::new(RollType::iter().collect())
}

#[derive(Deserialize)]
pub struct AdvantageOptions {
    culture: Option<CultureEnum>,
    creation: Option<bool>,
}

pub async fn get_virtues(
    Query(AdvantageOptions { culture, creation }): Query<AdvantageOptions>,
) -> CacheResponse<HashMap<VirtueEnum, Virtue>> {
    let mut out = HashMap::new();
    let creation = creation.unwrap_or(false);

    for virtue in VirtueEnum::iter() {
        let vir = Virtue::from_enum(virtue);
        let restricted_to = vir.get_culture_restrictions();
        // at creation? only creation : all
        let ok_creation = creation == vir.is_available_at_creation() || !creation;
        // culture? culture + none : all
        let ok_culture = culture.is_none()
            || restricted_to.is_none()
            || restricted_to.unwrap().contains(&culture.clone().unwrap());

        if vir.is_available() && ok_creation && ok_culture {
            out.insert(virtue, vir);
        }
    }
    CacheResponse::new(out)
}

pub async fn get_rewards(
    Query(AdvantageOptions { creation, .. }): Query<AdvantageOptions>,
) -> CacheResponse<HashMap<RewardEnum, Reward>> {
    let mut out = HashMap::new();
    for reward in RewardEnum::iter() {
        let rew = Reward::from_enum(reward);

        if creation.is_none()
            || !creation.unwrap()
            || (rew.is_available_at_creation() && creation.unwrap())
        {
            out.insert(reward, rew);
        }
    }
    CacheResponse::new(out)
}

pub async fn get_reward(Path(name): Path<RewardEnum>) -> CacheResponse<Reward> {
    CacheResponse::new(Reward::from_enum(name))
}

pub async fn get_virtue(Path(name): Path<VirtueEnum>) -> CacheResponse<Virtue> {
    CacheResponse::new(Virtue::from_enum(name))
}

pub async fn get_item(
    Path((eq_type, item_name)): Path<(ItemType, String)>,
) -> Result<CacheResponse<Item>> {
    Ok(CacheResponse::new(match eq_type {
        ItemType::Armor => Item::Armor(Armor::from(item_name.as_ref())),
        ItemType::Shield => Item::Shield(Shield::from(item_name.as_ref())),
        ItemType::Weapon => Item::Weapon(Weapon::from(item_name.as_ref())),
        ItemType::Equipment => Err(anyhow!("No standard equipment available"))?,
    }))
}

pub async fn get_list_items(Path(eq_type): Path<ItemType>) -> Result<CacheResponse<Vec<String>>> {
    Ok(CacheResponse::new(match eq_type {
        ItemType::Armor => ArmorEnum::iter()
            .map(|x| format!("{x:?}"))
            .collect::<Vec<String>>(),
        ItemType::Equipment => Err(anyhow!("No standard equipment available"))?,
        ItemType::Shield => ShieldEnum::iter()
            .map(|x| format!("{x:?}"))
            .collect::<Vec<String>>(),
        ItemType::Weapon => WeaponEnum::iter()
            .map(|x| format!("{x:?}"))
            .collect::<Vec<String>>(),
    }))
}

pub async fn get_eye_threshold(Path(eye): Path<u32>) -> CacheResponse<Option<Region>> {
    CacheResponse::new(Region::hunt_threshold(eye))
}
