use anyhow::{Result, bail};
use axum::Router;
use migration::MigratorTrait as _;
use sea_orm::{ConnectOptions, Database, DatabaseConnection};
use std::time::Duration;
use tokio::{task::spawn_blocking, time::sleep};
use tower_http::{cors::CorsLayer, trace::TraceLayer};

use crate::config::{CONFIG, Config};

use super::routes::{
    adversary, character, enums, equipment, fellowship, general, message as message_route,
    new_character, socket, tools,
};

#[cfg(not(feature = "cloud"))]
use super::users::User;

#[derive(Clone)]
pub struct AppState {
    pub db: DatabaseConnection,
}

async fn get_configured_config() -> Result<Config> {
    const LIMIT: u32 = 100;
    let mut step: u32 = 0;
    while step < LIMIT {
        step += 1;
        let config = CONFIG.lock().await;
        if config.is_configured() {
            return Ok(config.clone());
        }
        eprintln!("Not configured yet");
        sleep(Duration::new(0, 100)).await;
    }
    bail!("Failed to get a configuration")
}

pub async fn prepare_db(url: &str) -> Result<DatabaseConnection> {
    let mut options = ConnectOptions::new(url);
    options.sqlx_logging(false);
    let db = Database::connect(options).await?;

    migration::Migrator::up(&db, None).await?;

    #[cfg(not(feature = "cloud"))]
    User::create_empty_user(&db).await?;

    Ok(db)
}

pub async fn get_server() -> Result<Router> {
    tracing_subscriber::fmt()
        .with_max_level(tracing::Level::DEBUG)
        .init();
    eprintln!("Starting server");
    let config = spawn_blocking(get_configured_config).await?.await?;
    eprintln!("Configuration: {config:?}");

    // Cors
    let cors = CorsLayer::very_permissive();

    // DB
    let database_url = format!(
        "sqlite://{}?mode=rwc",
        config
            .get_directory()
            .join("data.db")
            .to_str()
            .expect("Should be able to convert to string")
    );
    eprintln!("Database: {database_url}");
    let state = AppState {
        db: prepare_db(&database_url).await?,
    };

    // TODO check rocket config
    let app = Router::<AppState>::new()
        .nest("/character", character::get_router())
        .nest("/adversary", adversary::get_router())
        .nest("/equipment", equipment::get_router())
        .nest("/new-character", new_character::get_router())
        .nest("/enums", enums::get_router())
        .nest("/general", general::get_router())
        .nest("/tools", tools::get_router())
        .nest("/fellowship", fellowship::get_router())
        .nest("/socket", socket::get_router())
        .nest("/message", message_route::get_router())
        .with_state(state)
        .layer(cors)
        .layer(TraceLayer::new_for_http());
    Ok(app)
}
