pub mod adversary;
pub(crate) mod assets;
pub(crate) mod character;
pub(crate) mod common;
pub mod config;
pub(crate) mod entity;
pub(crate) mod error;
pub(crate) mod fellowship;
pub(crate) mod message;
pub(crate) mod new_character;

pub(crate) mod roll;

pub mod server;
pub(crate) mod tools;
pub(crate) mod utils;
pub(crate) mod version;

// android
#[cfg(target_os = "android")]
use tauri;
#[cfg(feature = "tauri")]
use tauri::Wry;

#[cfg(feature = "tauri")]
fn add_updater(builder: tauri::Builder<Wry>) -> tauri::Builder<Wry> {
    #[cfg(feature = "tauri-updater")]
    {
        return builder.plugin(tauri_plugin_updater::Builder::new().build());
    }
    #[cfg(not(feature = "tauri-updater"))]
    {
        return builder;
    }
}

#[cfg(feature = "tauri")]
pub fn start_with_tauri() {
    eprintln!("Starting app");
    use config::CONFIG;
    use tauri::Manager;
    use tokio::runtime::Runtime;

    use crate::server::{backend, routes};

    // Start server
    let rt = Runtime::new().expect("Failed to create new runtime");
    rt.spawn(async {
        let server = backend::get_server().await;
        if let Ok(server) = server {
            let listener = tokio::net::TcpListener::bind(format!("localhost:8000"))
                .await
                .expect("Failed to start server");
            axum::serve(listener, server)
                .await
                .expect("An error happened when serving the server");
        } else {
            eprintln!("Error while starting app: {server:?}");
            std::process::exit(1);
        }
    });
    let mut config = rt.block_on(CONFIG.lock());

    // Start tauri
    let mut builder = tauri::Builder::default()
        .plugin(tauri_plugin_store::Builder::default().build())
        .plugin(tauri_plugin_http::init())
        .plugin(tauri_plugin_clipboard_manager::init());

    builder = add_updater(builder);

    builder
        .setup(move |app| {
            let config_dir = app.path().document_dir().expect("Failed to get config dir");

            config.configure(&config_dir);
            drop(config);

            let app_handle = app.handle().clone();
            tauri::async_runtime::spawn(async move {
                let mut handler = routes::general::HANDLER.lock().await;
                *handler = Some(app_handle);
            });

            Ok(())
        })
        .run(tauri::generate_context!())
        .expect("error while running tauri application");
}

#[cfg(target_os = "android")]
#[tauri::mobile_entry_point]
fn android_start() {
    start_with_tauri();
}
