use core::fmt;

use std::fmt::Formatter;
use std::fmt::Result as FmtResult;
use std::result::Result as StdResult;

use axum::http::StatusCode;
use axum::response::IntoResponse;
use axum::response::Response;
use tracing::error;

pub type Result<T = ()> = StdResult<T, OwnError>;

#[derive(Debug)]
pub struct OwnError(pub anyhow::Error);

impl<E> From<E> for OwnError
where
    E: Into<anyhow::Error>,
{
    fn from(error: E) -> Self {
        OwnError(error.into())
    }
}

impl IntoResponse for OwnError {
    fn into_response(self) -> Response {
        error!(?self, "Error: ");
        (StatusCode::INTERNAL_SERVER_ERROR, format!("{:?}", self.0)).into_response()
    }
}

impl fmt::Display for OwnError {
    fn fmt(&self, f: &mut Formatter<'_>) -> FmtResult {
        write!(f, "{self:?}")
    }
}
