use anyhow::{Result, bail};
use enums::reward::RewardEnum;
use sea_orm::{
    ActiveModelTrait as _,
    ActiveValue::{NotSet, Set},
    ColumnTrait as _, ConnectionTrait, EntityTrait as _, ModelTrait as _, QueryFilter as _,
};
use serde::{Deserialize, Serialize};
use ts_rs::TS;

use crate::entity::reward as entity;
use crate::utils::convert_option_to_set;
use crate::{assets::read_enum_from_assets, utils::to_enum};

use super::{
    advantage::AdvantageTrait,
    item::{
        Item, armor::Armor, equipment::Equipment, item_trait::ItemTrait, shield::Shield,
        weapon::Weapon,
    },
};

const DIRECTORY: &str = "rewards";

#[derive(Serialize, Deserialize, Clone, TS, Debug, Default)]
#[ts(export)]
pub struct Reward {
    id: Option<i32>,
    title: String,
    #[serde(default)]
    enum_value: Option<RewardEnum>,
    description: String,
    available_at_creation: bool,
    #[serde(default)]
    linked_item: Option<Item>,
    #[serde(default)]
    implemented: bool,
}

impl Reward {
    pub async fn set_linked_item_from_db<T: ItemTrait, C: ConnectionTrait>(
        &mut self,
        db: &C,
    ) -> Result<bool> {
        let item = T::from_db_reward(db, self.id.expect("Reward should have an id here")).await?;
        let out = if let Some(item) = item {
            self.linked_item = Some(item.get_item());
            true
        } else {
            false
        };

        Ok(out)
    }
    pub fn set_linked_item(&mut self, item: Item) {
        self.linked_item = Some(item);
    }

    pub fn get_linked_item(&self) -> Option<&Item> {
        self.linked_item.as_ref()
    }
}

// AdvantageTrait
impl AdvantageTrait for Reward {
    type Enum = RewardEnum;
    fn get_title(&self) -> &str {
        &self.title
    }
    fn set_id(&mut self, id: Option<i32>) {
        self.id = id;
    }

    fn get_id(&self) -> Option<i32> {
        self.id
    }

    fn from_enum(value: Self::Enum) -> Self {
        let mut out: Self = read_enum_from_assets(&value, DIRECTORY);
        out.enum_value = Some(value);
        out
    }

    fn get_enum(&self) -> Option<Self::Enum> {
        self.enum_value
    }
    fn find_enum(&mut self) -> bool {
        let title = to_enum(&self.title);
        let title = format!("\"{title}\"");
        if let Ok(out) = serde_json::from_str(&title) {
            self.enum_value = Some(out);
            true
        } else {
            false
        }
    }

    fn is_available_at_creation(&self) -> bool {
        self.available_at_creation
    }

    async fn from_db_model<C: ConnectionTrait>(db: &C, character_id: i32) -> Result<Vec<Self>> {
        let mut rewards: Vec<Self> = entity::Entity::find()
            .filter(entity::Column::CharacterId.eq(character_id))
            .all(db)
            .await?
            .iter()
            .map(|x| (*x).clone().into())
            .collect();

        for reward in &mut rewards {
            if reward.set_linked_item_from_db::<Weapon, _>(db).await? {
                continue;
            }
            if reward.set_linked_item_from_db::<Armor, _>(db).await? {
                continue;
            }
            if reward.set_linked_item_from_db::<Shield, _>(db).await? {
                continue;
            }
            reward.set_linked_item_from_db::<Equipment, _>(db).await?;
        }
        Ok(rewards)
    }

    async fn save<C: ConnectionTrait>(&self, db: &C, character_id: i32) -> Result<()> {
        let mut model: entity::ActiveModel = (*self).clone().into();
        model.character_id = Set(character_id);
        model.save(db).await?;
        Ok(())
    }

    async fn delete_by_character_id<C: ConnectionTrait>(db: &C, character_id: i32) -> Result<()> {
        let rewards = Self::from_db_model(db, character_id).await?;
        for reward in rewards {
            if let Some(item) = reward.linked_item {
                let reward_id = reward.id.expect("Reward should have an id here");
                match item {
                    Item::Armor(_) => Armor::unlink_reward(db, reward_id).await?,
                    Item::Equipment(_) => Equipment::unlink_reward(db, reward_id).await?,
                    Item::Weapon(_) => Weapon::unlink_reward(db, reward_id).await?,
                    Item::Shield(_) => Shield::unlink_reward(db, reward_id).await?,
                }
            }
        }

        entity::Entity::delete_many()
            .filter(entity::Column::CharacterId.eq(character_id))
            .exec(db)
            .await?;
        Ok(())
    }

    async fn delete_by_id<C: ConnectionTrait>(db: &C, character_id: i32, id: i32) -> Result<()> {
        let reward = entity::Entity::find_by_id(id).one(db).await?;

        if let Some(reward) = reward {
            if reward.character_id == character_id {
                Armor::unlink_reward(db, id).await?;
                Equipment::unlink_reward(db, id).await?;
                Weapon::unlink_reward(db, id).await?;
                Shield::unlink_reward(db, id).await?;

                reward.delete(db).await?;
                Ok(())
            } else {
                bail!("Cannot delete reward from someone else")
            }
        } else {
            bail!("Reward not found")
        }
    }
}

impl From<entity::Model> for Reward {
    fn from(value: entity::Model) -> Self {
        let enum_value = value.enum_value.as_ref().and_then(|name| {
            serde_json::from_str(name).expect("SQL and serde enums are not compatible")
        });

        let mut out = Self {
            id: Some(value.id),
            title: value.title,
            enum_value,
            description: value.description,
            available_at_creation: false,
            linked_item: None,
            implemented: false,
        };

        if enum_value.is_none() {
            out.find_enum();
        }
        if let Some(enum_value) = enum_value {
            out.implemented = Self::from_enum(enum_value).implemented;
        }
        out
    }
}

impl From<Reward> for entity::ActiveModel {
    fn from(value: Reward) -> Self {
        let id = convert_option_to_set(value.id);
        let enum_value = value
            .enum_value
            .as_ref()
            .map(|value| serde_json::to_string(value).expect("Failed to convert enum to string"));
        Self {
            id,
            title: Set(value.title),
            enum_value: Set(enum_value),
            description: Set(value.description),
            character_id: NotSet,
        }
    }
}

#[cfg(test)]
mod test {
    use strum::{EnumCount as _, IntoEnumIterator as _};

    use crate::{
        assets::get_number_files,
        character::{
            advantage::AdvantageTrait as _,
            reward::{DIRECTORY, Reward},
        },
    };

    use super::RewardEnum;

    #[test]
    fn check_reward_data() {
        let number_files = get_number_files(DIRECTORY);
        assert!(RewardEnum::COUNT == number_files);

        for reward in RewardEnum::iter() {
            eprintln!("Testing {reward:?}");
            Reward::from_enum(reward);
        }
    }

    #[test]
    fn find_enum() {
        let mut reward = Reward {
            title: "Fell".to_owned(),
            ..Default::default()
        };
        reward.find_enum();
        assert_eq!(reward.enum_value, Some(RewardEnum::Fell));
    }
}
