use std::collections::HashMap;

use anyhow::{Result, anyhow, bail};
use enums::virtue::VirtueEnum;
use sea_orm::{ConnectionTrait, DatabaseConnection};
use serde::{Deserialize, Serialize};
use ts_rs::TS;

use crate::entity::character as character_entity;
use armor::Armor;
use equipment::Equipment;
use item_trait::ItemTrait;
use shield::Shield;
use treasure::Treasure;
use weapon::Weapon;

use super::{characteristics::DerivedData, reward::Reward};

pub mod armor;
pub mod equipment;
pub mod item_trait;
pub mod shield;
pub mod treasure;
pub mod weapon;

#[derive(Serialize, Deserialize, Clone, Copy, Debug, TS)]
#[ts(export)]
pub enum ItemType {
    Armor,
    Weapon,
    Shield,
    Equipment,
}

#[derive(Serialize, Deserialize, Clone, Debug, TS)]
#[serde(untagged)]
#[ts(export)]
pub enum Item {
    Armor(Armor),
    Weapon(Weapon),
    Shield(Shield),
    // NOTE equipment needs to be last otherwise the other item are wrongly deserialized
    Equipment(Equipment),
}

impl Item {
    pub fn is_wearing(&self) -> bool {
        match self {
            Self::Armor(x) => x.is_wearing(),
            Self::Shield(x) => x.is_wearing(),
            Self::Weapon(x) => x.is_wearing(),
            Self::Equipment(x) => x.is_wearing(),
        }
    }
}

#[derive(Default, Serialize, Deserialize, Clone, TS, Debug)]
#[ts(export)]
#[ts(rename = "ItemsDerived")]
struct Derived {
    protection: u32,
    parry: u32,
    load: u32,
}

#[derive(Serialize, Deserialize, Clone, TS, Debug)]
#[ts(export)]
pub struct Items {
    weapons: Vec<Weapon>,
    armors: Vec<Armor>,
    shields: Vec<Shield>,
    equipments: Vec<Equipment>,
    pub treasure: Treasure,
    #[serde(skip_deserializing)]
    derived: Derived,
}

impl Default for Items {
    fn default() -> Self {
        Self::new()
    }
}

impl Items {
    pub fn new() -> Self {
        Self {
            weapons: vec![],
            armors: vec![],
            shields: vec![],
            equipments: vec![],
            treasure: Treasure::new(),
            derived: Derived::default(),
        }
    }

    pub fn get_items_id(&self, tp: ItemType) -> Vec<i32> {
        match tp {
            ItemType::Armor => self
                .armors
                .iter()
                .map(|item| item.get_id().expect("Armor should have an id here"))
                .collect(),
            ItemType::Equipment => self
                .equipments
                .iter()
                .map(|item| item.get_id().expect("Equipment should have an id here"))
                .collect(),

            ItemType::Weapon => self
                .weapons
                .iter()
                .map(|item| item.get_id().expect("Weapon should have an id here"))
                .collect(),

            ItemType::Shield => self
                .shields
                .iter()
                .map(|item| item.get_id().expect("Shieldi should have an id here"))
                .collect(),
        }
    }

    pub async fn from_db_model(
        db: &DatabaseConnection,
        character: &character_entity::Model,
    ) -> Result<Self> {
        let weapons = Weapon::from_db_model(db, character.id).await?;
        let armors = Armor::from_db_model(db, character.id).await?;
        let shields = Shield::from_db_model(db, character.id).await?;
        let equipments = Equipment::from_db_model(db, character.id).await?;
        let mut treasure = Treasure::new();
        treasure.set_treasure(character.items_treasure_total.try_into()?);
        treasure.set_carrying(character.items_treasure_carrying.try_into()?);

        Ok(Self {
            weapons,
            armors,
            shields,
            equipments,
            treasure,
            derived: Derived::default(),
        })
    }

    pub fn compute_derived_values(&mut self, data: &DerivedData) {
        for weapon in &mut self.weapons {
            weapon.compute_derived_values(&data.data);
        }
        for armor in &mut self.armors {
            armor.compute_derived_values(&data.data);
        }
        for shield in &mut self.shields {
            shield.compute_derived_values(&data.data);
        }
        for equipment in &mut self.equipments {
            equipment.compute_derived_values(&data.data);
        }
        self.derived.protection = self.get_protection();
        self.derived.parry = self.get_parry();
        self.derived.load = self.get_load(&data.virtues);
        self.treasure.compute_derived_values();
    }

    pub fn get_protection(&self) -> u32 {
        self.armors.iter().fold(0, |acc, elem| {
            acc + if elem.is_wearing() {
                elem.get_protection()
            } else {
                0
            }
        })
    }

    pub fn get_parry(&self) -> u32 {
        self.shields.iter().fold(0, |acc, elem| {
            acc + if elem.is_wearing() {
                elem.get_parry().computed_value
            } else {
                0
            }
        })
    }

    fn add_item_internal<T: ItemTrait + Clone>(items: &mut Vec<T>, item: &T) -> Result<()> {
        if let Some(id) = item.get_id() {
            let index = items
                .iter()
                .position(|x| {
                    if let Some(x_id) = x.get_id() {
                        x_id == id
                    } else {
                        false
                    }
                })
                .ok_or(anyhow!("Failed to find item"))?;
            items[index] = item.clone();
        } else {
            items.push(item.clone());
        }
        Ok(())
    }

    pub fn add_item(&mut self, item: &Item) -> Result<()> {
        match item {
            Item::Armor(x) => Self::add_item_internal(&mut self.armors, x),
            Item::Weapon(x) => Self::add_item_internal(&mut self.weapons, x),
            Item::Equipment(x) => Self::add_item_internal(&mut self.equipments, x),
            Item::Shield(x) => Self::add_item_internal(&mut self.shields, x),
        }
    }

    /// Not safe: does not check owner
    fn delete_item_intern<T>(vec: &mut Vec<T>, id: i32) -> Result<()>
    where
        T: ItemTrait,
    {
        let index = vec.iter().position(|x| {
            if let Some(x_id) = x.get_id() {
                x_id == id
            } else {
                false
            }
        });
        if let Some(index) = index {
            vec.remove(index);
            Ok(())
        } else {
            bail!(format!("Cannot find item {id}"))
        }
    }

    /// Not safe: does not check owner
    pub fn delete_item(&mut self, eq_type: ItemType, id: i32) -> Result<()> {
        match eq_type {
            ItemType::Armor => Items::delete_item_intern(&mut self.armors, id),
            ItemType::Equipment => Items::delete_item_intern(&mut self.equipments, id),
            ItemType::Weapon => Items::delete_item_intern(&mut self.weapons, id),
            ItemType::Shield => Items::delete_item_intern(&mut self.shields, id),
        }
    }

    /// Update wearing
    /// Not safe: does not check owner
    fn update_wearing_intern<T>(vec: &mut [T], id: i32, value: bool) -> Result<()>
    where
        T: ItemTrait,
    {
        let index = vec.iter().position(|x| {
            if let Some(x_id) = x.get_id() {
                x_id == id
            } else {
                false
            }
        });
        if let Some(index) = index {
            vec[index].set_wearing(value);
            Ok(())
        } else {
            bail!(format!("Cannot find item {id}"))
        }
    }

    /// Not safe: does not check owner
    pub fn update_wearing(&mut self, eq_type: ItemType, id: i32, value: bool) -> Result<()> {
        match eq_type {
            ItemType::Armor => Items::update_wearing_intern(&mut self.armors, id, value),
            ItemType::Equipment => Items::update_wearing_intern(&mut self.equipments, id, value),
            ItemType::Weapon => Items::update_wearing_intern(&mut self.weapons, id, value),
            ItemType::Shield => Items::update_wearing_intern(&mut self.shields, id, value),
        }
    }

    fn get_load_items<T>(items: &[T], virtues: &[VirtueEnum]) -> u32
    where
        T: ItemTrait,
    {
        items.iter().fold(0, |acc, elem| {
            acc + if elem.is_wearing() {
                elem.get_load(virtues).computed_value
            } else {
                0
            }
        })
    }

    pub async fn update_ids<C: ConnectionTrait>(
        &mut self,
        db: &C,
        character_id: i32,
    ) -> Result<()> {
        Equipment::update_ids(db, character_id, &mut self.equipments).await?;
        Weapon::update_ids(db, character_id, &mut self.weapons).await?;
        Shield::update_ids(db, character_id, &mut self.shields).await?;
        Armor::update_ids(db, character_id, &mut self.armors).await?;

        Ok(())
    }

    pub fn get_load(&self, virtues: &[VirtueEnum]) -> u32 {
        Self::get_load_items(&self.armors, virtues)
            + Self::get_load_items(&self.weapons, virtues)
            + Self::get_load_items(&self.shields, virtues)
            + Self::get_load_items(&self.equipments, virtues)
            + self.treasure.get_load()
    }

    fn get_starting_eye_intern<T>(vec: &[T]) -> u32
    where
        T: ItemTrait,
    {
        vec.iter()
            .fold(0, |acc, item| acc + item.is_famous() as u32)
    }

    pub fn get_starting_eye(&self) -> u32 {
        Self::get_starting_eye_intern(&self.armors)
            + Self::get_starting_eye_intern(&self.weapons)
            + Self::get_starting_eye_intern(&self.shields)
            + Self::get_starting_eye_intern(&self.equipments)
    }

    /// Not safe: does not check ownership
    pub async fn delete_by_character_id<C: ConnectionTrait>(
        db: &C,
        character_id: i32,
    ) -> Result<()> {
        Equipment::delete_by_character_id(db, character_id).await?;
        Weapon::delete_by_character_id(db, character_id).await?;
        Shield::delete_by_character_id(db, character_id).await?;
        Armor::delete_by_character_id(db, character_id).await
    }

    pub async fn update_links<C: ConnectionTrait>(
        &mut self,
        db: &C,
        character_id: i32,
        rewards: &mut [Reward],
    ) -> Result<()> {
        Equipment::update_links(db, character_id, &mut self.equipments, rewards).await?;
        Weapon::update_links(db, character_id, &mut self.weapons, rewards).await?;
        Shield::update_links(db, character_id, &mut self.shields, rewards).await?;
        Armor::update_links(db, character_id, &mut self.armors, rewards).await
    }

    /// Not safe: does not check ownership
    pub async fn save_items<C: ConnectionTrait, E: ItemTrait + Clone>(
        db: &C,
        character_id: i32,
        items: &[E],
    ) -> Result<()> {
        let previous = E::from_db_model(db, character_id).await?;

        // Convert to a map
        let mut in_db = HashMap::<i32, E>::new();
        for x in &previous {
            in_db.insert(x.get_id().expect("Should have an id"), x.clone());
        }

        // Add new items
        for item in items {
            let id = item.get_id();
            if let Some(id) = id
                && in_db.contains_key(&id)
            {
                in_db.remove(&id).unwrap();
            }
            item.save(db, character_id).await?;
        }

        // Remove old advantages
        for id in in_db.keys() {
            E::delete_by_id(db, character_id, *id).await?;
        }

        Ok(())
    }

    /// Not safe: does not check ownership
    pub async fn save<C: ConnectionTrait>(&self, db: &C, character_id: i32) -> Result<()> {
        Self::save_items(db, character_id, &self.weapons).await?;
        Self::save_items(db, character_id, &self.armors).await?;
        Self::save_items(db, character_id, &self.shields).await?;
        Self::save_items(db, character_id, &self.equipments).await
    }
}
