console.log('Loading utils');
import { Tolgee, FormatSimple, LanguageDetector } from '@tolgee/svelte';
import * as local_fr from '$lib/i18n/fr.json';
import * as local_en from '$lib/i18n/en.json';
import * as local_es from '$lib/i18n/es.json';
import type { Character } from 'the-one-app-api/the_one_app/bindings/Character';
import { store_set, store_get } from './store';
import { toast } from '@zerodevx/svelte-toast';

export const md_size = 768;

class Data {
    is_player: boolean = $state(true);
    locked: boolean = $state(true);
    character_id: null | number = $state(null);
    character: null | Character = $state(null);
    enhanced_accessibility: boolean = $state(false);
    dark: boolean = $state(true);

    async init(): Promise<void> {
        const id = await store_get('character_id', null);
        this.character_id = id == null ? null : Number(id);
        this.locked = (await store_get('locked', 'true')) == 'true';
        this.is_player = (await store_get('is_player', 'true')) == 'true';
        this.enhanced_accessibility = (await store_get('accessibility', 'false')) == 'true';
        this.update_dark();
    }

    update_dark() {
        this.dark = localStorage.getItem('THEME_PREFERENCE_KEY') === 'dark';
    }

    set_dark(dark: boolean) {
        this.dark = dark;
    }

    async set_character(id: number): Promise<void> {
        await store_set('character_id', id.toString());
        this.character_id = id;
    }
}

export const data = new Data();

export const tolgee = Tolgee()
    .use(FormatSimple())
    .use(LanguageDetector())
    .init({
        language: 'en',
        availableLanguages: ['en', 'fr', 'es'],
        fallbackLanguage: 'en',
        staticData: {
            en: local_en,
            fr: local_fr,
            es: local_es
        }
    });

export async function init_language(): Promise<void> {
    const lang = await store_get('language');
    if (lang != null) {
        tolgee.changeLanguage(lang);
    }
}

export function raise_error(text: string): never {
    toast.push(text);
    throw text;
}

export function capitalizeFirstLetter(str: string): string {
    return str.charAt(0).toUpperCase() + str.slice(1);
}

export function is_unique(array: string[]): boolean {
    const unique = array.filter((v, i, a) => {
        return a.indexOf(v) == i;
    });
    return unique.length == array.length;
}

export function is_mobile(): boolean {
    const regexp = /android|iphone|kindle|ipad/i;
    return regexp.test(navigator.userAgent) || window.innerWidth < 800;
}

export function is_wounded(character: Character): boolean {
    return character.injury.duration > 0;
}

export function compare_string(a: string, b: string): number {
    const nameA = a.toUpperCase(); // ignore upper and lowercase
    const nameB = b.toUpperCase(); // ignore upper and lowercase
    if (nameA < nameB) {
        return -1;
    }
    if (nameA > nameB) {
        return 1;
    }

    // names must be equal
    return 0;
}

interface OfficialPersonal {
    name: string;
    id: number | null;
}

export function zip_official_personal<K, V extends OfficialPersonal>(
    official: Map<K, V>,
    personal: Array<V>,
    t: (_: string) => string
): { value: (K | number)[]; shown: string[] } {
    const official_map = official
        .entries()
        .map((k) => {
            return { value: k[0], name: t(k[1].name) };
        })
        .toArray()
        .sort((a, b) => compare_string(a.name, b.name));
    const personal_map = personal
        .map((x) => {
            if (x.id == null) {
                toast.push('Id should not be null in personal data');
                throw 'Id should not be null in personal data';
            }
            return { value: x.id, name: x.name };
        })
        .sort((a, b) => compare_string(a.name, b.name));
    const both = [...official_map, ...personal_map];

    return {
        value: both.map((x) => x.value),
        shown: both.map((x) => x.name)
    };
}
