import { store_set, store_get, store_remove } from './store';
import type { Fellowship } from 'the-one-app-api/the_one_app/bindings/Fellowship';
import {
    get_fellowship_backend,
    get_backend,
    set_fellowship_backend,
    own_fetch
} from './queries.svelte';
import { toast } from '@zerodevx/svelte-toast';
import { data } from './utils.svelte';
import { BackendServer, FellowshipAuth } from 'the-one-app-api/the_one_app/queries';
import { setup_socket } from './socket.svelte';

class FellowshipData {
    auth: null | FellowshipAuth = $state(null);
    fellowship: null | Fellowship = $state(null);
    id: null | number = $state(null);

    async set_fellowship_auth(auth: FellowshipAuth): Promise<void> {
        await store_set('fellowship_auth', JSON.stringify(auth));
        this.auth = auth;
        const backend = new BackendServer(this.auth.server, auth, true);
        backend.fetch_function = own_fetch;
        set_fellowship_backend(backend);
        await setup_socket();
    }

    get_players(): string[] {
        if (this.fellowship == null) {
            return [];
        }
        return this.fellowship.characters.map((x) => x.player_name);
    }

    get_loremaster(): string {
        if (this.auth == null) {
            throw 'No fellowship available';
        }
        return this.auth.loremaster;
    }

    async init_fellowship(): Promise<void> {
        if (data.is_player) {
            const local_data = await store_get('fellowship_auth');
            if (local_data == null) {
                return;
            } else {
                const data: FellowshipAuth = JSON.parse(local_data);
                const local_auth = new FellowshipAuth(
                    data.fellowship,
                    data.loremaster,
                    data.server,
                    data.player
                );
                this.set_fellowship_auth(local_auth);
            }
        } else {
            const json_id = await store_get('fellowship_id');
            if (json_id == null) {
                return;
            } else {
                const id = JSON.parse(json_id) as number;
                this.id = id;
            }
        }

        await this.update_fellowship();
    }

    async update_fellowship(): Promise<void> {
        if (data.is_player && this.auth == null) {
            return;
        }
        try {
            if (data.is_player) {
                const backend = await get_fellowship_backend();
                this.fellowship = (await backend.get_player_fellowship()).expect(
                    'Failed to get fellowship'
                );
            } else {
                const backend = await get_backend();
                const id = this.id;
                if (id != null) {
                    this.fellowship = (await backend.get_fellowship(id)).expect(
                        'Failed to get fellowship'
                    );
                }
            }
        } catch (err) {
            await store_remove('fellowship_id');
            throw err;
        }
    }

    async set_fellowship_id(id: number): Promise<void> {
        await store_set('fellowship_id', JSON.stringify(id));
        this.id = id;
    }

    get_fellowship_id(): number {
        const f = this.fellowship;
        if (f == null) {
            throw 'No fellowship available';
        }
        if (f.id == null) {
            toast.push('No fellowship id available. This is a bug, please contact me');
            throw 'Fellowship id is null, should not happen';
        }
        return f.id;
    }

    get_headers(): { [id: string]: string } {
        const headers: { [id: string]: string } = {};
        if (this.auth) {
            headers['fellowship'] = this.auth.fellowship;
            headers['loremaster'] = this.auth.loremaster;
        }
        return headers;
    }
}

export const fellowship_data = new FellowshipData();
