import { AsyncOption } from './asyncoption.js';
import { toString } from './utils.js';
import { Ok, Err } from './result.js';
/**
 * Contains the None value
 */
var NoneImpl = /** @class */ (function () {
    function NoneImpl() {
    }
    NoneImpl.prototype.isSome = function () {
        return false;
    };
    NoneImpl.prototype.isNone = function () {
        return true;
    };
    NoneImpl.prototype[Symbol.iterator] = function () {
        return {
            next: function () {
                return { done: true, value: undefined };
            },
        };
    };
    NoneImpl.prototype.unwrapOr = function (val) {
        return val;
    };
    NoneImpl.prototype.unwrapOrElse = function (f) {
        return f();
    };
    NoneImpl.prototype.expect = function (msg) {
        throw new Error("".concat(msg));
    };
    NoneImpl.prototype.unwrap = function () {
        throw new Error("Tried to unwrap None");
    };
    NoneImpl.prototype.map = function (_mapper) {
        return this;
    };
    NoneImpl.prototype.mapOr = function (default_, _mapper) {
        return default_;
    };
    NoneImpl.prototype.mapOrElse = function (default_, _mapper) {
        return default_();
    };
    NoneImpl.prototype.or = function (other) {
        return other;
    };
    NoneImpl.prototype.orElse = function (other) {
        return other();
    };
    NoneImpl.prototype.andThen = function (op) {
        return this;
    };
    NoneImpl.prototype.toResult = function (error) {
        return Err(error);
    };
    NoneImpl.prototype.toString = function () {
        return 'None';
    };
    NoneImpl.prototype.toAsyncOption = function () {
        return new AsyncOption(None);
    };
    return NoneImpl;
}());
// Export None as a singleton, then freeze it so it can't be modified
export var None = new NoneImpl();
Object.freeze(None);
/**
 * Contains the success value
 */
var SomeImpl = /** @class */ (function () {
    function SomeImpl(val) {
        if (!(this instanceof SomeImpl)) {
            return new SomeImpl(val);
        }
        this.value = val;
    }
    SomeImpl.prototype.isSome = function () {
        return true;
    };
    SomeImpl.prototype.isNone = function () {
        return false;
    };
    SomeImpl.prototype[Symbol.iterator] = function () {
        return [this.value][Symbol.iterator]();
    };
    SomeImpl.prototype.unwrapOr = function (_val) {
        return this.value;
    };
    SomeImpl.prototype.unwrapOrElse = function (_f) {
        return this.value;
    };
    SomeImpl.prototype.expect = function (_msg) {
        return this.value;
    };
    SomeImpl.prototype.unwrap = function () {
        return this.value;
    };
    SomeImpl.prototype.map = function (mapper) {
        return Some(mapper(this.value));
    };
    SomeImpl.prototype.mapOr = function (_default_, mapper) {
        return mapper(this.value);
    };
    SomeImpl.prototype.mapOrElse = function (_default_, mapper) {
        return mapper(this.value);
    };
    SomeImpl.prototype.or = function (_other) {
        return this;
    };
    SomeImpl.prototype.orElse = function (_other) {
        return this;
    };
    SomeImpl.prototype.andThen = function (mapper) {
        return mapper(this.value);
    };
    SomeImpl.prototype.toResult = function (error) {
        return Ok(this.value);
    };
    SomeImpl.prototype.toAsyncOption = function () {
        return new AsyncOption(this);
    };
    /**
     * Returns the contained `Some` value, but never throws.
     * Unlike `unwrap()`, this method doesn't throw and is only callable on an Some<T>
     *
     * Therefore, it can be used instead of `unwrap()` as a maintainability safeguard
     * that will fail to compile if the type of the Option is later changed to a None that can actually occur.
     *
     * (this is the `into_Some()` in rust)
     */
    SomeImpl.prototype.safeUnwrap = function () {
        return this.value;
    };
    SomeImpl.prototype.toString = function () {
        return "Some(".concat(toString(this.value), ")");
    };
    SomeImpl.EMPTY = new SomeImpl(undefined);
    return SomeImpl;
}());
// This allows Some to be callable - possible because of the es5 compilation target
export var Some = SomeImpl;
export var Option;
(function (Option) {
    /**
     * Parse a set of `Option`s, returning an array of all `Some` values.
     * Short circuits with the first `None` found, if any
     */
    function all() {
        var options = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            options[_i] = arguments[_i];
        }
        var someOption = [];
        for (var _a = 0, options_1 = options; _a < options_1.length; _a++) {
            var option = options_1[_a];
            if (option.isSome()) {
                someOption.push(option.value);
            }
            else {
                return option;
            }
        }
        return Some(someOption);
    }
    Option.all = all;
    /**
     * Parse a set of `Option`s, short-circuits when an input value is `Some`.
     * If no `Some` is found, returns `None`.
     */
    function any() {
        var options = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            options[_i] = arguments[_i];
        }
        // short-circuits
        for (var _a = 0, options_2 = options; _a < options_2.length; _a++) {
            var option = options_2[_a];
            if (option.isSome()) {
                return option;
            }
            else {
                continue;
            }
        }
        // it must be None
        return None;
    }
    Option.any = any;
    function isOption(value) {
        return value instanceof Some || value === None;
    }
    Option.isOption = isOption;
})(Option || (Option = {}));
//# sourceMappingURL=option.js.map