import { MyError, Err, Ok, type Try } from '../utils/error';
import { type Result } from 'ts-results-es';
import { type Version } from './bindings/Version';
import { type NameGenerators } from './bindings/NameGenerators';
import { type Genders } from './bindings/Genders';
import { type RollType } from './bindings/RollType';
import { type RollStatistics } from './bindings/RollStatistics';
import { type Choices } from './bindings/Choices';
import { type Culture } from './bindings/Culture';
import { type CultureEnum } from './bindings/CultureEnum';
import { type Calling } from './bindings/Calling';
import { type CallingEnum } from './bindings/CallingEnum';
import { type CombatSkills } from './bindings/CombatSkills';
import { type SkillEnum } from './bindings/SkillEnum';
import { type CombatSkillEnum } from './bindings/CombatSkillEnum';
import { type Character } from './bindings/Character';
import { type SecondaryAttributeEnum } from './bindings/SecondaryAttributeEnum';
import { type CharacterRoll } from './bindings/CharacterRoll';
import { type Adversary } from './bindings/Adversary';
import { type HateEnum } from './bindings/HateEnum';
import { type CombatProficiency } from './bindings/CombatProficiency';
import { type UndertakingEnum } from './bindings/UndertakingEnum';
import { type StandardOfLiving } from './bindings/StandardOfLiving';
import { type EquipmentType } from './bindings/EquipmentType';
import { type Region } from './bindings/Region';

import { type Undertaking } from './bindings/Undertaking';
import { type Fellowship } from './bindings/Fellowship';
import { type Patron } from './bindings/Patron';
import { type PatronEnum } from './bindings/PatronEnum';
import { type AddToFellowship } from './bindings/AddToFellowship';
import { type AdversaryEnum } from './bindings/AdversaryEnum';
import { type Weapon } from './bindings/Weapon';
import { type SkillType } from './bindings/SkillType';
import { type Advantage, type AdvantageEnum, type AdvantageMap, AdvantageType } from './advantage';
import { type Item } from './bindings/Item';
import { type WeaponEnum } from './bindings/WeaponEnum';
import { type ArmorEnum } from './bindings/ArmorEnum';
import { type ShieldEnum } from './bindings/ShieldEnum';
import { type Ruin } from './bindings/Ruin';

export type AdversaryMap = Map<AdversaryEnum, Adversary>;
export type ItemEnum = WeaponEnum | ArmorEnum | ShieldEnum;
export enum ItemTypes {
    Shield = 'Shield',
    Armor = 'Armor',
    Weapon = 'Weapon',
    Equipment = 'Equipment'
}
export type PatronMap = Map<PatronEnum, Patron>;

interface Auth {
    get_headers(): { [id: string]: string };
    get_username(): string;
}

export class UserInClear implements Auth {
    username: string = '';
    password: string = '';
    constructor(username: string, password: string) {
        this.username = username;
        this.password = password;
    }

    get_headers(): { [id: string]: string } {
        return {
            username: this.username,
            password: this.password
        };
    }
    get_username(): string {
        return this.username;
    }
}

export class FellowshipAuth implements Auth {
    fellowship: string = '';
    loremaster: string = '';
    player: string = '';
    server: URL = new URL('http://localhost');

    constructor(fellowship: string, loremaster: string, server: URL, player: string) {
        this.fellowship = fellowship;
        this.loremaster = loremaster;
        this.server = server;
        this.player = player;
    }
    get_headers(): { [id: string]: string } {
        return {
            fellowship: this.fellowship,
            loremaster: this.loremaster
        };
    }
    get_username(): string {
        return this.player;
    }
}

export class BackendServer {
    backend_url: URL;
    auth: null | Auth = null;
    use_auth: boolean = true;
    fetch_function: (path: string, args: RequestInit | undefined) => Promise<Response> = (
        path,
        args
    ) => {
        const url = new URL(path, this.backend_url);
        return fetch(url, args);
    };

    constructor(backend_url: URL, auth: Auth, use_auth: boolean) {
        this.backend_url = backend_url;
        this.auth = auth;
        this.use_auth = use_auth;
    }

    get_headers(json: boolean): { [id: string]: string } {
        const headers: { [id: string]: string } = json
            ? { 'Content-Type': 'application/json' }
            : {};
        if (this.auth) {
            Object.assign(headers, this.auth.get_headers());
        }
        return headers;
    }

    reset() {
        this.auth = null;
    }

    async call_backend<Type>(
        method: string,
        path: string,
        body: BodyInit | null = null,
        no_answer: boolean = false,
        json: boolean = true
    ): Promise<Result<Type, MyError>> {
        try {
            const headers = this.get_headers(json);
            const response = await this.fetch_function(path, {
                method: method,
                body: body,
                headers: headers
            });
            if (!response.ok) {
                const text = await response.text();
                return Err(new MyError(text));
            }
            if (no_answer) {
                return Ok(null as Type);
            }
            return Ok(await response.json());
        } catch (error) {
            console.log(error);
            return Err(new MyError(String(error)));
        }
    }

    /**********************************/
    /*           GENERAL              */
    /**********************************/
    async create_user(username: string, password: string): Promise<Try<void>> {
        try {
            const response = await this.fetch_function(`/general/create-user/${username}`, {
                method: 'POST',
                body: password
            });
            if (!response.ok) {
                const text = await response.text();
                return Err(new MyError(`Failed to create account: ${text}`));
            }
            return Ok(void true);
        } catch (error) {
            console.log(error);
            return Err(new MyError(`Failed to create account: ${error}`));
        }
    }

    async check_authentication(): Promise<boolean> {
        if (this.auth == null) {
            return true;
        }
        const headers = this.auth.get_headers();

        const response = await this.fetch_function('/general/check-auth', {
            headers: headers
        });

        return response.ok;
    }
    async get_version(): Promise<Try<Version>> {
        const resp = await this.fetch_function('/general/version', undefined);
        if (!resp.ok) {
            const text = await resp.text();
            return Err(new MyError(`Failed to get version: ${text}`));
        }
        return Ok(await resp.json());
    }

    async requires_authentication(): Promise<Try<boolean>> {
        const out = await this.call_backend<string>(
            'GET',
            '/general/requires-authentication',
            null
        );

        if (out.isErr()) {
            return out;
        }

        return Ok(out.unwrap() === 'true');
    }

    async can_update(): Promise<Try<boolean>> {
        return await this.call_backend('GET', '/general/can-update');
    }

    async update(): Promise<Try<void>> {
        return await this.call_backend('POST', '/general/update', null, true);
    }

    /**********************************/
    /*            TOOLS               */
    /**********************************/
    async get_genders(race: NameGenerators): Promise<Try<Genders[]>> {
        return this.call_backend('GET', `/tools/name-generator/${race}/genders`, null);
    }

    async roll_name_generator(
        n_rolls: number,
        race: NameGenerators,
        gender: Genders
    ): Promise<Try<string[]>> {
        return this.call_backend(
            'POST',
            `/tools/name-generator/roll/${n_rolls}/${race}/${gender}`,
            null
        );
    }

    async roll_random_features(n_rolls: number, positive: boolean): Promise<Try<string[]>> {
        return this.call_backend(
            'POST',
            `/tools/random-features/roll/${n_rolls}/${positive}`,
            null
        );
    }

    async get_roll_statistics(
        n_rolls: number,
        number_dices: number,
        roll_type: RollType,
        miserable: boolean,
        weary: boolean
    ): Promise<Try<RollStatistics>> {
        return this.call_backend(
            'GET',
            `/tools/roll-statistics/${n_rolls}/${number_dices}/${roll_type}/${miserable}/${weary}`,
            null
        );
    }

    async get_random_ruin(): Promise<Try<Ruin>> {
        return this.call_backend('POST', `/tools/ruin`, null);
    }

    /**********************************/
    /*         NEW CHARACTER          */
    /**********************************/

    async get_choices(strider: boolean): Promise<Try<Choices>> {
        // Get empty character
        return this.call_backend('GET', `/new-character/empty/${strider}`, null);
    }

    async get_culture_options(culture: CultureEnum): Promise<Try<Culture>> {
        return this.call_backend('GET', `/new-character/culture/${culture}`, null);
    }

    async get_calling_options(calling: CallingEnum): Promise<Try<Calling>> {
        return this.call_backend('GET', `/new-character/calling/${calling}`, null);
    }

    async get_new_character_combat_skills(choices: Choices): Promise<Try<CombatSkills>> {
        return this.call_backend('POST', '/new-character/combat-skills', JSON.stringify(choices));
    }

    async can_upgrade_skill(
        previous_experience: number,
        skills: number[]
    ): Promise<Try<SkillEnum[]>> {
        return this.call_backend(
            'POST',
            `/new-character/skills/can-upgrade/${previous_experience}`,
            JSON.stringify(skills)
        );
    }

    async can_upgrade_combat_skill(
        previous_experience: number,
        combat_skills: number[]
    ): Promise<Try<CombatSkillEnum[]>> {
        return this.call_backend(
            'POST',
            `/new-character/combat-skills/can-upgrade/${previous_experience}`,
            JSON.stringify(combat_skills)
        );
    }

    async get_initial_skill_cost(new_value: number): Promise<Try<number>> {
        return this.call_backend('GET', `/new-character/skill-cost/${new_value}`, null);
    }

    // TODO merge with previous
    async get_initial_combat_skill_cost(new_value: number): Promise<Try<number>> {
        return this.call_backend('GET', `/new-character/combat-skill-cost/${new_value}`, null);
    }

    async create_character(choices: Choices): Promise<Try<Character>> {
        return await this.call_backend('PUT', '/new-character/create', JSON.stringify(choices));
    }
    /**********************************/
    /*            CHARACTER           */
    /**********************************/

    async get_character(character_id: number): Promise<Try<Character>> {
        const character = await this.call_backend<Character>(
            'GET',
            `/character/${character_id}`,
            null
        );

        // Convert fields for frontend
        return character;
    }

    async delete_character(character_id: number) {
        return this.call_backend('DELETE', `/character/${character_id}`, null, true);
    }

    async set_character(character: Character): Promise<Try<number>> {
        const out = await this.call_backend<number>(
            'PUT',
            `/character`,
            JSON.stringify(character),
            true
        );
        return out;
    }

    async get_list_characters(): Promise<Try<number[]>> {
        return this.call_backend('GET', '/character', null);
    }

    async get_notes(character_id: number): Promise<Try<string>> {
        return this.call_backend('GET', `/character/${character_id}/notes/`, null);
    }

    async set_notes(character_id: number, notes: string): Promise<Try<void>> {
        return this.call_backend('POST', `/character/${character_id}/notes/`, notes, true, false);
    }

    async get_upgrade_skills(character_id: number): Promise<Try<SkillEnum[]>> {
        return this.call_backend('GET', `/character/${character_id}/skills/can-upgrade`);
    }

    async get_upgrade_combat_skills(character_id: number): Promise<Try<CombatSkillEnum[]>> {
        return this.call_backend('GET', `/character/${character_id}/combat-skills/can-upgrade`);
    }

    async get_upgrade_secondary_attributes(
        character_id: number
    ): Promise<Try<SecondaryAttributeEnum[]>> {
        return this.call_backend(
            'GET',
            `/character/${character_id}/secondary-attributes/can-upgrade`
        );
    }

    async update_skill(
        character_id: number,
        skill_type: 'skill' | 'combat-skill',
        delta: number,
        label: SkillEnum | CombatSkillEnum
    ): Promise<Try<void>> {
        const out = await this.call_backend<void>(
            'POST',
            `/character/${character_id}/increase-${skill_type}/${label}/${delta}`,
            label
        );
        return out;
    }

    async increase_secondary_attribute(
        character_id: number,
        label: SecondaryAttributeEnum,
        advantage: Advantage,
        delta: number,
        free_change = false
    ): Promise<Try<[number, Advantage[]]>> {
        const free = free_change ? '?free' : '';
        const out = await this.call_backend<[number, Advantage[]]>(
            'POST',
            `/character/${character_id}/increase-secondary-attribute/${label}/${delta}${free}`,
            JSON.stringify(advantage)
        );
        return out;
    }

    async loose(
        character_id: number,
        type: 'endurance' | 'hope',
        amount: number
    ): Promise<Try<number>> {
        const resp = await this.call_backend<number>(
            'POST',
            `/character/${character_id}/loose-${type}/${amount}`
        );
        return resp;
    }

    async get_current_skill_points(character_id: number): Promise<Try<number>> {
        return this.call_backend('GET', `/character/${character_id}/current-skill-points`, null);
    }

    async get_current_adventure_points(character_id: number): Promise<Try<number>> {
        return this.call_backend(
            'GET',
            `/character/${character_id}/current-adventure-points`,
            null
        );
    }

    async rest(character_id: number, long: boolean): Promise<Try<void>> {
        const desc = long ? 'long' : 'short';
        const out = await this.call_backend<void>(
            'POST',
            `/character/${character_id}/${desc}-rest`,
            null,
            true
        );
        return out;
    }

    async take_fellowship_phase(
        character_id: number,
        yule: boolean,
        undertaking: UndertakingEnum
    ): Promise<Try<void>> {
        const out = await this.call_backend<void>(
            'POST',
            `/character/${character_id}/fellowship-phase/${undertaking}/${yule}`,
            null,
            true
        );
        return out;
    }

    async reward_xp(character_id: number, skill: number, adventure: number): Promise<Try<void>> {
        const out = await this.call_backend<void>(
            'POST',
            `/character/${character_id}/reward-xp/skill/${skill}/adventure/${adventure}`,
            null,
            true
        );
        return out;
    }

    async convert_shadow(character_id: number): Promise<Try<void>> {
        const out = await this.call_backend<void>(
            'POST',
            `/character/${character_id}/convert-shadow`,
            null,
            true
        );
        return out;
    }

    async roll(
        character_id: number,
        skill: SkillType,
        bonus_dices: number,
        roll_type: RollType | null
    ): Promise<Try<CharacterRoll>> {
        const opts: string[][] = [];
        if (roll_type != null) {
            opts.push(['roll_type', roll_type]);
        }
        opts.push(['bonus_dices', String(bonus_dices)]);
        const params = new URLSearchParams(opts).toString();

        return this.call_backend(
            'POST',
            `/character/${character_id}/roll?${params}`,
            JSON.stringify(skill)
        );
    }

    /**********************************/
    /*            ADVERSARY           */
    /**********************************/

    async get_official_adversaries(): Promise<Try<AdversaryMap>> {
        const out: Try<{ [K in AdversaryEnum]: Adversary }> = await this.call_backend(
            'GET',
            `/adversary/official`,
            null
        );
        if (out.isOk()) {
            return Ok(new Map(Object.entries(out.unwrap())) as AdversaryMap);
        } else {
            return out;
        }
    }

    async get_personal_adversaries(): Promise<Try<Adversary[]>> {
        return this.call_backend('GET', `/adversary/personal`, null);
    }

    async get_adversary(name_or_id: string | number): Promise<Try<Adversary>> {
        return this.call_backend('GET', `/adversary/${name_or_id}`, null);
    }

    async delete_adversary(id: number): Promise<Try<null>> {
        return this.call_backend('DELETE', `/adversary/${id}`, null, true);
    }

    async set_adversary(adv: Adversary): Promise<Try<null>> {
        return this.call_backend('PUT', `/adversary`, JSON.stringify(adv), true);
    }

    async adversary_loose(
        adversary_id: number,
        type: 'Endurance' | HateEnum,
        amount: number
    ): Promise<Try<null>> {
        return await this.call_backend(
            'POST',
            `/adversary/${adversary_id}/loose-${type}/${amount}`,
            null,
            true
        );
    }

    /**********************************/
    /*              ENUM              */
    /**********************************/

    async get_cultures(): Promise<Try<CultureEnum[]>> {
        return await this.call_backend('GET', '/enums/cultures', null);
    }

    async get_callings(): Promise<Try<CallingEnum[]>> {
        return this.call_backend('GET', '/enums/callings', null);
    }

    async get_combat_skills(): Promise<Try<CombatSkillEnum[]>> {
        return this.call_backend('GET', '/enums/combat-skills', null);
    }

    async get_combat_proficiencies(): Promise<Try<CombatProficiency[]>> {
        return this.call_backend('GET', '/enums/combat-proficiencies', null);
    }

    async get_skills(): Promise<Try<SkillEnum[]>> {
        return this.call_backend('GET', '/enums/skills', null);
    }
    async get_undertakings(): Promise<Try<UndertakingEnum[]>> {
        return this.call_backend('GET', '/enums/undertaking', null);
    }

    async get_secondary_attributes(): Promise<Try<SecondaryAttributeEnum[]>> {
        return this.call_backend('GET', '/enums/secondary-attributes', null);
    }

    async get_standard_living(): Promise<Try<StandardOfLiving[]>> {
        return this.call_backend('GET', '/enums/standard-living', null);
    }

    async get_empty_item(type: EquipmentType): Promise<Try<Item>> {
        return this.call_backend('GET', `/enums/empty/item/${type}`, null);
    }

    async get_advantage(type: SecondaryAttributeEnum): Promise<Try<Advantage>> {
        return this.call_backend('GET', `/enums/empty/advantage/${type}`, null);
    }

    async get_name_generator_race(): Promise<Try<NameGenerators[]>> {
        return this.call_backend('GET', '/enums/name-generator/races', null);
    }

    async get_roll_type(): Promise<Try<RollType[]>> {
        return this.call_backend('GET', '/enums/roll-type', null);
    }

    async get_item(item_type: EquipmentType, item_name: ItemEnum): Promise<Try<Item>> {
        return this.call_backend('GET', `/enums/${item_type}/${item_name}`, null);
    }

    async get_list_items(item_type: EquipmentType): Promise<Try<ItemEnum[]>> {
        return this.call_backend('GET', `/enums/item/${item_type}`, null);
    }

    async get_standard_advantages(
        type: AdvantageType,
        culture: CultureEnum,
        at_creation: boolean
    ): Promise<Try<AdvantageMap>> {
        const list_params = [];
        if (culture != null && type == AdvantageType.Virtue) {
            list_params.push(`culture=${culture}`);
        }
        if (at_creation != null) {
            list_params.push(`creation=${at_creation}`);
        }
        const params = list_params.length == 0 ? '' : '?' + list_params.join('&');
        const out = await this.call_backend<Map<string, Advantage>>(
            'GET',
            `/enums/${type}${params}`
        );
        if (!out.isOk()) {
            return out;
        }
        return Ok(new Map(Object.entries(out.unwrap())) as AdvantageMap);
    }

    async get_standard_advantage(
        type: AdvantageType,
        name: AdvantageEnum
    ): Promise<Try<Advantage>> {
        return this.call_backend('GET', `/enums/${type}/${name}`);
    }

    async get_eye_threshold(eye: number): Promise<Try<Region>> {
        return this.call_backend('GET', `/enums/eye-threshold/${eye}`, null);
    }

    /**********************************/
    /*           EQUIPMENT            */
    /**********************************/

    async set_item(character_id: number, item_type: ItemTypes, item: Item): Promise<Try<void>> {
        if (
            item_type == ItemTypes.Weapon &&
            (item as unknown as Weapon).two_hand_injury?.value == 0
        ) {
            (item as unknown as Weapon).two_hand_injury = null;
        }
        const out = await this.call_backend<void>(
            'PUT',
            `/equipment/${character_id}/item`,
            JSON.stringify(item),
            true
        );
        return out;
    }

    async delete_item(character_id: number, item_type: ItemTypes, id: number): Promise<Try<void>> {
        const out = await this.call_backend<void>(
            'DELETE',
            `/equipment/${character_id}/${item_type}/${id}`,
            null,
            true
        );
        return out;
    }

    async update_wearing(
        character_id: number,
        item_type: ItemTypes,
        id: number,
        value: boolean
    ): Promise<Try<void>> {
        const out = await this.call_backend<void>(
            'PUT',
            `/equipment/${character_id}/${item_type}/wearing/${id}/${value}`,
            null,
            true
        );
        return out;
    }

    async link_reward(
        character_id: number,
        reward_id: number,
        item_type: ItemTypes,
        item_id: number
    ): Promise<Try<void>> {
        const out = await this.call_backend<void>(
            'POST',
            `/equipment/${character_id}/link/${reward_id}/${item_type}/${item_id}`,
            null,
            true
        );
        return out;
    }

    async unlink_reward(
        character_id: number,
        reward_id: number,
        item_type: ItemTypes,
        item_id: number
    ): Promise<Try<void>> {
        const out = await this.call_backend<void>(
            'DELETE',
            `/equipment/${character_id}/link/${reward_id}/${item_type}/${item_id}`,
            null,
            true
        );
        return out;
    }

    /**********************************/
    /*           FELLOWSHIP           */
    /**********************************/

    async get_undertaking(undertaking: UndertakingEnum): Promise<Try<Undertaking>> {
        return this.call_backend('GET', `/fellowship/undertaking/${undertaking}`, null);
    }

    async get_fellowships(): Promise<Try<Fellowship[]>> {
        return this.call_backend('GET', '/fellowship', null);
    }

    async get_fellowship(id: number): Promise<Try<Fellowship>> {
        return this.call_backend('GET', `/fellowship/${id}`, null);
    }

    async set_fellowship(fellowship: Fellowship): Promise<Try<number>> {
        return this.call_backend('POST', '/fellowship', JSON.stringify(fellowship));
    }
    async delete_fellowship(id: number): Promise<Try<Patron>> {
        return this.call_backend('DELETE', `/fellowship/${id}`, null, true);
    }

    async get_official_patrons(): Promise<Try<PatronMap>> {
        const out: Try<{ [K in PatronEnum]: Patron }> = await this.call_backend(
            'GET',
            '/fellowship/patron/official',
            null
        );
        if (out.isOk()) {
            return Ok(new Map(Object.entries(out.unwrap())) as PatronMap);
        } else {
            return out;
        }
    }
    async get_personal_patrons(): Promise<Try<Patron[]>> {
        return this.call_backend('GET', '/fellowship/patron/personal', null);
    }

    async get_patron(id: number | string): Promise<Try<Patron>> {
        return this.call_backend('GET', `/fellowship/patron/${id}`, null);
    }
    async set_patron(patron: Patron): Promise<Try<number>> {
        return this.call_backend('POST', `/fellowship/patron`, JSON.stringify(patron));
    }

    async delete_patron(id: number): Promise<Try<void>> {
        return this.call_backend('DELETE', `/fellowship/patron/${id}`, null, true);
    }

    async add_character_to_fellowship(data: AddToFellowship): Promise<Try<void>> {
        return this.call_backend('POST', `/fellowship/character`, JSON.stringify(data), true, true);
    }

    async add_adversary_to_fellowship(
        fellowship_id: number,
        adversary: Adversary
    ): Promise<Try<void>> {
        return this.call_backend(
            'POST',
            `/fellowship/${fellowship_id}/adversary`,
            JSON.stringify(adversary),
            true
        );
    }

    async delete_adversary_from_fellowship(
        fellowship_id: number,
        adversary_id: number
    ): Promise<Try<void>> {
        return this.call_backend(
            'DELETE',
            `/fellowship/${fellowship_id}/adversary/${adversary_id}`,
            null,
            true
        );
    }

    async delete_character_from_fellowship(
        fellowship_id: number,
        character_id: number
    ): Promise<Try<void>> {
        return this.call_backend(
            'DELETE',
            `/fellowship/${fellowship_id}/character/${character_id}`,
            null,
            true
        );
    }

    async get_player_fellowship(): Promise<Try<Fellowship>> {
        return this.call_backend('GET', '/fellowship/player', null);
    }

    async increase_eye(fellowship_id: number, inc: number): Promise<Try<number>> {
        return this.call_backend('POST', `/fellowship/${fellowship_id}/increase-eye/${inc}`, null);
    }

    async reset_fellowship_points(fellowship_id: number): Promise<Try<number>> {
        return this.call_backend('POST', `/fellowship/${fellowship_id}/reset-points`, null);
    }

    async reset_eye(fellowship_id: number): Promise<Try<number>> {
        return this.call_backend('POST', `/fellowship/${fellowship_id}/reset-eye`, null);
    }

    async increase_points(fellowship_id: number, inc: number): Promise<Try<number>> {
        return this.call_backend(
            'POST',
            `/fellowship/${fellowship_id}/increase-points/${inc}`,
            null
        );
    }
}
