var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { glob } from 'tinyglobby';
import { exitWithError } from './logger.js';
const GLOB_EXISTING_DOUBLE_STAR = /(\*\*)/g;
const GLOB_EXISTING_STAR = /(\*)/g;
const GLOB_EXISTING_ENUM = /\{([^}]*?,[^}]*?)\}/g;
const PLACEHOLDER_DOUBLE_ASTERISK = '__double_asterisk';
const PLACEHOLDER_ASTERISK = '__asterisk';
const PLACEHOLDER_ENUM_PREFIX = '__enum:';
export class FileMatcherException extends Error {
    constructor(message) {
        super(message);
        this.name = this.constructor.name;
    }
}
function splitToParts(template) {
    return template.split(/(\{.*?\})/g).filter(Boolean);
}
function getVariableName(part) {
    if (part.startsWith('{') && part.endsWith('}')) {
        return part.substring(1, part.length - 1).trim();
    }
    return false;
}
export function sanitizeTemplate(template) {
    var _a;
    let value = template;
    const matchedEnums = [...(((_a = value.match(GLOB_EXISTING_ENUM)) === null || _a === void 0 ? void 0 : _a.values()) || [])];
    matchedEnums.forEach((val) => {
        value = value.replace(val, `{${PLACEHOLDER_ENUM_PREFIX}${getVariableName(val)}}`);
    });
    value = value.replaceAll(GLOB_EXISTING_DOUBLE_STAR, '{' + PLACEHOLDER_DOUBLE_ASTERISK + '}');
    value = value.replaceAll(GLOB_EXISTING_STAR, '{' + PLACEHOLDER_ASTERISK + '}');
    return value;
}
export function getFileMatcher(file, template) {
    let fileName = file;
    const allVariables = {};
    const templateParts = splitToParts(template);
    for (const [i, part] of templateParts.entries()) {
        const variable = getVariableName(part);
        if (!variable) {
            if (fileName.startsWith(part)) {
                fileName = fileName.substring(part.length);
            }
            else {
                throw new FileMatcherException(`Unexpected part "${part}"`);
            }
        }
        else {
            const next = templateParts[i + 1];
            if (next) {
                const variableEnd = fileName.indexOf(next);
                if (getVariableName(next) || variableEnd === -1) {
                    throw new FileMatcherException(`Can't have two variables without separator (${part} + ${next})`);
                }
                else {
                    allVariables[variable] = fileName.substring(0, variableEnd);
                    fileName = fileName.substring(variableEnd);
                }
            }
            else {
                allVariables[variable] = fileName;
            }
        }
    }
    const result = { path: file };
    for (const [variable, value] of Object.entries(allVariables)) {
        if (variable === 'languageTag') {
            result.language = value;
        }
        else if (variable === 'snakeLanguageTag') {
            result.language = value.replaceAll('_', '-');
        }
        else if (variable === 'androidLanguageTag') {
            if (value[3] === 'r') {
                result.language =
                    value.substring(0, 3) + value.substring(4, value.length);
            }
            else {
                result.language = value;
            }
        }
        else if (variable === 'namespace') {
            result.namespace = value;
        }
        else if (variable !== 'extension' &&
            ![PLACEHOLDER_ASTERISK, PLACEHOLDER_DOUBLE_ASTERISK].includes(variable) &&
            !variable.startsWith(PLACEHOLDER_ENUM_PREFIX)) {
            throw new FileMatcherException(`Unknown variable "${variable}"`);
        }
    }
    return result;
}
export function getGlobPattern(template) {
    let value = template.replaceAll(GLOB_EXISTING_DOUBLE_STAR, '{__double_asterisk}');
    value = value.replaceAll(GLOB_EXISTING_STAR, '{__asterisk}');
    const parts = splitToParts(value);
    const globPattern = parts
        .map((part) => {
        const variableName = getVariableName(part);
        if (variableName) {
            if (variableName === PLACEHOLDER_DOUBLE_ASTERISK) {
                return '**';
            }
            else if (variableName.startsWith(PLACEHOLDER_ENUM_PREFIX)) {
                return ('{' + variableName.substring(PLACEHOLDER_ENUM_PREFIX.length) + '}');
            }
            else {
                return '*';
            }
        }
        else {
            return part;
        }
    })
        .join('');
    return globPattern;
}
export function findFilesByTemplate(template) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const sanitized = sanitizeTemplate(template);
            const globPattern = getGlobPattern(sanitized);
            const files = yield glob(globPattern, { onlyFiles: true, absolute: true });
            return files.map((file) => {
                return getFileMatcher(file, sanitized);
            });
        }
        catch (e) {
            if (e instanceof FileMatcherException) {
                exitWithError(e.message + ` in template ${template}`);
            }
            else {
                throw e;
            }
        }
    });
}
