import { extractString, isString } from './nodeUtils.js';
function shouldBeIgnored(context, line) {
    const commentAtLine = context.commentMap.get(line - 1);
    const isIgnore = (commentAtLine === null || commentAtLine === void 0 ? void 0 : commentAtLine.type) === 'MAGIC_COMMENT' && commentAtLine.kind === 'ignore';
    if (isIgnore) {
        context.unusedComments.delete(commentAtLine);
    }
    return isIgnore;
}
function commentKeyInfoOnLine(context, line) {
    const commentAtLine = context.commentMap.get(line - 1);
    const isKeyInfo = (commentAtLine === null || commentAtLine === void 0 ? void 0 : commentAtLine.type) === 'MAGIC_COMMENT' && commentAtLine.kind === 'key';
    if (isKeyInfo) {
        context.unusedComments.delete(commentAtLine);
        return keyInfoFromComment(context, commentAtLine);
    }
    return undefined;
}
function keyInfoFromComment(context, info) {
    var _a;
    return {
        keyName: info.keyName,
        namespace: (_a = info.namespace) !== null && _a !== void 0 ? _a : context.options.defaultNamespace,
        defaultValue: info.defaultValue,
        line: info.line,
    };
}
function reportKey(context, node, contextNs) {
    var _a;
    const { strictNamespace, defaultNamespace } = context.options;
    const { keys, warnings } = context;
    const { keyName, namespace: keyNs, defaultValue, line, dependsOnContext, optionsDynamic, } = node;
    if (shouldBeIgnored(context, line)) {
        return { keys, warnings };
    }
    const overrideInfo = commentKeyInfoOnLine(context, node.line);
    if (overrideInfo) {
        // key info is overriten by comment
        keys.push(overrideInfo);
        return;
    }
    if (!keyName || !isString(keyName)) {
        // dynamic key or key not present
        warnings.push({ line, warning: 'W_DYNAMIC_KEY' });
        return;
    }
    if (dependsOnContext && !contextNs && !keyNs && strictNamespace) {
        // there is no namespace source so namespace is ambiguous
        warnings.push({ line, warning: 'W_MISSING_T_SOURCE' });
        return;
    }
    if (optionsDynamic && strictNamespace) {
        // options of the key can't be analyzed, so again ambiguous namespace
        warnings.push({ line, warning: 'W_DYNAMIC_OPTIONS' });
        return;
    }
    const namespace = keyNs !== null && keyNs !== void 0 ? keyNs : (dependsOnContext ? contextNs === null || contextNs === void 0 ? void 0 : contextNs.name : undefined);
    if (namespace && !isString(namespace)) {
        // namespace is dynamic
        if (namespace === (contextNs === null || contextNs === void 0 ? void 0 : contextNs.name)) {
            // namespace coming from context
            warnings.push({ line, warning: 'W_UNRESOLVABLE_NAMESPACE' });
        }
        else {
            // namespace is directly on key
            warnings.push({
                line,
                warning: 'W_DYNAMIC_NAMESPACE',
            });
        }
        return;
    }
    if (defaultValue !== undefined && !isString(defaultValue)) {
        // this is just warning, we can still extract
        warnings.push({ line, warning: 'W_DYNAMIC_DEFAULT_VALUE' });
    }
    keys.push({
        line,
        keyName: extractString(keyName),
        namespace: (_a = extractString(namespace)) !== null && _a !== void 0 ? _a : defaultNamespace,
        defaultValue: extractString(defaultValue),
    });
}
function reportNs(context, node) {
    const { warnings } = context;
    const { line, name } = node;
    if (name && !isString(name)) {
        warnings.push({ line, warning: 'W_DYNAMIC_NAMESPACE' });
    }
}
function reportGeneral(context, node, contextNs) {
    if (!node) {
        return;
    }
    if (node.type === 'expr' || node.type === 'array') {
        let namespace = contextNs;
        for (const item of node.values) {
            if (item.type === 'nsInfo') {
                const oldNamespace = namespace;
                if (!shouldBeIgnored(context, item.line)) {
                    reportNs(context, item);
                    namespace = item;
                }
                // there might be nested stuff
                reportGeneral(context, item.name, oldNamespace);
                for (const i of item.values) {
                    reportGeneral(context, i, oldNamespace);
                }
            }
            else {
                reportGeneral(context, item, namespace);
            }
        }
    }
    else if (node.type === 'keyInfo') {
        reportKey(context, node, contextNs);
        // there might be nested stuff
        reportGeneral(context, node.keyName, contextNs);
        reportGeneral(context, node.namespace, contextNs);
        reportGeneral(context, node.defaultValue, contextNs);
        for (const i of node.values) {
            reportGeneral(context, i, contextNs);
        }
    }
    else if (node.type === 'dict') {
        for (const item of Object.values(node.value)) {
            reportGeneral(context, item, contextNs);
        }
        // go through values with unknown keynames
        for (const item of node.unknown) {
            reportGeneral(context, item, contextNs);
        }
    }
}
export function generateReport({ node, contextNs, comments, options, }) {
    const commentMap = new Map();
    comments.forEach((item) => {
        commentMap.set(item.line, item);
    });
    const unusedComments = new Set(comments);
    const context = {
        commentMap,
        unusedComments,
        options,
        keys: [],
        warnings: [],
    };
    reportGeneral(context, node, contextNs);
    unusedComments.forEach((value) => {
        if (value.type === 'WARNING') {
            context.warnings.push({ line: value.line, warning: value.kind });
        }
        else if (value.type === 'MAGIC_COMMENT' && value.kind === 'key') {
            context.keys.push(keyInfoFromComment(context, value));
        }
        else if (value.type === 'MAGIC_COMMENT' && value.kind === 'ignore') {
            context.warnings.push({ line: value.line, warning: 'W_UNUSED_IGNORE' });
        }
    });
    return { keys: context.keys, warnings: context.warnings };
}
