var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Command, Option } from 'commander';
import ansi from 'ansi-colors';
import { extractKeysOfFiles, filterExtractionResult, } from '../../extractor/runner.js';
import { dumpWarnings } from '../../extractor/warnings.js';
import { compareKeys, printKey } from './syncUtils.js';
import { prepareDir } from '../../utils/prepareDir.js';
import { unzipBuffer } from '../../utils/zip.js';
import { askBoolean } from '../../utils/ask.js';
import { loading, exitWithError } from '../../utils/logger.js';
import { handleLoadableError, } from '../../client/TolgeeClient.js';
function backup(client, dest) {
    return __awaiter(this, void 0, void 0, function* () {
        const loadable = yield client.export.export({
            format: 'JSON',
            supportArrays: false,
            filterState: ['UNTRANSLATED', 'TRANSLATED', 'REVIEWED'],
            structureDelimiter: '',
            escapeHtml: false,
        });
        handleLoadableError(loadable);
        const blob = loadable.data;
        yield unzipBuffer(blob, dest);
    });
}
function askForConfirmation(keys, operation) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!process.stdout.isTTY) {
            exitWithError('You must run this command interactively, or specify --yes to proceed.');
        }
        const str = `The following keys will be ${operation}:`;
        console.log(operation === 'created' ? ansi.bold.green(str) : ansi.bold.red(str));
        keys.forEach((k) => printKey(k, operation === 'deleted'));
        const shouldContinue = yield askBoolean('Does this look correct?', true);
        if (!shouldContinue) {
            exitWithError('Aborting.');
        }
    });
}
const syncHandler = (config) => function () {
    return __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c, _d, _e, _f;
        const opts = this.optsWithGlobals();
        const rawKeys = yield loading('Analyzing code...', extractKeysOfFiles(opts));
        const warnCount = dumpWarnings(rawKeys);
        if (!opts.continueOnWarning && warnCount) {
            console.log(ansi.bold.red('Aborting as warnings have been emitted.'));
            process.exit(1);
        }
        const localKeys = filterExtractionResult(rawKeys);
        if ((_a = opts.namespaces) === null || _a === void 0 ? void 0 : _a.length) {
            for (const namespace of Object.keys(localKeys)) {
                if (!((_b = opts.namespaces) === null || _b === void 0 ? void 0 : _b.includes(namespace))) {
                    localKeys[namespace].clear();
                }
            }
        }
        const allKeysLoadable = yield opts.client.GET('/v2/projects/{projectId}/all-keys', {
            params: { path: { projectId: opts.client.getProjectId() } },
        });
        handleLoadableError(allKeysLoadable);
        let remoteKeys = (_e = (_d = (_c = allKeysLoadable.data) === null || _c === void 0 ? void 0 : _c._embedded) === null || _d === void 0 ? void 0 : _d.keys) !== null && _e !== void 0 ? _e : [];
        if ((_f = opts.namespaces) === null || _f === void 0 ? void 0 : _f.length) {
            remoteKeys = remoteKeys.filter((key) => {
                var _a, _b;
                return (_a = opts.namespaces) === null || _a === void 0 ? void 0 : _a.includes((_b = key.namespace) !== null && _b !== void 0 ? _b : '');
            });
        }
        const diff = compareKeys(localKeys, remoteKeys);
        if (!diff.added.length && !diff.removed.length) {
            console.log(ansi.green('Your code project is in sync with the associated Tolgee project!'));
            process.exit(0);
        }
        // Load project settings. We're interested in the default locale here.
        const projectLoadable = yield opts.client.GET('/v2/projects/{projectId}', {
            params: { path: { projectId: opts.client.getProjectId() } },
        });
        handleLoadableError(projectLoadable);
        const baseLanguage = projectLoadable.data.baseLanguage;
        if (!baseLanguage) {
            // I'm highly unsure how we could reach this state, but this is what the OAI spec tells me ¯\_(ツ)_/¯
            exitWithError('Your project does not have a base language!');
        }
        // Prepare backup
        if (opts.backup) {
            yield prepareDir(opts.backup, opts.yes);
            yield loading('Backing up Tolgee project', backup(opts.client, opts.backup));
        }
        // Create new keys
        if (diff.added.length) {
            if (!opts.yes) {
                yield askForConfirmation(diff.added, 'created');
            }
            const keys = diff.added.map((key) => ({
                name: key.keyName,
                namespace: key.namespace,
                translations: key.defaultValue
                    ? { [baseLanguage.tag]: key.defaultValue }
                    : {},
                tags: opts.tagNewKeys,
            }));
            const loadable = yield loading('Creating missing keys...', opts.client.POST('/v2/projects/{projectId}/keys/import', {
                params: { path: { projectId: opts.client.getProjectId() } },
                body: { keys },
            }));
            handleLoadableError(loadable);
        }
        if (opts.removeUnused) {
            // Delete unused keys.
            if (diff.removed.length) {
                if (!opts.yes) {
                    yield askForConfirmation(diff.removed, 'deleted');
                }
                const ids = yield diff.removed.map((k) => k.id);
                const loadable = yield loading('Deleting unused keys...', opts.client.DELETE('/v2/projects/{projectId}/keys', {
                    params: { path: { projectId: opts.client.getProjectId() } },
                    body: { ids },
                }));
                handleLoadableError(loadable);
            }
        }
        console.log(ansi.bold.green('Sync complete!'));
        console.log(ansi.green(`+ ${diff.added.length} string${diff.added.length === 1 ? '' : 's'}`));
        if (opts.removeUnused) {
            console.log(ansi.red(`- ${diff.removed.length} string${diff.removed.length === 1 ? '' : 's'}`));
        }
        else {
            console.log(ansi.italic(`${diff.removed.length} unused key${diff.removed.length === 1 ? '' : 's'} could be deleted.`));
        }
        if (opts.backup) {
            console.log(ansi.blueBright(`A backup of the project prior to the synchronization has been dumped in ${opts.backup}.`));
        }
    });
};
export default (config) => {
    var _a, _b, _c, _d, _e, _f, _g;
    return new Command()
        .name('sync')
        .description('Synchronizes the keys in your code project and in the Tolgee project, by creating missing keys and optionally deleting unused ones. For a dry-run, use `tolgee compare`.')
        .addOption(new Option('-B, --backup <path>', 'Store translation files backup (only translation files, not states, comments, tags, etc.). If something goes wrong, the backup can be used to restore the project to its previous state.').default((_b = (_a = config.sync) === null || _a === void 0 ? void 0 : _a.backup) !== null && _b !== void 0 ? _b : false))
        .addOption(new Option('--continue-on-warning', 'Set this flag to continue the sync if warnings are detected during string extraction. By default, as warnings may indicate an invalid extraction, the CLI will abort the sync.').default((_d = (_c = config.sync) === null || _c === void 0 ? void 0 : _c.continueOnWarning) !== null && _d !== void 0 ? _d : false))
        .addOption(new Option('-n, --namespaces <namespaces...>', 'Specifies which namespaces should be synchronized.').default((_e = config.sync) === null || _e === void 0 ? void 0 : _e.namespaces))
        .addOption(new Option('-Y, --yes', 'Skip prompts and automatically say yes to them. You will not be asked for confirmation before creating/deleting keys.').default(false))
        .addOption(new Option('--remove-unused', 'Delete unused keys from the Tolgee project (within selected namespaces if specified).').default((_g = (_f = config.sync) === null || _f === void 0 ? void 0 : _f.removeUnused) !== null && _g !== void 0 ? _g : false))
        .option('--tag-new-keys <tags...>', 'Specify tags that will be added to newly created keys.')
        .action(syncHandler(config));
};
