"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PhraseyLogger = void 0;
const picocolors_1 = __importDefault(require("picocolors"));
const util_1 = require("util");
const utils_1 = require("./utils");
class PhraseyLogger {
    options;
    constructor(options) {
        this.options = options;
    }
    success(text) {
        this.log("success", text);
    }
    info(text) {
        this.log("info", text);
    }
    warn(text) {
        this.log("warn", text);
    }
    error(text, error) {
        this.log("error", text);
        if (!error)
            return;
        this.logErrors(error);
    }
    logErrors(errors) {
        const errorsArray = Array.isArray(errors) ? errors : [errors];
        errorsArray.forEach((x) => {
            const raw = (0, util_1.inspect)(x, {
                colors: false,
            });
            const pretty = raw
                .split(/\s*{\s*\[cause\]:\s*/g)
                .map((x, i) => {
                const pretty = x
                    .replaceAll(/^\s*}\s*$/gm, "")
                    .replace(/^\s+/gm, utils_1.PhraseyTreeLike.tab(i))
                    .trim();
                return utils_1.PhraseyTreeLike.build(pretty, {
                    prefix: utils_1.PhraseyTreeLike.tab(i),
                });
            })
                .join("\n");
            this.writeError(picocolors_1.default.red(pretty));
        });
    }
    grayed(text) {
        this.write(picocolors_1.default.gray(text));
    }
    ln() {
        this.write("");
    }
    log(level, text) {
        this.write(`${this._prefix(level)} ${text}`);
    }
    write(text) {
        this.options.write(text);
    }
    writeError(error) {
        this.options.writeError(error);
    }
    inherit(tag) {
        const log = new PhraseyLogger({
            tag,
            write: this.options.write,
            writeError: this.options.writeError,
        });
        return log;
    }
    _level(level) {
        switch (level) {
            case "success":
                return picocolors_1.default.green("[success]");
            case "info":
                return picocolors_1.default.cyan("[info]");
            case "warn":
                return picocolors_1.default.yellow("[warn]");
            case "error":
                return picocolors_1.default.red("[error]");
        }
    }
    _prefix(level) {
        const tag = this.options.tag
            ? ` ${picocolors_1.default.magenta(`(${this.options.tag})`)}`
            : "";
        return `${this._level(level)}${tag}`;
    }
    static console(tag) {
        const log = new PhraseyLogger({
            tag,
            write: (text) => console.log(text),
            writeError: (error) => console.error(error),
        });
        return log;
    }
}
exports.PhraseyLogger = PhraseyLogger;
