"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.symbolPostMap = exports.SummaryCommand = void 0;
/* eslint-disable no-case-declarations */
const commander_1 = require("commander");
const fs_extra_1 = require("fs-extra");
const path_1 = __importDefault(require("path"));
const __1 = require("../../");
const log_1 = require("../utils/log");
const parseConfigOptions_1 = require("../utils/parseConfigOptions");
exports.SummaryCommand = new commander_1.Command()
    .name("summary")
    .description("Build the project.")
    .addOption(parseConfigOptions_1.PhraseyCliConfigOptionFlags.configFile)
    .addOption(parseConfigOptions_1.PhraseyCliConfigOptionFlags.configFormat)
    .action(async (options) => {
    const outputFile = options.outputFile;
    const outputFormat = options.outputFormat ?? "console";
    const disableOutputPrint = options.disableOutputPrint || true;
    const configOptions = (0, parseConfigOptions_1.parsePhraseyCliConfigOptions)(options);
    const result = await __1.PhraseyBuilder.summary({
        phrasey: {
            cwd: configOptions.cwd,
            log: log_1.log,
            source: "summary",
        },
        builder: {
            config: configOptions.config,
        },
    });
    if (!result.success) {
        log_1.log.error("Summary failed.");
        log_1.log.logErrors(result.error);
        log_1.log.ln();
        process.exit(1);
    }
    const summary = result.data;
    const data = summary.json();
    let serialized = "";
    switch (outputFormat) {
        case "console":
            log_1.log.write(log_1.pico.bold("Full"));
            printFullStatsTree(data.full);
            log_1.log.ln();
            log_1.log.write(log_1.pico.bold("Individual"));
            Object.entries(data.individual).forEach(([k, v]) => {
                const { isBuildable } = summary.individualStats.get(k);
                const titleTree = __1.PhraseyTreeLike.build([log_1.pico.bold(isBuildable ? log_1.pico.green(k) : log_1.pico.red(k))], { symbolPostMap: exports.symbolPostMap });
                log_1.log.write(titleTree);
                printIndividualStatsTree(v);
            });
            log_1.log.ln();
            if (outputFile) {
                log_1.log.error("Default output format does not support saving to a file.");
                log_1.log.ln();
                process.exit(1);
            }
            break;
        default:
            const formatter = __1.PhraseyContentFormats.resolve(outputFormat);
            const serializedResult = (0, __1.PhraseySafeRun)(() => {
                serialized = formatter.serialize(data);
            });
            if (!serializedResult.success) {
                log_1.log.error("Serializing summary failed due to error.");
                log_1.log.logErrors(serializedResult.error);
                log_1.log.ln();
                process.exit(1);
            }
            if (!disableOutputPrint) {
                log_1.log.write(serialized);
                log_1.log.ln();
            }
    }
    if (outputFile) {
        const outputFilePath = path_1.default.resolve(process.cwd(), outputFile);
        await (0, fs_extra_1.ensureFile)(outputFilePath);
        await (0, fs_extra_1.writeFile)(outputFilePath, serialized);
        log_1.log.success(`Generated "${outputFilePath}".`);
        log_1.log.ln();
    }
});
const symbolPostMap = (symbol) => log_1.pico.gray(symbol);
exports.symbolPostMap = symbolPostMap;
function printFullStatsTree(stats) {
    const data = [
        `Set      : ${stats.setCount}`,
        `Fallback : ${stats.fallbackCount}`,
        `Unset    : ${stats.unsetCount}`,
        `Total    : ${stats.total}`,
        `Keys     : ${stats.keysCount}`,
    ];
    const tree = __1.PhraseyTreeLike.build(data, {
        symbolPostMap: exports.symbolPostMap,
    });
    log_1.log.write(tree);
}
function printIndividualStatsTree(stats) {
    const p = (value) => `${value.toPrecision(3)}%`;
    const data = [
        `Set      : ${stats.set.count} (${p(stats.set.percent)})`,
        `Fallback : ${stats.fallback.count} (${p(stats.fallback.percent)})`,
        `Unset    : ${stats.unset.count} (${p(stats.unset.percent)})`,
        `Total    : ${stats.total}`,
    ];
    const tree = __1.PhraseyTreeLike.build(data, {
        prefix: __1.PhraseyTreeLike.tab(1),
        symbolPostMap: exports.symbolPostMap,
    });
    log_1.log.write(tree);
}
