/// <reference types="node" />
import type { Readable, Writable } from "node:stream";
/**
 * Environment required for several Taplo functions.
 *
 * This is required because WebAssembly is not self-contained and is sand-boxed.
 */
export interface Environment {
    /**
     * Return the current date.
     */
    now: () => Date;
    /**
     * Return the environment variable, if any.
     */
    envVar: (name: string) => string | undefined;
    /**
     * Return all environment variables as `[key, value]` pairs.
     */
    envVars: () => Array<[string, string]>;
    /**
     * Return whether the standard error output is a tty or not.
     */
    stdErrAtty: () => boolean;
    /**
     * Read `n` bytes from the standard input.
     *
     * If the returned array is empty, EOF is reached.
     *
     * This function must not return more than `n` bytes.
     */
    stdin: Readable | ((n: bigint) => Promise<Uint8Array>);
    /**
     * Write the given bytes to the standard output returning
     * the number of bytes written.
     */
    stdout: Writable | ((bytes: Uint8Array) => Promise<number>);
    /**
     * Write the given bytes to the standard error output returning
     * the number of bytes written.
     */
    stderr: Writable | ((bytes: Uint8Array) => Promise<number>);
    /**
     * Search a glob file pattern and return the matched files.
     */
    glob: (pattern: string) => Array<string>;
    /**
     * Read the contents of the file at the given path.
     */
    readFile: (path: string) => Promise<Uint8Array>;
    /**
     * Write and overwrite a file at the given path.
     */
    writeFile: (path: string, bytes: Uint8Array) => Promise<void>;
    /**
     * Turn an URL into a file path.
     */
    urlToFilePath: (url: string) => string;
    /**
     * Return whether a path is absolute.
     */
    isAbsolute: (path: string) => boolean;
    /**
     * Return the path to the current working directory.
     */
    cwd: () => string;
    /**
     * Find the Taplo config file from the given directory
     * and return the path if found.
     *
     * The following files should be searched in order from the given root:
     *
     * - `.taplo.toml`
     * - `taplo.toml`
     */
    findConfigFile: (from: string) => string | undefined;
    /**
     * The fetch function if it is not defined on the global Window.
     *
     * This is required for environments like NodeJs where the fetch API is not available,
     * so a package like `node-fetch` must be used instead.
     *
     */
    fetch?: {
        fetch: any;
        Headers: any;
        Request: any;
        Response: any;
    };
}
/**
 * @private
 */
export declare function prepareEnv(environment: Environment): void;
/**
 * @private
 */
export declare function convertEnv(env: Environment): any;
