package io.github.sspanak.tt9.languages;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import io.github.sspanak.tt9.languages.exceptions.InvalidLanguageCharactersException;
import io.github.sspanak.tt9.util.Text;
import io.github.sspanak.tt9.util.chars.Characters;


public class NaturalLanguage extends TranscribedLanguage {
	protected final ArrayList<ArrayList<String>> layout = new ArrayList<>();
	private final HashMap<Character, String> characterKeyMap = new HashMap<>();
	@NonNull private HashMap<Integer, String> numerals = new HashMap<>();


	public static NaturalLanguage fromDefinition(LanguageDefinition definition) throws Exception {
		if (definition.dictionaryFile.isEmpty()) {
			throw new Exception("Invalid definition. Dictionary file must be set.");
		}

		NaturalLanguage lang = new NaturalLanguage();
		lang.abcString = definition.abcString.isEmpty() ? null : definition.abcString;
		lang.currency = definition.currency;
		lang.dictionaryFile = definition.getDictionaryFile();
		lang.hasABC = definition.hasABC;
		lang.hasSpaceBetweenWords = definition.hasSpaceBetweenWords;
		lang.hasUpperCase = definition.hasUpperCase;
		lang.hasTranscriptionsEmbedded = definition.filterBySounds;
		lang.iconABC = definition.iconABC;
		lang.iconT9 = definition.iconT9;
		lang.isTranscribed = definition.isTranscribed;
		lang.name = definition.name.isEmpty() ? lang.name : definition.name;
		lang.numerals = definition.numerals;
		lang.setLocale(definition);
		lang.setLayout(definition);

		return lang;
	}


	private void setLocale(LanguageDefinition definition) throws Exception {
		if (definition.locale.isEmpty()) {
			throw new Exception("Invalid definition. Locale cannot be empty.");
		}

		if (definition.locale.equals("en")) {
			locale = Locale.ENGLISH;
		} else {
			String[] parts = definition.locale.split("-", 2);
			if (parts.length == 2) {
				locale = new Locale(parts[0], parts[1]);
			} else if (parts.length == 1) {
				locale = new Locale(parts[0]);
			} else {
				throw new Exception("Unrecognized locale format: '" + definition.locale + "'.");
			}
		}
	}


	private void setLayout(LanguageDefinition definition) {
		for (int key = 0; key <= 9 && key < definition.layout.size(); key++) {
				layout.add(
					key,
					key > 1 ? definition.layout.get(key) : generateSpecialChars(definition.layout.get(key))
				);
		}

		generateCharacterKeyMap();
	}


	private ArrayList<String> generateSpecialChars(ArrayList<String> definitionChars) {
		final String SPECIAL_CHARS_PLACEHOLDER = "SPECIAL";
		final String PUNCTUATION_PLACEHOLDER = "PUNCTUATION";

		final Map<String, List<String>> specialChars = new HashMap<>();
		specialChars.put(SPECIAL_CHARS_PLACEHOLDER, new ArrayList<>(Characters.Special));
		specialChars.put(PUNCTUATION_PLACEHOLDER, Characters.PunctuationEnglish);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_AR", Characters.PunctuationArabic);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_BP", Characters.PunctuationChineseBopomofo);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_ZH", Characters.PunctuationChinese);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_FA", Characters.PunctuationFarsi);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_FR", Characters.PunctuationFrench);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_DE", Characters.PunctuationGerman);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_GR", Characters.PunctuationGreek);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_IE", Characters.PunctuationIrish);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_IN", Characters.PunctuationIndic);
		specialChars.put(PUNCTUATION_PLACEHOLDER + "_KR", Characters.PunctuationKorean);

		ArrayList<String> keyChars = new ArrayList<>();
		for (String defChar : definitionChars) {
			List<String> keySpecialChars = specialChars.containsKey(defChar) ? specialChars.get(defChar) : null;
			if (keySpecialChars != null) {
				keyChars.addAll(keySpecialChars);
			} else {
				keyChars.add(defChar);
			}
		}

		return keyChars;
	}


	/**
	 * generateId
	 * Uses the letters of the Locale to generate an ID for the language.
	 * Each letter is converted to uppercase and used as a 5-bit integer. Then the 5-bits
	 * are packed to form a 10-bit or a 20-bit integer, depending on the Locale length.
	 *
	 * Example (2-letter Locale)
	 * 	"en"
	 * 	-> "E" | "N"
	 * 	-> 5 | 448 (shift the 2nd number by 5 bits, so its bits would not overlap with the 1st one)
	 *	-> 543
	 *
	 * Example (4-letter Locale)
	 * 	"bg-BG"
	 * 	-> "B" | "G" | "B" | "G"
	 * 	-> 2 | 224 | 2048 | 229376 (shift each 5-bit number, not overlap with the previous ones)
	 *	-> 231650
	 *
	 * Minimum ID: "aa" -> 33
	 * Maximum ID: "zz-ZZ" -> 879450
	 */
	@Override
	public int getId() {
		if (id == 0) {
			String idString = new LocaleCompat(locale).toString();
			for (int i = 0; i < idString.length(); i++) {
				id |= (idString.codePointAt(i) & 31) << (i * 5);
			}
		}

		return id;
	}

	@Override
	protected String getSortingId() {
		if (isTranscribed) {
			return super.getSortingId();
		}

		if ("IN".equals(getLocale().getCountry()) && "en".equals(getLocale().getLanguage())) {
			return "hi";
		}

		return switch (getLocale().getLanguage()) {
			case "fi" -> "su";
			case "sw" -> "ki";
			case "zgh" -> "tam";
			default -> getLocale().toString();
		};
	}


	@NonNull
	@Override
	public String getAbcString() {
		if (abcString == null) {
			ArrayList<String> lettersList = getKeyCharacters(2);

			StringBuilder sb = new StringBuilder();
			for (int i = 0; i < lettersList.size() && i < 3; i++) {
				sb.append(lettersList.get(i));
			}

			abcString = sb.toString();
		}

		return abcString;
	}


	@NonNull
	@Override
	public String getCode() {
		if (code == null) {
			code = new LocaleCompat(locale).getUniqueLanguageCode();
		}

		return code;
	}


	@NonNull
	@Override
	public String getName() {
		if (name == null) {
			name = new Text(this, locale.getDisplayLanguage(locale)).capitalize();
		}

		return name;
	}


	private void generateCharacterKeyMap() {
		characterKeyMap.clear();
		for (int digit = 0; digit <= 9; digit++) {
			characterKeyMap.put(getKeyNumeral(digit).charAt(0), String.valueOf(digit));
			for (String keyChar : getKeyCharacters(digit)) {
				characterKeyMap.put(keyChar.charAt(0), String.valueOf(digit));
			}
		}
	}


	@NonNull
	@Override
	public ArrayList<String> getKeyCharacters(int key) {
		if (key < 0 || key >= layout.size()) {
			return new ArrayList<>();
		}

		return layout.get(key);
	}


	@NonNull
	public String getKeyNumeral(int key) {
		String digit = numerals.containsKey(key) ? numerals.get(key) : null;
		return  digit != null ? digit : super.getKeyNumeral(key);
	}


	@NonNull
	public String getDigitSequenceForWord(String word) throws InvalidLanguageCharactersException {
		StringBuilder sequence = new StringBuilder();
		String lowerCaseWord = word.toLowerCase(locale);

		for (int i = 0; i < lowerCaseWord.length(); i++) {
			char letter = lowerCaseWord.charAt(i);
			if (!characterKeyMap.containsKey(letter)) {
				throw new InvalidLanguageCharactersException(this, "Failed generating digit sequence for word: '" + word);
			}

			sequence.append(characterKeyMap.get(letter));
		}

		return sequence.toString();
	}



	public boolean isValidWord(String word) {
		if (
			word == null
			|| word.isEmpty()
			|| (word.length() == 1 && Character.isDigit(word.charAt(0)))
		) {
			return true;
		}

		if (isTranscribed) {
			return super.isValidWord(word);
		}

		String lowerCaseWord = word.toLowerCase(locale);

		for (int i = 0; i < lowerCaseWord.length(); i++) {
			if (!characterKeyMap.containsKey(lowerCaseWord.charAt(i))) {
				return false;
			}
		}

		return true;
	}
}
