const fs = require("node:fs");

// Load environment variables from process.env
const {
  KEYSTORE_FILE,
  KEYSTORE_STORE_PASSWORD,
  KEYSTORE_KEY_ALIAS,
  KEYSTORE_KEY_PASSWORD,
} = process.env;

// Extract current versionCode from build.gradle.kts
const buildGradlePath = "app/build.gradle.kts";
const buildGradleContents = fs.readFileSync(buildGradlePath, "utf8");
const versionCodeMatch = buildGradleContents.match(/versionCode\s*=\s*(\d+)/);
if (!versionCodeMatch) {
  throw new Error(`versionCode not found in ${buildGradlePath}`);
}
const CURRENT_VERSION_CODE = Number(versionCodeMatch[1]);
const NEXT_VERSION_CODE = CURRENT_VERSION_CODE + 1;

module.exports = {
  plugins: [
    "@semantic-release/commit-analyzer",
    "@semantic-release/release-notes-generator",
    "@semantic-release/changelog",
    [
      "@semantic-release/git",
      {
        assets: [
          // Generated by @semantic-release/changelog
          "CHANGELOG.md",
          // Modified by @semantic-release/exec
          "app/build.gradle.kts",
          // Generated by @semantic-release/exec
          "metadata/en-US/changelogs/*.txt",
        ],
      },
    ],
    [
      "@semantic-release/exec",
      {
        generateNotesCmd: [
          // Update versionCode and versionName in build.gradle.kts
          //
          // NEXT_VERSION_CODE is JavaScript variable but nextRelease.version is LogDash variable
          `sed -i -E "s/versionCode = [0-9]+/versionCode = ${NEXT_VERSION_CODE}/" ${buildGradlePath}`,
          "sed -i -E 's/versionName = .*/versionName = \"${nextRelease.version}\"/' app/build.gradle.kts",
          // Add release notes to F-Droid metadata
          "mkdir -p metadata/en-US/changelogs",
          'echo "${nextRelease.notes}"' +
            ` > metadata/en-US/changelogs/${NEXT_VERSION_CODE}.txt`,
        ].join("\n"),
        prepareCmd: [
          // Build the release APK
          //
          // Use KEYSTORE_* JavaScript variables for signing since
          // @semantic-release/exec does not support environment variables
          //
          // Build is done AFTER commits are made with updated version info
          // so that "reproducible builds" in F-Droid or IzzyOnDroid work correctly
          "./gradlew :app:assembleRelease \\",
          `  -Pandroid.injected.signing.store.file="${KEYSTORE_FILE}" \\`,
          `  -Pandroid.injected.signing.store.password="${KEYSTORE_STORE_PASSWORD}" \\`,
          `  -Pandroid.injected.signing.key.alias="${KEYSTORE_KEY_ALIAS}" \\`,
          `  -Pandroid.injected.signing.key.password="${KEYSTORE_KEY_PASSWORD}"`,
        ].join("\n"),
      },
    ],
    [
      "@semantic-release/github",
      {
        assets: [
          // Upload the release APK to GitHub Releases
          { path: "app/build/outputs/apk/release/*.apk", label: "Android APK" },
        ],
      },
    ],
  ],
};
