package io.github.sds100.keymapper.actions

import android.text.InputType
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import io.github.sds100.keymapper.R
import io.github.sds100.keymapper.actions.pinchscreen.PinchPickCoordinateResult
import io.github.sds100.keymapper.actions.swipescreen.SwipePickCoordinateResult
import io.github.sds100.keymapper.actions.tapscreen.PickCoordinateResult
import io.github.sds100.keymapper.system.camera.CameraLens
import io.github.sds100.keymapper.system.camera.CameraLensUtils
import io.github.sds100.keymapper.system.display.Orientation
import io.github.sds100.keymapper.system.display.OrientationUtils
import io.github.sds100.keymapper.system.intents.ConfigIntentResult
import io.github.sds100.keymapper.system.network.HttpMethod
import io.github.sds100.keymapper.system.volume.DndMode
import io.github.sds100.keymapper.system.volume.DndModeUtils
import io.github.sds100.keymapper.system.volume.RingerMode
import io.github.sds100.keymapper.system.volume.RingerModeUtils
import io.github.sds100.keymapper.system.volume.VolumeStream
import io.github.sds100.keymapper.system.volume.VolumeStreamUtils
import io.github.sds100.keymapper.util.ui.MultiChoiceItem
import io.github.sds100.keymapper.util.ui.NavDestination
import io.github.sds100.keymapper.util.ui.NavigationViewModel
import io.github.sds100.keymapper.util.ui.PopupUi
import io.github.sds100.keymapper.util.ui.PopupViewModel
import io.github.sds100.keymapper.util.ui.ResourceProvider
import io.github.sds100.keymapper.util.ui.navigate
import io.github.sds100.keymapper.util.ui.showPopup
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

/**
 * Created by sds100 on 26/07/2021.
 */

class CreateActionDelegate(
    private val coroutineScope: CoroutineScope,
    private val useCase: CreateActionUseCase,
    popupViewModel: PopupViewModel,
    navigationViewModelImpl: NavigationViewModel,
    resourceProvider: ResourceProvider,
) : ResourceProvider by resourceProvider,
    PopupViewModel by popupViewModel,
    NavigationViewModel by navigationViewModelImpl {

    val actionResult: MutableStateFlow<ActionData?> = MutableStateFlow(null)
    var enableFlashlightActionState: EnableFlashlightActionState? by mutableStateOf(null)
    var changeFlashlightStrengthActionState: ChangeFlashlightStrengthActionState? by mutableStateOf(
        null,
    )

    var httpRequestBottomSheetState: ActionData.HttpRequest? by mutableStateOf(null)

    init {
        coroutineScope.launch {
            useCase.isFlashlightEnabled().collectLatest { enabled ->
                enableFlashlightActionState?.also { state ->
                    enableFlashlightActionState = state.copy(isFlashEnabled = enabled)
                }
            }
        }
    }

    fun onDoneConfigEnableFlashlightClick() {
        enableFlashlightActionState?.also { state ->
            val flashInfo = state.lensData[state.selectedLens] ?: return

            val strengthPercent =
                state.flashStrength
                    .takeIf { it != flashInfo.defaultStrength }
                    ?.let { it.toFloat() / flashInfo.maxStrength }
                    ?.coerceIn(0f, 1f)

            val action = when (state.actionToCreate) {
                ActionId.TOGGLE_FLASHLIGHT -> ActionData.Flashlight.Toggle(
                    state.selectedLens,
                    strengthPercent,
                )

                ActionId.ENABLE_FLASHLIGHT -> ActionData.Flashlight.Enable(
                    state.selectedLens,
                    strengthPercent,
                )

                else -> return
            }

            enableFlashlightActionState = null

            useCase.disableFlashlight()

            actionResult.update { action }
        }
    }

    fun onDoneChangeFlashlightBrightnessClick() {
        changeFlashlightStrengthActionState?.also { state ->
            val flashInfo = state.lensData[state.selectedLens] ?: return

            val strengthPercent =
                (state.flashStrength.toFloat() / flashInfo.maxStrength).coerceIn(-1f, 1f)

            val action = ActionData.Flashlight.ChangeStrength(
                state.selectedLens,
                strengthPercent,
            )

            changeFlashlightStrengthActionState = null

            actionResult.update { action }
        }
    }

    fun onSelectStrength(strength: Int) {
        enableFlashlightActionState?.also { state ->
            enableFlashlightActionState = state.copy(flashStrength = strength)

            if (state.isFlashEnabled) {
                val lensData = state.lensData[state.selectedLens] ?: return

                useCase.setFlashlightBrightness(
                    state.selectedLens,
                    strength / lensData.maxStrength.toFloat(),
                )
            }
        }
    }

    fun onTestFlashlightConfigClick() {
        enableFlashlightActionState?.also { state ->
            val lensData = state.lensData[state.selectedLens] ?: return

            useCase.toggleFlashlight(
                state.selectedLens,
                state.flashStrength / lensData.maxStrength.toFloat(),
            )
        }
    }

    suspend fun editAction(oldData: ActionData) {
        if (!oldData.isEditable()) {
            throw IllegalArgumentException("This action ${oldData.javaClass.name} can't be edited!")
        }

        configAction(oldData.id, oldData)?.let { action ->
            actionResult.update { action }
        }
    }

    suspend fun createAction(id: ActionId) {
        configAction(id)?.let { action ->
            actionResult.update { action }
        }
    }

    private suspend fun configAction(actionId: ActionId, oldData: ActionData? = null): ActionData? {
        when (actionId) {
            ActionId.SWITCH_KEYBOARD -> {
                val inputMethods = useCase.getInputMethods()
                val items = inputMethods.map {
                    it.id to it.label
                }

                val imeId =
                    showPopup("choose_ime", PopupUi.SingleChoice(items)) ?: return null
                val imeName = inputMethods.single { it.id == imeId }.label

                return ActionData.SwitchKeyboard(imeId, imeName)
            }

            ActionId.PLAY_PAUSE_MEDIA_PACKAGE,
            ActionId.PLAY_MEDIA_PACKAGE,
            ActionId.PAUSE_MEDIA_PACKAGE,
            ActionId.NEXT_TRACK_PACKAGE,
            ActionId.PREVIOUS_TRACK_PACKAGE,
            ActionId.FAST_FORWARD_PACKAGE,
            ActionId.REWIND_PACKAGE,
            ActionId.STOP_MEDIA_PACKAGE,
            ActionId.STEP_FORWARD_PACKAGE,
            ActionId.STEP_BACKWARD_PACKAGE,
            -> {
                val packageName =
                    navigate(
                        "choose_app_for_media_action",
                        NavDestination.ChooseApp(allowHiddenApps = true),
                    )
                        ?: return null

                val action = when (actionId) {
                    ActionId.PAUSE_MEDIA_PACKAGE ->
                        ActionData.ControlMediaForApp.Pause(packageName)

                    ActionId.PLAY_MEDIA_PACKAGE ->
                        ActionData.ControlMediaForApp.Play(packageName)

                    ActionId.PLAY_PAUSE_MEDIA_PACKAGE ->
                        ActionData.ControlMediaForApp.PlayPause(packageName)

                    ActionId.NEXT_TRACK_PACKAGE ->
                        ActionData.ControlMediaForApp.NextTrack(packageName)

                    ActionId.PREVIOUS_TRACK_PACKAGE ->
                        ActionData.ControlMediaForApp.PreviousTrack(packageName)

                    ActionId.FAST_FORWARD_PACKAGE ->
                        ActionData.ControlMediaForApp.FastForward(packageName)

                    ActionId.REWIND_PACKAGE ->
                        ActionData.ControlMediaForApp.Rewind(packageName)

                    ActionId.STOP_MEDIA_PACKAGE ->
                        ActionData.ControlMediaForApp.Stop(packageName)

                    ActionId.STEP_FORWARD_PACKAGE ->
                        ActionData.ControlMediaForApp.StepForward(packageName)

                    ActionId.STEP_BACKWARD_PACKAGE ->
                        ActionData.ControlMediaForApp.StepBackward(packageName)

                    else -> throw Exception("don't know how to create action for $actionId")
                }

                return action
            }

            ActionId.VOLUME_UP,
            ActionId.VOLUME_DOWN,
            ActionId.VOLUME_MUTE,
            ActionId.VOLUME_UNMUTE,
            ActionId.VOLUME_TOGGLE_MUTE,
            -> {
                val showVolumeUiId = 0
                val isVolumeUiChecked =
                    when (oldData) {
                        is ActionData.Volume.Up -> oldData.showVolumeUi
                        is ActionData.Volume.Down -> oldData.showVolumeUi
                        is ActionData.Volume.Mute -> oldData.showVolumeUi
                        is ActionData.Volume.UnMute -> oldData.showVolumeUi
                        is ActionData.Volume.ToggleMute -> oldData.showVolumeUi
                        else -> false
                    }

                val dialogItems = listOf(
                    MultiChoiceItem(
                        showVolumeUiId,
                        getString(R.string.flag_show_volume_dialog),
                        isVolumeUiChecked,
                    ),
                )

                val showVolumeUiDialog = PopupUi.MultiChoice(items = dialogItems)

                val chosenFlags = showPopup("show_volume_ui", showVolumeUiDialog) ?: return null

                val showVolumeUi = chosenFlags.contains(showVolumeUiId)

                val action = when (actionId) {
                    ActionId.VOLUME_UP -> ActionData.Volume.Up(showVolumeUi)
                    ActionId.VOLUME_DOWN -> ActionData.Volume.Down(showVolumeUi)
                    ActionId.VOLUME_MUTE -> ActionData.Volume.Mute(showVolumeUi)
                    ActionId.VOLUME_UNMUTE -> ActionData.Volume.UnMute(showVolumeUi)
                    ActionId.VOLUME_TOGGLE_MUTE -> ActionData.Volume.ToggleMute(
                        showVolumeUi,
                    )

                    else -> throw Exception("don't know how to create action for $actionId")
                }

                return action
            }

            ActionId.VOLUME_INCREASE_STREAM,
            ActionId.VOLUME_DECREASE_STREAM,
            -> {
                val showVolumeUiId = 0
                val isVolumeUiChecked = if (oldData is ActionData.Volume.Stream) {
                    oldData.showVolumeUi
                } else {
                    false
                }

                val dialogItems = listOf(
                    MultiChoiceItem(
                        showVolumeUiId,
                        getString(R.string.flag_show_volume_dialog),
                        isVolumeUiChecked,
                    ),
                )

                val showVolumeUiDialog = PopupUi.MultiChoice(items = dialogItems)

                val chosenFlags =
                    showPopup("show_volume_ui", showVolumeUiDialog) ?: return null

                val showVolumeUi = chosenFlags.contains(showVolumeUiId)

                val items = VolumeStream.values()
                    .map { it to getString(VolumeStreamUtils.getLabel(it)) }

                val stream = showPopup("pick_volume_stream", PopupUi.SingleChoice(items))
                    ?: return null

                val action = when (actionId) {
                    ActionId.VOLUME_INCREASE_STREAM ->
                        ActionData.Volume.Stream.Increase(showVolumeUi = showVolumeUi, stream)

                    ActionId.VOLUME_DECREASE_STREAM ->
                        ActionData.Volume.Stream.Decrease(showVolumeUi = showVolumeUi, stream)

                    else -> throw Exception("don't know how to create action for $actionId")
                }

                return action
            }

            ActionId.CHANGE_RINGER_MODE -> {
                val items = RingerMode.values()
                    .map { it to getString(RingerModeUtils.getLabel(it)) }

                val ringerMode =
                    showPopup("pick_ringer_mode", PopupUi.SingleChoice(items))
                        ?: return null

                return ActionData.Volume.SetRingerMode(ringerMode)
            }

            // don't need to show options for disabling do not disturb
            ActionId.TOGGLE_DND_MODE,
            ActionId.ENABLE_DND_MODE,
            -> {
                val items = DndMode.values()
                    .map { it to getString(DndModeUtils.getLabel(it)) }

                val dndMode =
                    showPopup("pick_dnd_mode", PopupUi.SingleChoice(items))
                        ?: return null

                val action = when (actionId) {
                    ActionId.TOGGLE_DND_MODE -> ActionData.DoNotDisturb.Toggle(dndMode)

                    ActionId.ENABLE_DND_MODE -> ActionData.DoNotDisturb.Enable(dndMode)

                    else -> throw Exception("don't know how to create action for $actionId")
                }

                return action
            }

            ActionId.CYCLE_ROTATIONS -> {
                val items = Orientation.values().map { orientation ->
                    val isChecked = if (oldData is ActionData.Rotation.CycleRotations) {
                        oldData.orientations.contains(orientation)
                    } else {
                        false
                    }

                    MultiChoiceItem(
                        orientation,
                        getString(OrientationUtils.getLabel(orientation)),
                        isChecked,
                    )
                }

                val orientations =
                    showPopup("pick_orientations", PopupUi.MultiChoice(items)) ?: return null

                return ActionData.Rotation.CycleRotations(orientations)
            }

            ActionId.TOGGLE_FLASHLIGHT, ActionId.ENABLE_FLASHLIGHT -> {
                val lenses = useCase.getFlashlightLenses()
                val selectedLens = if (oldData is ActionData.Flashlight) {
                    oldData.lens
                } else if (lenses.contains(CameraLens.BACK)) {
                    CameraLens.BACK
                } else {
                    lenses.first()
                }

                val lensData = lenses.associateWith { useCase.getFlashInfo(it)!! }
                val lensInfo = lensData[selectedLens] ?: lensData.values.first()

                val strength: Int = when (oldData) {
                    is ActionData.Flashlight.Toggle -> if (oldData.strengthPercent == null) {
                        lensInfo.defaultStrength
                    } else {
                        (oldData.strengthPercent * lensInfo.maxStrength).toInt()
                    }

                    is ActionData.Flashlight.Enable -> if (oldData.strengthPercent == null) {
                        lensInfo.defaultStrength
                    } else {
                        (oldData.strengthPercent * lensInfo.maxStrength).toInt()
                    }

                    else -> lensInfo.defaultStrength
                }

                enableFlashlightActionState = EnableFlashlightActionState(
                    actionToCreate = actionId,
                    selectedLens = selectedLens,
                    lensData = lensData,
                    flashStrength = strength,
                    isFlashEnabled = useCase.isFlashlightEnabled().first(),
                )

                return null
            }

            ActionId.CHANGE_FLASHLIGHT_STRENGTH -> {
                val lenses = useCase.getFlashlightLenses()
                val selectedLens = if (oldData is ActionData.Flashlight.ChangeStrength) {
                    oldData.lens
                } else if (lenses.contains(CameraLens.BACK)) {
                    CameraLens.BACK
                } else {
                    lenses.first()
                }

                val lensData = lenses.associateWith { useCase.getFlashInfo(it)!! }
                val lensInfo = lensData[selectedLens] ?: lensData.values.first()

                val strength: Int = when (oldData) {
                    is ActionData.Flashlight.ChangeStrength -> {
                        (oldData.percent * lensInfo.maxStrength).toInt()
                    }

                    else -> (0.1f * lensInfo.maxStrength).toInt()
                }

                changeFlashlightStrengthActionState = ChangeFlashlightStrengthActionState(
                    selectedLens = selectedLens,
                    lensData = lensData,
                    flashStrength = strength,
                )

                return null
            }

            ActionId.DISABLE_FLASHLIGHT,
            -> {
                val items = useCase.getFlashlightLenses().map {
                    it to getString(CameraLensUtils.getLabel(it))
                }

                if (items.size == 1) {
                    return ActionData.Flashlight.Disable(items.first().first)
                } else {
                    val lens = showPopup("pick_lens", PopupUi.SingleChoice(items))
                        ?: return null

                    return ActionData.Flashlight.Disable(lens)
                }
            }

            ActionId.APP -> {
                val packageName =
                    navigate(
                        "choose_app_for_app_action",
                        NavDestination.ChooseApp(allowHiddenApps = false),
                    )
                        ?: return null

                return ActionData.App(packageName)
            }

            ActionId.APP_SHORTCUT -> {
                val appShortcutResult =
                    navigate("choose_app_shortcut", NavDestination.ChooseAppShortcut)
                        ?: return null

                return ActionData.AppShortcut(
                    appShortcutResult.packageName,
                    appShortcutResult.shortcutName,
                    appShortcutResult.uri,
                )
            }

            ActionId.KEY_CODE -> {
                val keyCode =
                    navigate("choose_key_code", NavDestination.ChooseKeyCode)
                        ?: return null

                return ActionData.InputKeyEvent(keyCode = keyCode)
            }

            ActionId.KEY_EVENT -> {
                val keyEventAction = if (oldData is ActionData.InputKeyEvent) {
                    navigate("config_key_event", NavDestination.ConfigKeyEventAction(oldData))
                } else {
                    navigate("config_key_event", NavDestination.ConfigKeyEventAction())
                }

                return keyEventAction
            }

            ActionId.TAP_SCREEN -> {
                val oldResult = if (oldData is ActionData.TapScreen) {
                    PickCoordinateResult(
                        oldData.x,
                        oldData.y,
                        oldData.description ?: "",
                    )
                } else {
                    null
                }

                val result = navigate(
                    "pick_display_coordinate_for_action",
                    NavDestination.PickCoordinate(oldResult),
                ) ?: return null

                val description = if (result.description.isEmpty()) {
                    null
                } else {
                    result.description
                }

                return ActionData.TapScreen(
                    result.x,
                    result.y,
                    description,
                )
            }

            ActionId.SWIPE_SCREEN -> {
                val oldResult = if (oldData is ActionData.SwipeScreen) {
                    SwipePickCoordinateResult(
                        oldData.xStart,
                        oldData.yStart,
                        oldData.xEnd,
                        oldData.yEnd,
                        oldData.fingerCount,
                        oldData.duration,
                        oldData.description ?: "",
                    )
                } else {
                    null
                }

                val result = navigate(
                    "pick_swipe_coordinate_for_action",
                    NavDestination.PickSwipeCoordinate(oldResult),
                ) ?: return null

                val description = if (result.description.isEmpty()) {
                    null
                } else {
                    result.description
                }

                return ActionData.SwipeScreen(
                    result.xStart,
                    result.yStart,
                    result.xEnd,
                    result.yEnd,
                    result.fingerCount,
                    result.duration,
                    description,
                )
            }

            ActionId.PINCH_SCREEN -> {
                val oldResult = if (oldData is ActionData.PinchScreen) {
                    PinchPickCoordinateResult(
                        oldData.x,
                        oldData.y,
                        oldData.distance,
                        oldData.pinchType,
                        oldData.fingerCount,
                        oldData.duration,
                        oldData.description ?: "",
                    )
                } else {
                    null
                }

                val result = navigate(
                    "pick_pinch_coordinate_for_action",
                    NavDestination.PickPinchCoordinate(oldResult),
                ) ?: return null

                val description = if (result.description.isEmpty()) {
                    null
                } else {
                    result.description
                }

                return ActionData.PinchScreen(
                    result.x,
                    result.y,
                    result.distance,
                    result.pinchType,
                    result.fingerCount,
                    result.duration,
                    description,
                )
            }

            ActionId.TEXT -> {
                val oldText = if (oldData is ActionData.Text) {
                    oldData.text
                } else {
                    ""
                }

                val text = showPopup(
                    "create_text_action",
                    PopupUi.Text(
                        hint = getString(R.string.hint_create_text_action),
                        allowEmpty = false,
                        text = oldText,
                    ),
                ) ?: return null

                return ActionData.Text(text)
            }

            ActionId.URL -> {
                val oldUrl = if (oldData is ActionData.Url) {
                    oldData.url
                } else {
                    ""
                }

                val text = showPopup(
                    "create_url_action",
                    PopupUi.Text(
                        hint = getString(R.string.hint_create_url_action),
                        allowEmpty = false,
                        inputType = InputType.TYPE_TEXT_VARIATION_URI,
                        text = oldUrl,
                    ),
                ) ?: return null

                return ActionData.Url(text)
            }

            ActionId.INTENT -> {
                val oldIntent = if (oldData is ActionData.Intent) {
                    ConfigIntentResult(
                        oldData.uri,
                        oldData.target,
                        oldData.description,
                        oldData.extras,
                    )
                } else {
                    null
                }

                val result = navigate(
                    "config_intent",
                    NavDestination.ConfigIntent(oldIntent),
                ) ?: return null

                return ActionData.Intent(
                    description = result.description,
                    target = result.target,
                    uri = result.uri,
                    extras = result.extras,
                )
            }

            ActionId.PHONE_CALL -> {
                val oldText = if (oldData is ActionData.PhoneCall) {
                    oldData.number
                } else {
                    ""
                }

                val text = showPopup(
                    "create_phone_call_action",
                    PopupUi.Text(
                        hint = getString(R.string.hint_create_phone_call_action),
                        allowEmpty = false,
                        inputType = InputType.TYPE_CLASS_PHONE,
                        text = oldText,
                    ),
                ) ?: return null

                return ActionData.PhoneCall(text)
            }

            ActionId.SOUND -> {
                return navigate(
                    "choose_sound_file",
                    NavDestination.ChooseSound,
                )
            }

            ActionId.TOGGLE_WIFI -> return ActionData.Wifi.Toggle
            ActionId.ENABLE_WIFI -> return ActionData.Wifi.Enable
            ActionId.DISABLE_WIFI -> return ActionData.Wifi.Disable

            ActionId.TOGGLE_BLUETOOTH -> return ActionData.Bluetooth.Toggle
            ActionId.ENABLE_BLUETOOTH -> return ActionData.Bluetooth.Enable
            ActionId.DISABLE_BLUETOOTH -> return ActionData.Bluetooth.Disable

            ActionId.TOGGLE_MOBILE_DATA -> return ActionData.MobileData.Toggle
            ActionId.ENABLE_MOBILE_DATA -> return ActionData.MobileData.Enable
            ActionId.DISABLE_MOBILE_DATA -> return ActionData.MobileData.Disable

            ActionId.TOGGLE_AUTO_BRIGHTNESS -> return ActionData.Brightness.ToggleAuto
            ActionId.DISABLE_AUTO_BRIGHTNESS -> return ActionData.Brightness.DisableAuto
            ActionId.ENABLE_AUTO_BRIGHTNESS -> return ActionData.Brightness.EnableAuto
            ActionId.INCREASE_BRIGHTNESS -> return ActionData.Brightness.Increase
            ActionId.DECREASE_BRIGHTNESS -> return ActionData.Brightness.Decrease

            ActionId.TOGGLE_AUTO_ROTATE -> return ActionData.Rotation.ToggleAuto
            ActionId.ENABLE_AUTO_ROTATE -> return ActionData.Rotation.EnableAuto
            ActionId.DISABLE_AUTO_ROTATE -> return ActionData.Rotation.DisableAuto
            ActionId.PORTRAIT_MODE -> return ActionData.Rotation.Portrait
            ActionId.LANDSCAPE_MODE -> return ActionData.Rotation.Landscape
            ActionId.SWITCH_ORIENTATION -> return ActionData.Rotation.SwitchOrientation

            ActionId.VOLUME_SHOW_DIALOG -> return ActionData.Volume.ShowDialog
            ActionId.CYCLE_RINGER_MODE -> return ActionData.Volume.CycleRingerMode
            ActionId.CYCLE_VIBRATE_RING -> return ActionData.Volume.CycleVibrateRing

            ActionId.EXPAND_NOTIFICATION_DRAWER -> return ActionData.StatusBar.ExpandNotifications
            ActionId.TOGGLE_NOTIFICATION_DRAWER -> return ActionData.StatusBar.ToggleNotifications
            ActionId.EXPAND_QUICK_SETTINGS -> return ActionData.StatusBar.ExpandQuickSettings
            ActionId.TOGGLE_QUICK_SETTINGS -> return ActionData.StatusBar.ToggleQuickSettings
            ActionId.COLLAPSE_STATUS_BAR -> return ActionData.StatusBar.Collapse

            ActionId.PAUSE_MEDIA -> return ActionData.ControlMedia.Pause
            ActionId.PLAY_MEDIA -> return ActionData.ControlMedia.Play
            ActionId.PLAY_PAUSE_MEDIA -> return ActionData.ControlMedia.PlayPause
            ActionId.NEXT_TRACK -> return ActionData.ControlMedia.NextTrack
            ActionId.PREVIOUS_TRACK -> return ActionData.ControlMedia.PreviousTrack
            ActionId.FAST_FORWARD -> return ActionData.ControlMedia.FastForward
            ActionId.REWIND -> return ActionData.ControlMedia.Rewind
            ActionId.STOP_MEDIA -> return ActionData.ControlMedia.Stop
            ActionId.STEP_FORWARD -> return ActionData.ControlMedia.StepForward
            ActionId.STEP_BACKWARD -> return ActionData.ControlMedia.StepBackward

            ActionId.GO_BACK -> return ActionData.GoBack
            ActionId.GO_HOME -> return ActionData.GoHome
            ActionId.OPEN_RECENTS -> return ActionData.OpenRecents
            ActionId.TOGGLE_SPLIT_SCREEN -> return ActionData.ToggleSplitScreen
            ActionId.GO_LAST_APP -> return ActionData.GoLastApp
            ActionId.OPEN_MENU -> return ActionData.OpenMenu

            ActionId.ENABLE_NFC -> return ActionData.Nfc.Enable
            ActionId.DISABLE_NFC -> return ActionData.Nfc.Disable
            ActionId.TOGGLE_NFC -> return ActionData.Nfc.Toggle

            ActionId.MOVE_CURSOR_TO_END -> return ActionData.MoveCursorToEnd
            ActionId.TOGGLE_KEYBOARD -> return ActionData.ToggleKeyboard
            ActionId.SHOW_KEYBOARD -> return ActionData.ShowKeyboard
            ActionId.HIDE_KEYBOARD -> return ActionData.HideKeyboard
            ActionId.SHOW_KEYBOARD_PICKER -> return ActionData.ShowKeyboardPicker
            ActionId.TEXT_CUT -> return ActionData.CutText
            ActionId.TEXT_COPY -> return ActionData.CopyText
            ActionId.TEXT_PASTE -> return ActionData.PasteText
            ActionId.SELECT_WORD_AT_CURSOR -> return ActionData.SelectWordAtCursor

            ActionId.TOGGLE_AIRPLANE_MODE -> return ActionData.AirplaneMode.Toggle
            ActionId.ENABLE_AIRPLANE_MODE -> return ActionData.AirplaneMode.Enable
            ActionId.DISABLE_AIRPLANE_MODE -> return ActionData.AirplaneMode.Disable

            ActionId.SCREENSHOT -> return ActionData.Screenshot
            ActionId.OPEN_VOICE_ASSISTANT -> return ActionData.VoiceAssistant
            ActionId.OPEN_DEVICE_ASSISTANT -> return ActionData.DeviceAssistant

            ActionId.OPEN_CAMERA -> return ActionData.OpenCamera
            ActionId.LOCK_DEVICE -> return ActionData.LockDevice
            ActionId.POWER_ON_OFF_DEVICE -> return ActionData.ScreenOnOff
            ActionId.SECURE_LOCK_DEVICE -> return ActionData.SecureLock
            ActionId.CONSUME_KEY_EVENT -> return ActionData.ConsumeKeyEvent
            ActionId.OPEN_SETTINGS -> return ActionData.OpenSettings
            ActionId.SHOW_POWER_MENU -> return ActionData.ShowPowerMenu
            ActionId.DISABLE_DND_MODE -> return ActionData.DoNotDisturb.Disable
            ActionId.DISMISS_MOST_RECENT_NOTIFICATION -> return ActionData.DismissLastNotification
            ActionId.DISMISS_ALL_NOTIFICATIONS -> return ActionData.DismissAllNotifications
            ActionId.ANSWER_PHONE_CALL -> return ActionData.AnswerCall
            ActionId.END_PHONE_CALL -> return ActionData.EndCall
            ActionId.DEVICE_CONTROLS -> return ActionData.DeviceControls
            ActionId.HTTP_REQUEST -> {
                if (oldData == null) {
                    httpRequestBottomSheetState = ActionData.HttpRequest(
                        description = "",
                        method = HttpMethod.GET,
                        url = "http://",
                        body = "",
                        authorizationHeader = "",
                    )
                } else {
                    httpRequestBottomSheetState = oldData as? ActionData.HttpRequest
                }
                return null
            }

            ActionId.INTERACT_UI_ELEMENT -> {
                val oldAction = oldData as? ActionData.InteractUiElement

                return navigate(
                    "config_interact_ui_element_action",
                    NavDestination.InteractUiElement(oldAction),
                )
            }
        }
    }
}
