/*
 * PlayOnDlna - An Android application to play media on dlna devices
 * Copyright (C) 2025 Lukas Scheerer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package io.github.scovillo.playondlna

import android.app.NotificationChannel
import android.app.NotificationManager
import android.content.Intent
import android.net.ConnectivityManager
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.core.content.ContextCompat
import androidx.core.splashscreen.SplashScreen.Companion.installSplashScreen
import androidx.lifecycle.ViewModelProvider
import com.arthenica.ffmpegkit.FFmpegKit
import io.github.scovillo.playondlna.model.DlnaListScreenModel
import io.github.scovillo.playondlna.model.SettingsState
import io.github.scovillo.playondlna.model.VideoJobModel
import io.github.scovillo.playondlna.persistence.SettingsRepository
import io.github.scovillo.playondlna.stream.OkHttpDownloadClient
import io.github.scovillo.playondlna.stream.WebServerService
import io.github.scovillo.playondlna.stream.WifiConnectionState
import io.github.scovillo.playondlna.theme.PlayOnDlnaTheme
import io.github.scovillo.playondlna.ui.DlnaListScreen
import io.github.scovillo.playondlna.ui.MainScreen
import io.github.scovillo.playondlna.ui.PlayScreen
import io.github.scovillo.playondlna.ui.SettingsScreen
import org.schabi.newpipe.extractor.NewPipe

class MainActivity : ComponentActivity() {
    private lateinit var dlnaModel: DlnaListScreenModel
    private lateinit var settingsState: SettingsState
    private lateinit var videoJobModel: VideoJobModel
    private lateinit var wifiConnectionState: WifiConnectionState

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        installSplashScreen()
        NewPipe.init(OkHttpDownloadClient())
        getSystemService(NotificationManager::class.java)
            .createNotificationChannel(
                NotificationChannel(
                    "http_channel",
                    "HTTP Server",
                    NotificationManager.IMPORTANCE_LOW
                )
            )
        ContextCompat.startForegroundService(this, Intent(this, WebServerService::class.java))
        dlnaModel = ViewModelProvider(this)[DlnaListScreenModel::class.java]

        val settingsRepository = SettingsRepository(this)
        wifiConnectionState =
            WifiConnectionState(getSystemService(CONNECTIVITY_SERVICE) as ConnectivityManager)
        this.videoJobModel = VideoJobModel(
            settingsRepository,
            wifiConnectionState
        )
        this.settingsState =
            SettingsState(settingsRepository, onClearCache = { this.clearCache() })
        setContent {
            PlayOnDlnaTheme {
                MainScreen(
                    playScreen = {
                        PlayScreen(videoJobModel) {
                            DlnaListScreen(
                                videoJobModel,
                                dlnaModel
                            )
                        }
                    },
                    settingsScreen = { SettingsScreen(settingsState) }
                )
            }
        }
        handleShareIntent(intent)
    }

    override fun onNewIntent(intent: Intent) {
        super.onNewIntent(intent)
        setIntent(intent)
        handleShareIntent(intent)
    }

    fun clearCache() {
        if (!cacheDir.exists())
            return
        val runningSessions = FFmpegKit.listSessions()
        val currentSession = this.videoJobModel.currentSession.value
        runningSessions.forEach {
            if (currentSession == null || it.sessionId != currentSession.sessionId) {
                Log.i("clearCache", "Cancel FFmpegKit with id ${it.sessionId}")
                FFmpegKit.cancel(it.sessionId)
            }
        }
        val currentVideoFile = this.videoJobModel.currentVideoFile.value
        cacheDir.listFiles()?.forEach { file ->
            if (file.exists() && (currentVideoFile == null || !file.name.contains(currentVideoFile.id))) {
                file.delete()
            }
        }
        Toast.makeText(this, "Cache cleared!", Toast.LENGTH_SHORT).show()
    }

    override fun onDestroy() {
        super.onDestroy()
    }

    private fun handleShareIntent(intent: Intent?) {
        if (intent?.action == Intent.ACTION_SEND) {
            if (intent.type == "text/plain") {
                val url = intent.extras?.getString("android.intent.extra.TEXT")
                if (url != null) {
                    this.videoJobModel.prepareVideo(url, cacheDir)
                }
            }
        }
    }
}