/*
 * Copyright (c) 2025 Sam Crow
 *
 * This file is part of JRBPSurvey.
 *
 * JRBPSurvey is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *
 * JRBPSurvey is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with Foobar.
 * If not, see <https://www.gnu.org/licenses/>.
 */

package org.samcrow.ridgesurvey;

import static org.samcrow.ridgesurvey.map.RouteGraphicsKt.createRouteLayers;
import static org.samcrow.ridgesurvey.map.RouteGraphicsKt.readRoutes;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.graphics.drawable.ColorDrawable;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MenuItem.OnMenuItemClickListener;
import android.view.View;
import android.widget.Toast;

import androidx.activity.result.ActivityResultLauncher;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresPermission;
import androidx.appcompat.app.ActionBar;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import androidx.room.Room;

import com.google.android.material.snackbar.BaseTransientBottomBar;
import com.google.android.material.snackbar.Snackbar;

import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.maplibre.android.MapLibre;
import org.maplibre.android.camera.CameraPosition;
import org.maplibre.android.geometry.LatLngBounds;
import org.maplibre.android.location.LocationComponent;
import org.maplibre.android.location.LocationComponentActivationOptions;
import org.maplibre.android.location.LocationComponentOptions;
import org.maplibre.android.location.engine.LocationEngineRequest;
import org.maplibre.android.location.permissions.PermissionsListener;
import org.maplibre.android.location.permissions.PermissionsManager;
import org.maplibre.android.maps.MapLibreMap;
import org.maplibre.android.maps.MapView;
import org.maplibre.android.maps.Style;
import org.maplibre.android.style.layers.Layer;
import org.samcrow.ridgesurvey.data.Database;
import org.samcrow.ridgesurvey.data.IdentifiedObservation;
import org.samcrow.ridgesurvey.data.UploadTrigger;
import org.samcrow.ridgesurvey.data.ObservationDatabase;
import org.samcrow.ridgesurvey.data.RouteState;
import org.samcrow.ridgesurvey.data.SimpleTimedEvent;
import org.samcrow.ridgesurvey.data.SimpleTimedEventDao;
import org.samcrow.ridgesurvey.data.UploadMenuItemController;
import org.samcrow.ridgesurvey.data.UploadService;
import org.samcrow.ridgesurvey.data.UploadStatusTracker;
import org.samcrow.ridgesurvey.map.RouteLayer;

import java.io.IOException;
import java.util.List;
import java.util.Objects;

public class MainActivity extends AppCompatActivity {

    /**
     * Key corresponding to a RouteState extra that must be provided when launching this activity
     */
    public static final String EXTRA_ROUTE_STATE = MainActivity.class.getName() + ".EXTRA_ROUTE_STATE";
    /**
     * A tag that identifies this class, used for logging and preferences
     */
    private static final String TAG = MainActivity.class.getSimpleName();
    /**
     * The initial position of the map
     */
    public static final LatLngBounds START_POSITION = new LatLngBounds(37.4175457, -122.1919819, 37.3909509, -122.2600484);
    /**
     * The map view
     */
    private MapView mMapView;
    /**
     * The map in the map view
     */
    private MapLibreMap mMap;

    /**
     * An immutable list of all routes, with their sites
     * <p>
     * This is always non-null after {@link #onCreate(Bundle)}.
     */
    private List<Route> mRoutes;
    /**
     * The preferences interface
     */
    private Preferences mPreferences;
    /**
     * The selection manager
     */
    private SelectionManager mSelectionManager;
    /**
     * The upload status tracker
     */
    private UploadStatusTracker mUploadStatusTracker;
    /**
     * The active route / other information
     */
    private RouteState mRouteState;

    private Database mDatabase;
    private RouteLayer mRouteLayer;
    private PermissionsManager mLocationPermissions;
    private UploadTrigger mUploadTrigger;
    private ActivityResultLauncher<DataEntryActivity.Arguments> mDataEntryLauncher;
    private ActivityResultLauncher<IdentifiedObservation> mObservationEditLauncher;
    private ActivityResultLauncher<Void> mObservationListLauncher;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setTitle(getString(R.string.map));

        mRouteState = getIntent().getParcelableExtra(EXTRA_ROUTE_STATE);
        if (mRouteState == null) {
            throw new RuntimeException("Route state extra required");
        }
        // Check that the route state isn't too old
        if (mRouteState.isExpired()) {
            new AlertDialog.Builder(this).setTitle("Verify route")
                    .setMessage("Please go back and start a new route")
                    .setCancelable(false)
                    .setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            // This should go back to the welcome activity
                            finish();
                        }
                    }).show();
            // Don't load the rest of the activity
            return;
        }

        if (mRouteState.isTestMode()) {
            setTitle("Map - Test Mode");
        } else {
            setTitle(String.format("%s - %s", getString(R.string.map), mRouteState.getRouteName()));
        }

        mDataEntryLauncher = registerForActivityResult(new DataEntryActivity.EntryContract(),
                this::onDataEntryClosed);
        mObservationEditLauncher =
                registerForActivityResult(new ObservationEditActivity.EditContract(),
                        this::onDataEntryClosed);
        mObservationListLauncher = registerForActivityResult(new ObservationListActivity.Contract(),
                (unused) -> onObservationListClosed());
        mUploadTrigger = new UploadTrigger(this);

        // Set up map graphics
        MapLibre.getInstance(this);

        setContentView(R.layout.activity_main);

        final ActionBar bar = getSupportActionBar();
        if (bar != null) {
            bar.setDisplayHomeAsUpEnabled(true);
            if (mRouteState.isTestMode()) {
                final int color = getResources().getColor(R.color.testModeToolbar, null);
                bar.setBackgroundDrawable(new ColorDrawable(color));
                final View timerFragment = findViewById(R.id.timer_fragment);
                timerFragment.setBackgroundColor(color);
            }
        }

        mRoutes = readRoutes(this);
        mPreferences = new Preferences(this);
        mSelectionManager = new SelectionManager(mRoutes);
        final @Nullable Integer selectedSiteId = mPreferences.getSelectedSiteId();
        if (selectedSiteId != null) {
            mSelectionManager.setSelectedSiteById(selectedSiteId);
        }

        // Set up upload status tracker
        mUploadStatusTracker = new UploadStatusTracker(this);
        mUploadStatusTracker.addListener(
                findViewById(R.id.upload_status_bar));

        final LocalBroadcastManager manager = LocalBroadcastManager.getInstance(this);
        final IntentFilter filter = new IntentFilter();
        filter.addAction(UploadStatusTracker.ACTION_OBSERVATION_MADE);
        filter.addAction(UploadStatusTracker.ACTION_UPLOAD_STARTED);
        filter.addAction(UploadStatusTracker.ACTION_UPLOAD_SUCCESS);
        filter.addAction(UploadStatusTracker.ACTION_UPLOAD_FAILED);
        manager.registerReceiver(mUploadStatusTracker, filter);

        try {
            setUpMap(savedInstanceState);
        } catch (IOException e) {
            new AlertDialog.Builder(this)
                    .setTitle(R.string.failed_to_load_map)
                    .setMessage(e.getLocalizedMessage())
                    .show();
            Log.e(TAG, "Failed to set up map", e);
        }

        mDatabase = Room.databaseBuilder(getApplicationContext(), Database.class, "events")
                .allowMainThreadQueries()
                .build();

        startUpload();
    }

    @Override
    protected void onStart() {
        super.onStart();
        mMapView.onStart();
    }

    @Override
    protected void onPause() {
        super.onPause();
        mMapView.onPause();
        savePreferences();
    }

    private void savePreferences() {
        final Preferences.Editor editor = mPreferences.edit();
        if (mSelectionManager != null) {
            final Site selectedSite = mSelectionManager.getSelectedSite();
            editor.setSelectedSiteId(selectedSite != null ? selectedSite.getId() : null);
        }
        if (mMap != null) {
            editor.setCamera(mMap.getCameraPosition());
        }
        editor.apply();
    }

    @Override
    protected void onResume() {
        super.onResume();
        mMapView.onResume();
    }

    @Override
    protected void onStop() {
        super.onStop();
        mMapView.onStop();
    }

    @Override
    public void onLowMemory() {
        super.onLowMemory();
        mMapView.onLowMemory();
    }

    @Override
    protected void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        mMapView.onSaveInstanceState(outState);
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions,
                                           @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (mLocationPermissions != null) {
            mLocationPermissions.onRequestPermissionsResult(requestCode, permissions, grantResults);
        }
    }

    /**
     * Sets up the map view in {@link #mMapView}
     */
    @SuppressLint("MissingPermission")
    private void setUpMap(@Nullable Bundle savedInstanceState) throws IOException {
        mMapView = findViewById(R.id.map);
        mMapView.onCreate(savedInstanceState);
        mMapView.getMapAsync(map -> {
            mMap = map;
            mRouteLayer = new RouteLayer(new ObservationDatabase(this), mRoutes, mSelectionManager);
            mSelectionManager.addSelectionListener(mRouteLayer);

            final Style.Builder style = new Style.Builder()
                    .fromUri("asset://map_style.json").withSources(mRouteLayer.getSource());
            for (Layer layer : createRouteLayers(this)) {
                style.withLayerBelow(layer, "all_site_labels");
            }
            map.setStyle(style);

            if (savedInstanceState == null) {
                final CameraPosition savedCamera = mPreferences.getCamera();
                if (savedCamera != null) {
                    map.setCameraPosition(savedCamera);
                } else {
                    final CameraPosition initialCamera =
                            Objects.requireNonNull(map.getCameraForLatLngBounds(START_POSITION));
                    map.setCameraPosition(initialCamera);
                }
            }
            map.getUiSettings().setRotateGesturesEnabled(false);
            map.getUiSettings().setTiltGesturesEnabled(false);
            map.getUiSettings().setAttributionEnabled(false);
            map.getUiSettings().setLogoEnabled(false);
            map.addOnMapClickListener(mSelectionManager);
        });

        if (PermissionsManager.areLocationPermissionsGranted(this)) {
            onLocationPermissionGranted();
        } else {
            mLocationPermissions = new PermissionsManager(new PermissionsListener() {
                @Override
                public void onExplanationNeeded(List<String> permissionsToExplain) {
                    Toast.makeText(MainActivity.this,
                            getString(R.string.request_location_explanation,
                                    getString(R.string.app_name)),
                            Toast.LENGTH_LONG).show();
                }

                @Override
                public void onPermissionResult(boolean granted) {
                    if (granted) {
                        onLocationPermissionGranted();
                    }
                }
            });
            mLocationPermissions.requestLocationPermissions(this);
        }
    }

    @RequiresPermission(allOf = {Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.ACCESS_COARSE_LOCATION})
    private void onLocationPermissionGranted() {
        mMapView.getMapAsync(map -> {
            map.getStyle(style -> {
                final LocationComponent location = map.getLocationComponent();
                location.activateLocationComponent(LocationComponentActivationOptions.builder(this,
                                style)
                        .locationEngineRequest(new LocationEngineRequest.Builder(1000).setPriority(
                                        LocationEngineRequest.PRIORITY_HIGH_ACCURACY)
                                .setFastestInterval(1000)
                                .build())
                        .locationComponentOptions(LocationComponentOptions.builder(this)
                                .accuracyAnimationEnabled(true)
                                .pulseEnabled(true)
                                .build())
                        .locationEngine(new GpsLocationEngine(this))
                        .build());
                location.setLocationComponentEnabled(true);
            });
        });
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        super.onCreateOptionsMenu(menu);
        getMenuInflater().inflate(R.menu.home_menu, menu);

        final MenuItem editItem = menu.findItem(R.id.edit_item);
        editItem.setOnMenuItemClickListener(new OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(@NonNull MenuItem item) {
                final Site selectedSite = mSelectionManager.getSelectedSite();
                final Route selectedSiteRoute = mSelectionManager.getSelectedSiteRoute();
                if (selectedSite != null && selectedSiteRoute != null) {
                    // Look up observations for this site
                    final ObservationDatabase database = new ObservationDatabase(MainActivity.this);
                    final IdentifiedObservation lastObservation = database.getObservationForSite(selectedSite.getId());
                    // If this site has been visited, edit the most recent observation
                    if (lastObservation != null) {
                        mObservationEditLauncher.launch(lastObservation);
                    } else {
                        // Otherwise create a new observation
                        mDataEntryLauncher.launch(new DataEntryActivity.Arguments(selectedSite,
                                selectedSiteRoute,
                                mRouteState));
                    }
                } else {
                    new AlertDialog.Builder(MainActivity.this)
                            .setTitle(R.string.no_site_selected)
                            .setMessage(R.string.select_a_site)
                            .setNeutralButton(android.R.string.ok, null)
                            .show();
                }
                return true;
            }
        });

        final MenuItem viewObservationsItem = menu.findItem(R.id.view_observations_item);
        viewObservationsItem.setOnMenuItemClickListener(new OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(@NonNull MenuItem item) {
                mObservationListLauncher.launch(null);
                return true;
            }
        });

        final MenuItem uploadItem = menu.findItem(R.id.upload_item);
        uploadItem.setOnMenuItemClickListener(new OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(@NonNull MenuItem item) {
                // Start the upload service
                startUpload();
                return true;
            }
        });
        final UploadMenuItemController controller = new UploadMenuItemController(uploadItem);
        if (mUploadStatusTracker != null) {
            mUploadStatusTracker.addListener(controller);
        }

        final MenuItem placeSensorItem = menu.findItem(R.id.home_item_place_sensor);
        initSensorMenuItem(placeSensorItem, "Sensor placement time", "Sensor placed");
        final MenuItem pickUpSensorItem = menu.findItem(R.id.home_item_pick_up_sensor);
        initSensorMenuItem(pickUpSensorItem, "Sensor pickup time", "Sensor picked up");

        final MenuItem viewEventsItem = menu.findItem(R.id.view_events_item);
        viewEventsItem.setOnMenuItemClickListener(new OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(@NonNull MenuItem item) {
                final TimedEventFragment fragment = TimedEventFragment.newInstance();
                fragment.show(getSupportFragmentManager(), "timed events");
                return true;
            }
        });

        return true;
    }

    private void initSensorMenuItem(@NonNull MenuItem item, @NonNull String title, @NonNull String eventName) {
        item.setOnMenuItemClickListener(new OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(@NonNull MenuItem item) {
                final TimePickerDialogFragment fragment = new TimePickerDialogFragment(title);
                fragment.setOnTimePickedListener(new TimePickerDialogFragment.TimePickedListener() {
                    @Override
                    public void onTimePicked(@NonNull TimePickerDialogFragment fragment, @NonNull DateTime selectedDateTime) {
                        final String activeRoute = mRouteState.getRouteName();

                        final SimpleTimedEvent event = new SimpleTimedEvent(selectedDateTime, eventName, activeRoute);
                        final SimpleTimedEventDao dao = mDatabase.simpleTimedEventDao();
                        dao.insert(event);
                        // Upload the new event if possible
                        startUpload();

                        final String timeString = DateTimeFormat.shortTime().print(selectedDateTime);
                        final Snackbar bar = Snackbar.make(mMapView,
                                String.format("Recorded \"%s\" at %s", eventName, timeString),
                                BaseTransientBottomBar.LENGTH_LONG
                        );
                        bar.show();
                    }
                });

                fragment.show(getSupportFragmentManager(), title);
                return true;
            }
        });
    }

    /**
     * The data entry or observation edit activity closed, and we may need to update the map
     */
    private void onDataEntryClosed(boolean createdOrUpdated) {
        if (!createdOrUpdated) {
            return;
        }
        // The data entry activity just returned and an observation was recorded
        // Deselect the site so that the user does not accidentally enter an observation
        // for it after moving to another site
        mSelectionManager.setSelectedSite(null, null);
        mRouteLayer.updateVisitedSites();
    }

    private void onObservationListClosed() {
        // Observation list has closed, clear selection
        mSelectionManager.setSelectedSite(null, null);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        mMapView.onDestroy();
        mUploadTrigger.close();
    }

    /**
     * Starts a service to upload observations
     */
    private void startUpload() {
        startService(new Intent(getApplicationContext(), UploadService.class));
    }

    // Close this activity when the user presses the back button
    @Override
    public boolean onSupportNavigateUp() {
        finish();
        return true;
    }
}
