package share

import (
	"encoding/base64"
	"encoding/json"
	"fmt"
	"net/url"
	"strconv"
	"strings"

	"github.com/xtls/xray-core/infra/conf"
)

// https://github.com/XTLS/Xray-core/discussions/716
// Convert share text to XrayJson
// support v2rayN plain text, v2rayN base64 text
func ConvertShareLinksToXrayJson(links string) (*conf.Config, error) {
	text := strings.TrimSpace(links)
	if strings.HasPrefix(text, "{") {
		var xray *conf.Config
		err := json.Unmarshal([]byte(text), &xray)
		if err != nil {
			return nil, err
		}

		outbounds := xray.OutboundConfigs
		if len(outbounds) == 0 {
			return nil, fmt.Errorf("no valid outbounds")
		}

		return xray, nil
	}

	text = FixWindowsReturn(text)
	if checkSupportedShareLink(text) {
		xray, err := parsePlainShareText(text)
		if err != nil {
			return xray, err
		}
		return xray, nil
	} else {
		xray, err := tryParse(text)
		if err != nil {
			return nil, err
		}
		return xray, nil
	}
}

func checkSupportedShareLink(text string) bool {
	supportedSchemes := []string{"vless://", "vmess://", "socks://", "ss://", "trojan://", "hysteria2://", "hy2://"}
	for _, scheme := range supportedSchemes {
		if strings.HasPrefix(text, scheme) {
			return true
		}
	}
	return false
}

func FixWindowsReturn(text string) string {
	if strings.Contains(text, "\r\n") {
		text = strings.ReplaceAll(text, "\r\n", "\n")
	}
	return text
}

func parsePlainShareText(text string) (*conf.Config, error) {
	proxies := strings.Split(text, "\n")

	xray := &conf.Config{}
	var outbounds []conf.OutboundDetourConfig
	for _, proxy := range proxies {
		link, err := url.Parse(proxy)
		if err == nil {
			var shareLink xrayShareLink
			shareLink.link = link
			shareLink.rawText = proxy
			if outbound, err := shareLink.outbound(); err == nil {
				outbounds = append(outbounds, *outbound)
			} else {
				fmt.Println(err)
			}
		} else {
			fmt.Println(err)
		}
	}
	if len(outbounds) == 0 {
		return nil, fmt.Errorf("no valid outbound found")
	}
	xray.OutboundConfigs = outbounds
	return xray, nil
}

func tryParse(text string) (*conf.Config, error) {
	base64Text, err := decodeBase64Text(text)
	if err == nil {
		cleanText := FixWindowsReturn(base64Text)
		return parsePlainShareText(cleanText)
	}
	return tryToParseClashYaml(text)
}

func decodeBase64Text(text string) (string, error) {
	content, err := base64.StdEncoding.DecodeString(text)
	if err == nil {
		return string(content), nil
	}
	if strings.Contains(text, "-") {
		text = strings.ReplaceAll(text, "-", "+")
	}
	if strings.Contains(text, "_") {
		text = strings.ReplaceAll(text, "_", "/")
	}
	missingPadding := len(text) % 4
	if missingPadding != 0 {
		padding := strings.Repeat("=", 4-missingPadding)
		text += padding
	}
	content, err = base64.StdEncoding.DecodeString(text)
	if err != nil {
		return "", err
	}
	return string(content), nil
}

type xrayShareLink struct {
	link    *url.URL
	rawText string
}

func (proxy xrayShareLink) outbound() (*conf.OutboundDetourConfig, error) {
	switch proxy.link.Scheme {
	case "ss":
		outbound, err := proxy.shadowsocksOutbound()
		if err != nil {
			return nil, err
		}
		return outbound, nil

	case "vmess":
		outbound, err := proxy.vmessOutbound()
		if err != nil {
			return nil, err
		}
		return outbound, nil

	case "vless":
		outbound, err := proxy.vlessOutbound()
		if err != nil {
			return nil, err
		}
		return outbound, nil

	case "socks":
		outbound, err := proxy.socksOutbound()
		if err != nil {
			return nil, err
		}
		return outbound, nil
	case "trojan":
		outbound, err := proxy.trojanOutbound()
		if err != nil {
			return nil, err
		}
		return outbound, nil
	case "hysteria2", "hy2":
		outbound, err := proxy.hysteriaOutbound()
		if err != nil {
			return nil, err
		}
		return outbound, nil
	}
	return nil, fmt.Errorf("unsupport link: %s", proxy.rawText)
}

func (proxy xrayShareLink) shadowsocksOutbound() (*conf.OutboundDetourConfig, error) {
	outbound := &conf.OutboundDetourConfig{}
	outbound.Protocol = "shadowsocks"
	setOutboundName(outbound, proxy.link.Fragment)

	settings := &conf.ShadowsocksClientConfig{}

	settings.Address = parseAddress(proxy.link.Hostname())
	port, err := strconv.Atoi(proxy.link.Port())
	if err != nil {
		return nil, err
	}
	settings.Port = uint16(port)

	user := proxy.link.User.String()
	passwordText, err := decodeBase64Text(user)
	if err != nil {
		return nil, err
	}
	pwConfig := strings.SplitN(passwordText, ":", 2)
	if len(pwConfig) != 2 {
		return nil, fmt.Errorf("unsupport link shadowsocks password: %s", passwordText)
	}
	settings.Cipher = pwConfig[0]
	settings.Password = pwConfig[1]

	settingsRawMessage, err := convertJsonToRawMessage(settings)
	if err != nil {
		return nil, err
	}
	outbound.Settings = &settingsRawMessage

	streamSettings, err := proxy.streamSettings(proxy.link)
	if err != nil {
		return nil, err
	}
	outbound.StreamSetting = streamSettings

	return outbound, nil
}

func (proxy xrayShareLink) vmessOutbound() (*conf.OutboundDetourConfig, error) {
	// try vmessQrCode
	text := strings.ReplaceAll(proxy.rawText, "vmess://", "")
	base64Text, err := decodeBase64Text(text)
	if err == nil {
		return parseVMessQrCode(base64Text)
	}

	outbound := &conf.OutboundDetourConfig{}
	outbound.Protocol = "vmess"
	setOutboundName(outbound, proxy.link.Fragment)

	query := proxy.link.Query()

	settings := conf.VMessOutboundConfig{}

	settings.Address = parseAddress(proxy.link.Hostname())
	port, err := strconv.Atoi(proxy.link.Port())
	if err != nil {
		return nil, err
	}
	settings.Port = uint16(port)

	id, err := url.QueryUnescape(proxy.link.User.String())
	if err != nil {
		return nil, err
	}
	settings.ID = id
	security := query.Get("encryption")
	if len(security) > 0 {
		settings.Security = security
	}

	settingsRawMessage, err := convertJsonToRawMessage(settings)
	if err != nil {
		return nil, err
	}
	outbound.Settings = &settingsRawMessage

	streamSettings, err := proxy.streamSettings(proxy.link)
	if err != nil {
		return nil, err
	}
	outbound.StreamSetting = streamSettings

	return outbound, nil
}

func (proxy xrayShareLink) vlessOutbound() (*conf.OutboundDetourConfig, error) {
	outbound := &conf.OutboundDetourConfig{}
	outbound.Protocol = "vless"
	setOutboundName(outbound, proxy.link.Fragment)

	query := proxy.link.Query()

	settings := &conf.VLessOutboundConfig{}
	settings.Address = parseAddress(proxy.link.Hostname())
	port, err := strconv.Atoi(proxy.link.Port())
	if err != nil {
		return nil, err
	}
	settings.Port = uint16(port)

	id, err := url.QueryUnescape(proxy.link.User.String())
	if err != nil {
		return nil, err
	}
	settings.Id = id
	flow := query.Get("flow")
	if len(flow) > 0 {
		settings.Flow = flow
	}
	encryption := query.Get("encryption")
	if len(encryption) > 0 {
		settings.Encryption = encryption
	} else {
		settings.Encryption = "none"
	}

	settingsRawMessage, err := convertJsonToRawMessage(settings)
	if err != nil {
		return nil, err
	}
	outbound.Settings = &settingsRawMessage

	streamSettings, err := proxy.streamSettings(proxy.link)
	if err != nil {
		return nil, err
	}
	outbound.StreamSetting = streamSettings

	return outbound, nil
}

func (proxy xrayShareLink) socksOutbound() (*conf.OutboundDetourConfig, error) {
	outbound := &conf.OutboundDetourConfig{}
	outbound.Protocol = "socks"
	setOutboundName(outbound, proxy.link.Fragment)

	settings := &conf.SocksClientConfig{}

	settings.Address = parseAddress(proxy.link.Hostname())
	port, err := strconv.Atoi(proxy.link.Port())
	if err != nil {
		return nil, err
	}
	settings.Port = uint16(port)

	userPassword := proxy.link.User.String()
	if len(userPassword) > 0 {
		passwordText, err := decodeBase64Text(userPassword)
		if err != nil {
			return nil, err
		}
		pwConfig := strings.SplitN(passwordText, ":", 2)
		if len(pwConfig) != 2 {
			return nil, fmt.Errorf("unsupport link socks user password: %s", passwordText)
		}

		settings.Username = pwConfig[0]
		settings.Password = pwConfig[1]
	}

	settingsRawMessage, err := convertJsonToRawMessage(settings)
	if err != nil {
		return nil, err
	}
	outbound.Settings = &settingsRawMessage

	streamSettings, err := proxy.streamSettings(proxy.link)
	if err != nil {
		return nil, err
	}
	outbound.StreamSetting = streamSettings

	return outbound, nil
}

func (proxy xrayShareLink) trojanOutbound() (*conf.OutboundDetourConfig, error) {
	outbound := &conf.OutboundDetourConfig{}
	outbound.Protocol = "trojan"
	setOutboundName(outbound, proxy.link.Fragment)

	settings := &conf.TrojanClientConfig{}

	settings.Address = parseAddress(proxy.link.Hostname())
	port, err := strconv.Atoi(proxy.link.Port())
	if err != nil {
		return nil, err
	}
	settings.Port = uint16(port)

	password, err := url.QueryUnescape(proxy.link.User.String())
	if err != nil {
		return nil, err
	}
	settings.Password = password

	settingsRawMessage, err := convertJsonToRawMessage(settings)
	if err != nil {
		return nil, err
	}
	outbound.Settings = &settingsRawMessage

	streamSettings, err := proxy.streamSettings(proxy.link)
	if err != nil {
		return nil, err
	}
	outbound.StreamSetting = streamSettings

	return outbound, nil
}

func (proxy xrayShareLink) hysteriaOutbound() (*conf.OutboundDetourConfig, error) {
	outbound := &conf.OutboundDetourConfig{}
	outbound.Protocol = "hysteria"
	setOutboundName(outbound, proxy.link.Fragment)

	settings := &conf.HysteriaClientConfig{}

	settings.Version = 2
	settings.Address = parseAddress(proxy.link.Hostname())
	port, err := strconv.Atoi(proxy.link.Port())
	if err != nil {
		return nil, err
	}
	settings.Port = uint16(port)

	settingsRawMessage, err := convertJsonToRawMessage(settings)
	if err != nil {
		return nil, err
	}
	outbound.Settings = &settingsRawMessage

	streamSettings := &conf.StreamConfig{}
	network := conf.TransportProtocol("hysteria")
	streamSettings.Network = &network

	hysteriaSettings := &conf.HysteriaConfig{}
	hysteriaSettings.Version = 2
	auth, err := url.QueryUnescape(proxy.link.User.String())
	if err != nil {
		return nil, err
	}
	hysteriaSettings.Auth = auth
	streamSettings.HysteriaSettings = hysteriaSettings

	streamSettings.Security = "tls"

	query := proxy.link.Query()
	if len(query) > 0 {
		tlsSettings := &conf.TLSConfig{}
		sni := query.Get("sni")
		if len(sni) > 0 {
			tlsSettings.ServerName = sni
		}
		streamSettings.TLSSettings = tlsSettings

		obfsPassword := query.Get("obfs-password")
		if len(obfsPassword) > 0 {
			obfs := conf.Mask{}
			obfs.Type = "salamander"

			settings := &conf.Salamander{}
			settings.Password = obfsPassword

			settingsRawMessage, err := convertJsonToRawMessage(settings)
			if err != nil {
				return nil, err
			}

			obfs.Settings = &settingsRawMessage

			udp := []conf.Mask{obfs}
			finalMask := conf.FinalMask{}
			finalMask.Udp = udp

			streamSettings.FinalMask = &finalMask
		}
	}

	outbound.StreamSetting = streamSettings

	return outbound, nil
}

func (proxy xrayShareLink) streamSettings(link *url.URL) (*conf.StreamConfig, error) {
	query := link.Query()
	if len(query) == 0 {
		return nil, nil
	}

	streamSettings := &conf.StreamConfig{}

	network := query.Get("type")
	if len(network) == 0 {
		network = "raw"
	}
	transportProtocol := conf.TransportProtocol(network)
	streamSettings.Network = &transportProtocol

	switch network {
	case "raw", "tcp":
		headerType := query.Get("headerType")
		if headerType == "http" {
			var request XrayRawSettingsHeaderRequest
			path := query.Get("path")
			if len(path) > 0 {
				request.Path = strings.Split(path, ",")
			}
			host := query.Get("host")
			if len(host) > 0 {
				var headers XrayRawSettingsHeaderRequestHeaders
				headers.Host = strings.Split(host, ",")
				request.Headers = &headers
			}
			var header XrayRawSettingsHeader
			header.Type = headerType
			header.Request = &request

			rawSettings := &conf.TCPConfig{}

			headerRawMessage, err := convertJsonToRawMessage(header)
			if err != nil {
				return nil, err
			}
			rawSettings.HeaderConfig = headerRawMessage

			streamSettings.RAWSettings = rawSettings
		}
	case "kcp", "mkcp":
		kcpSettings := &conf.KCPConfig{}
		headerType := query.Get("headerType")
		if len(headerType) > 0 {
			var header XrayFakeHeader
			header.Type = headerType

			headerRawMessage, err := convertJsonToRawMessage(header)
			if err != nil {
				return nil, err
			}
			kcpSettings.HeaderConfig = headerRawMessage
		}
		seed := query.Get("seed")
		kcpSettings.Seed = &seed

		streamSettings.KCPSettings = kcpSettings
	case "ws", "websocket":
		wsSettings := &conf.WebSocketConfig{}
		wsSettings.Path = query.Get("path")
		wsSettings.Host = query.Get("host")

		streamSettings.WSSettings = wsSettings
	case "grpc", "gun":
		grcpSettings := &conf.GRPCConfig{}
		grcpSettings.Authority = query.Get("authority")
		grcpSettings.ServiceName = query.Get("serviceName")
		grcpSettings.MultiMode = query.Get("mode") == "multi"

		streamSettings.GRPCSettings = grcpSettings
	case "httpupgrade":
		httpupgradeSettings := &conf.HttpUpgradeConfig{}
		httpupgradeSettings.Host = query.Get("host")
		httpupgradeSettings.Path = query.Get("path")

		streamSettings.HTTPUPGRADESettings = httpupgradeSettings
	case "xhttp", "splithttp":
		xhttpSettings := &conf.SplitHTTPConfig{}
		xhttpSettings.Host = query.Get("host")
		xhttpSettings.Path = query.Get("path")
		xhttpSettings.Mode = query.Get("mode")

		extra := query.Get("extra")
		if len(extra) > 0 {
			var extraConfig *conf.SplitHTTPConfig
			err := json.Unmarshal([]byte(extra), &extraConfig)
			if err != nil {
				return nil, err
			}
			extraRawMessage, err := convertJsonToRawMessage(extraConfig)
			if err != nil {
				return nil, err
			}
			xhttpSettings.Extra = extraRawMessage
		}

		streamSettings.XHTTPSettings = xhttpSettings
	}

	fm := query.Get("fm")
	if len(fm) > 0 {
		var finalMask *conf.FinalMask
		err := json.Unmarshal([]byte(fm), &finalMask)
		if err != nil {
			return nil, err
		}
		streamSettings.FinalMask = finalMask
	}

	err := proxy.parseSecurity(link, streamSettings)
	if err != nil {
		return nil, err
	}

	return streamSettings, nil
}

func (proxy xrayShareLink) parseSecurity(link *url.URL, streamSettings *conf.StreamConfig) error {
	query := link.Query()

	tlsSettings := &conf.TLSConfig{}
	realitySettings := &conf.REALITYConfig{}

	fp := query.Get("fp")
	tlsSettings.Fingerprint = fp
	realitySettings.Fingerprint = fp

	sni := query.Get("sni")
	tlsSettings.ServerName = sni
	realitySettings.ServerName = sni

	ech := query.Get("ech")
	tlsSettings.ECHConfigList = ech

	pcs := query.Get("pcs")
	tlsSettings.PinnedPeerCertSha256 = pcs
	vcn := query.Get("vcn")
	tlsSettings.VerifyPeerCertByName = vcn

	alpn := query.Get("alpn")
	if len(alpn) > 0 {
		alpn := conf.StringList(strings.Split(alpn, ","))
		tlsSettings.ALPN = &alpn
	}

	pbk := query.Get("pbk")
	realitySettings.Password = pbk
	realitySettings.PublicKey = pbk
	sid := query.Get("sid")
	realitySettings.ShortId = sid
	pqv := query.Get("pqv")
	realitySettings.Mldsa65Verify = pqv
	spx := query.Get("spx")
	realitySettings.SpiderX = spx

	security := query.Get("security")
	if len(security) == 0 {
		streamSettings.Security = "none"
	} else {
		streamSettings.Security = security
	}

	// some link omits too many params, here is some fixing
	if proxy.link.Scheme == "trojan" && streamSettings.Security == "none" {
		streamSettings.Security = "tls"
	}

	network, err := streamSettings.Network.Build()
	if err != nil {
		return err
	}
	if network == "websocket" && len(tlsSettings.ServerName) == 0 {
		if streamSettings.WSSettings != nil && len(streamSettings.WSSettings.Host) > 0 {
			tlsSettings.ServerName = streamSettings.WSSettings.Host
		}
	}

	switch streamSettings.Security {
	case "tls":
		streamSettings.TLSSettings = tlsSettings
	case "reality":
		streamSettings.REALITYSettings = realitySettings
	}
	return nil
}
