package io.github.kitswas.virtualgamepadmobile.ui.composables

import androidx.compose.foundation.gestures.detectDragGestures
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.size
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalView
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.dp
import io.github.kitswas.VGP_Data_Exchange.GameButtons
import io.github.kitswas.VGP_Data_Exchange.GamepadReading
import io.github.kitswas.virtualgamepadmobile.ui.theme.darken
import io.github.kitswas.virtualgamepadmobile.ui.theme.lighten
import io.github.kitswas.virtualgamepadmobile.ui.utils.HapticUtils
import kotlin.math.roundToInt
import kotlin.math.sqrt

enum class AnalogStickType {
    LEFT, RIGHT
}

@Composable
fun AnalogStick(
    modifier: Modifier = Modifier,
    ringColor: Color = MaterialTheme.colorScheme.outline,
    ringWidth: Dp = 4.dp,
    outerCircleColor: Color = darken(MaterialTheme.colorScheme.primary, 0.8f),
    outerCircleWidth: Dp = 4.dp,
    innerCircleColor: Color = lighten(MaterialTheme.colorScheme.primary, 0.2f),
    innerCircleRadius: Dp = 32.dp,
    gamepadState: GamepadReading,
    type: AnalogStickType,
) {
    val density = LocalDensity.current
    val view = LocalView.current

    Box(
        modifier = modifier,
        contentAlignment = Alignment.Center
    ) {
        // First draw the glow ring
        Circle(
            colour = ringColor,
            modifier = Modifier
                .size((innerCircleRadius + outerCircleWidth + ringWidth) * 2),
            contentAlignment = Alignment.Center
        ) {
            // Then draw the outer circle
            Circle(
                modifier = Modifier
                    .size((innerCircleRadius + outerCircleWidth) * 2),
                contentAlignment = Alignment.Center,
                colour = outerCircleColor,
            ) {
                // Raw offset values (more efficient than using a data class)
                var offsetX by remember { mutableFloatStateOf(0f) }
                var offsetY by remember { mutableFloatStateOf(0f) }

                // Visual position values (calculated once and reused)
                var visualX by remember { mutableFloatStateOf(0f) }
                var visualY by remember { mutableFloatStateOf(0f) }

                // Calculate maximum offset once
                val maxOffset = with(density) {
                    (innerCircleRadius + outerCircleWidth).toPx()
                }

                var isPressed by remember { mutableStateOf(false) }

                // Then draw the inner circle
                Circle(
                    colour = innerCircleColor,
                    modifier = Modifier
                        .size(innerCircleRadius * 2)
                        .offset {
                            IntOffset(
                                visualX.roundToInt(),
                                visualY.roundToInt()
                            )
                        }
                        .pointerInput(Unit) {
                            detectDragGestures(
                                onDragStart = { _ ->
                                    HapticUtils.performGestureStartFeedback(view)
                                },
                                onDragEnd = {
                                    // Reset position
                                    offsetX = 0f
                                    offsetY = 0f
                                    visualX = 0f
                                    visualY = 0f

                                    // Update gamepad state
                                    when (type) {
                                        AnalogStickType.LEFT -> {
                                            gamepadState.LeftThumbstickX = 0f
                                            gamepadState.LeftThumbstickY = 0f
                                        }

                                        AnalogStickType.RIGHT -> {
                                            gamepadState.RightThumbstickX = 0f
                                            gamepadState.RightThumbstickY = 0f
                                        }
                                    }
                                    HapticUtils.performGestureEndFeedback(view)
                                },
                                onDrag = { change, dragAmount ->
                                    change.consume()

                                    // Update raw position
                                    offsetX += dragAmount.x
                                    offsetY += dragAmount.y

                                    // Calculate magnitude for normalized position
                                    val magnitude = sqrt(offsetX * offsetX + offsetY * offsetY)
                                    // Calculate normalized distance (0-1 range) for haptic intensity
                                    val normalizedDistance =
                                        (magnitude / maxOffset).coerceIn(0f, 1f)

                                    if (normalizedDistance > 0.3f) {
                                        // Subtle movement feedback for better tactile experience
                                        HapticUtils.performAnalogMovementFeedback(
                                            view,
                                            normalizedDistance
                                        )
                                    }

                                    // Only update when magnitude > 0
                                    if (magnitude > 0f) {
                                        // Normalize with max offset (calculate once and reuse)
                                        val scaleFactor =
                                            if (magnitude > maxOffset) maxOffset / magnitude else 1f
                                        visualX = offsetX * scaleFactor
                                        visualY = offsetY * scaleFactor

                                        when (type) {
                                            AnalogStickType.LEFT -> {
                                                gamepadState.LeftThumbstickX = visualX / maxOffset
                                                gamepadState.LeftThumbstickY = visualY / maxOffset
                                            }

                                            AnalogStickType.RIGHT -> {
                                                gamepadState.RightThumbstickX = visualX / maxOffset
                                                gamepadState.RightThumbstickY = visualY / maxOffset
                                            }
                                        }
                                    }
                                }
                            )
                        }
                        .pointerInput(Unit) {
                            detectTapGestures(
                                onLongPress = { _ ->
                                    if (!isPressed) {
                                        isPressed = true
                                        gamepadState.ButtonsDown = when (type) {
                                            AnalogStickType.LEFT -> gamepadState.ButtonsDown or GameButtons.LeftThumbstick.value
                                            AnalogStickType.RIGHT -> gamepadState.ButtonsDown or GameButtons.RightThumbstick.value
                                        }
                                        HapticUtils.performButtonPressFeedback(view)
                                    } else {
                                        isPressed = false
                                        gamepadState.ButtonsDown = when (type) {
                                            AnalogStickType.LEFT -> gamepadState.ButtonsDown and GameButtons.LeftThumbstick.value.inv()
                                            AnalogStickType.RIGHT -> gamepadState.ButtonsDown and GameButtons.RightThumbstick.value.inv()
                                        }
                                        gamepadState.ButtonsUp = when (type) {
                                            AnalogStickType.LEFT -> gamepadState.ButtonsUp or GameButtons.LeftThumbstick.value
                                            AnalogStickType.RIGHT -> gamepadState.ButtonsUp or GameButtons.RightThumbstick.value
                                        }
                                        HapticUtils.performGestureEndFeedback(view)
                                    }
                                }
                            )
                        }
                ) {}
            }
        }
    }
}

@Preview(showBackground = false)
@Composable
fun AnalogStickPreview() {
    AnalogStick(
        gamepadState = GamepadReading(),
        type = AnalogStickType.LEFT,
    )
}
