import 'package:win32/win32.dart';

/// Represents one of the predefined registry keys in Windows, each pointing to
/// a specific area of the Windows registry.
///
/// These predefined keys serve as entry points into the Windows registry and
/// help applications navigate the registry structure. They allow system
/// administrators and applications to access, modify, and categorize registry
/// data. When adding data to the registry, applications should ideally work
/// within the framework of these predefined keys to maintain compatibility
/// with administrative tools.
enum RegistryHive {
  /// Points to registry entries defining types (or classes) of documents and
  /// the properties associated with those types.
  ///
  /// Used by Shell and COM applications for storing class-specific information.
  classesRoot._(HKEY_CLASSES_ROOT),

  /// Points to registry entries defining the preferences and settings for the
  /// current user, including environment variables, network connections,
  /// application preferences, and more.
  ///
  /// This key is mapped to the current user's data in `HKEY_USERS`.
  currentUser._(HKEY_CURRENT_USER),

  /// Points to registry entries defining the physical and software
  /// configuration of the local machine, such as system memory, hardware,
  /// installed software, and other system-wide configurations.
  localMachine._(HKEY_LOCAL_MACHINE),

  /// Points to registry entries defining the default settings for new users on
  /// the local machine as well as the configuration settings for the current
  /// user.
  allUsers._(HKEY_USERS),

  /// Points to a location providing access to performance data, dynamically
  /// generated by the system.
  ///
  /// The data itself is not stored in the registry but is collected when
  /// requested.
  performanceData._(HKEY_PERFORMANCE_DATA),

  /// Contains information about the current hardware profile of the local
  /// computer system, describing differences between the current configuration
  /// and the standard hardware configuration.
  currentConfig._(HKEY_CURRENT_CONFIG);

  const RegistryHive._(this.value);

  /// Creates a [RegistryHive] instance from a Win32 handle.
  ///
  /// Throws an [ArgumentError] if the provided handle does not match any
  /// predefined registry hive.
  factory RegistryHive.fromWin32(int value) => switch (value) {
    HKEY_CLASSES_ROOT => classesRoot,
    HKEY_CURRENT_USER => currentUser,
    HKEY_LOCAL_MACHINE => localMachine,
    HKEY_USERS => allUsers,
    HKEY_PERFORMANCE_DATA => performanceData,
    HKEY_CURRENT_CONFIG => currentConfig,
    _ =>
      throw ArgumentError.value(
        value,
        'value',
        'Unknown registry hive: $value',
      ),
  };

  /// The handle representing a predefined registry key.
  final int value;

  @override
  String toString() => switch (this) {
    classesRoot => 'HKEY_CLASSES_ROOT',
    currentUser => 'HKEY_CURRENT_USER',
    localMachine => 'HKEY_LOCAL_MACHINE',
    allUsers => 'HKEY_USERS',
    performanceData => 'HKEY_PERFORMANCE_DATA',
    currentConfig => 'HKEY_CURRENT_CONFIG',
  };
}
