package io.github.dorumrr.de1984.data.service

import io.github.dorumrr.de1984.utils.AppLogger
import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.app.Service
import android.content.Context
import android.content.Intent
import android.os.Build
import android.os.IBinder
import androidx.core.app.NotificationCompat
import io.github.dorumrr.de1984.De1984Application
import io.github.dorumrr.de1984.R
import io.github.dorumrr.de1984.data.firewall.ConnectivityManagerFirewallBackend
import io.github.dorumrr.de1984.data.firewall.IptablesFirewallBackend
import io.github.dorumrr.de1984.data.firewall.NetworkPolicyManagerFirewallBackend
import io.github.dorumrr.de1984.data.monitor.NetworkStateMonitor
import io.github.dorumrr.de1984.data.monitor.ScreenStateMonitor
import io.github.dorumrr.de1984.domain.firewall.FirewallBackend
import io.github.dorumrr.de1984.domain.firewall.FirewallBackendType
import io.github.dorumrr.de1984.domain.model.NetworkType
import io.github.dorumrr.de1984.domain.repository.FirewallRepository
import io.github.dorumrr.de1984.ui.MainActivity
import io.github.dorumrr.de1984.utils.Constants
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch

/**
 * Foreground service for privileged firewall backends (iptables, ConnectivityManager, NetworkPolicyManager).
 * 
 * This service keeps the app process alive to maintain:
 * - Backend health monitoring
 * - Network/screen state monitoring
 * - Rule application on state changes
 * - State persistence across process death
 * 
 * Follows the same pattern as FirewallVpnService but for privileged backends.
 */
class PrivilegedFirewallService : Service() {

    private lateinit var firewallRepository: FirewallRepository
    private lateinit var networkStateMonitor: NetworkStateMonitor
    private lateinit var screenStateMonitor: ScreenStateMonitor

    private val serviceScope = CoroutineScope(SupervisorJob() + Dispatchers.IO)
    private var monitoringJob: Job? = null
    private var healthMonitoringJob: Job? = null
    private var ruleApplicationJob: Job? = null

    private var currentBackend: FirewallBackend? = null
    private var currentBackendType: FirewallBackendType? = null
    private var isServiceActive = false
    private var wasExplicitlyStopped = false

    // Adaptive health check tracking
    private var consecutiveSuccessfulHealthChecks = 0
    private var currentHealthCheckInterval = Constants.HealthCheck.BACKEND_HEALTH_CHECK_INTERVAL_INITIAL_MS

    private var currentNetworkType: NetworkType = NetworkType.NONE
    private var isScreenOn: Boolean = true

    private val rulesChangedReceiver = object : android.content.BroadcastReceiver() {
        override fun onReceive(context: android.content.Context?, intent: android.content.Intent?) {
            if (intent?.action == "io.github.dorumrr.de1984.FIREWALL_RULES_CHANGED") {
                AppLogger.d(TAG, "🔥 [TIMING] Broadcast RECEIVED: timestamp=${System.currentTimeMillis()}")
                if (isServiceActive) {
                    // Clear backend cache to force re-evaluation of all packages
                    // This ensures rule changes take effect immediately without restart
                    val backend = currentBackend
                    if (backend is ConnectivityManagerFirewallBackend) {
                        backend.clearAppliedPoliciesCache()
                        AppLogger.d(TAG, "Cleared ConnectivityManager cache on rule change")
                    } else if (backend is NetworkPolicyManagerFirewallBackend) {
                        backend.clearAppliedPoliciesCache()
                        AppLogger.d(TAG, "Cleared NetworkPolicyManager cache on rule change")
                    }
                    scheduleRuleApplication("broadcast")
                }
            }
        }
    }

    companion object {
        private const val TAG = "PrivilegedFirewallService"
        private const val NOTIFICATION_ID = 1002
        private const val CHANNEL_ID = "firewall_privileged_channel"
        private const val CHANNEL_NAME = "Firewall Service"

        const val ACTION_START = "io.github.dorumrr.de1984.action.START_PRIVILEGED_FIREWALL"
        const val ACTION_STOP = "io.github.dorumrr.de1984.action.STOP_PRIVILEGED_FIREWALL"
        const val EXTRA_BACKEND_TYPE = "backend_type"
    }

    override fun onCreate() {
        super.onCreate()

        // Initialize dependencies manually
        val app = application as De1984Application
        val deps = app.dependencies
        firewallRepository = deps.firewallRepository
        networkStateMonitor = deps.networkStateMonitor
        screenStateMonitor = deps.screenStateMonitor

        createNotificationChannel()

        val filter = android.content.IntentFilter("io.github.dorumrr.de1984.FIREWALL_RULES_CHANGED")
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            registerReceiver(rulesChangedReceiver, filter, Context.RECEIVER_NOT_EXPORTED)
        } else {
            @Suppress("UnspecifiedRegisterReceiverFlag")
            registerReceiver(rulesChangedReceiver, filter)
        }

        AppLogger.d(TAG, "Service created")
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        AppLogger.d(TAG, "onStartCommand: action=${intent?.action}, wasExplicitlyStopped=$wasExplicitlyStopped")

        if (wasExplicitlyStopped && intent?.action != ACTION_START) {
            AppLogger.d(TAG, "Service was explicitly stopped and no START action - stopping self")
            stopSelf()
            return START_NOT_STICKY
        }

        when (intent?.action) {
            ACTION_START -> {
                val backendTypeStr = intent.getStringExtra(EXTRA_BACKEND_TYPE)
                val backendType = when (backendTypeStr) {
                    "IPTABLES" -> FirewallBackendType.IPTABLES
                    "CONNECTIVITY_MANAGER" -> FirewallBackendType.CONNECTIVITY_MANAGER
                    "NETWORK_POLICY_MANAGER" -> FirewallBackendType.NETWORK_POLICY_MANAGER
                    else -> {
                        AppLogger.e(TAG, "Invalid backend type: $backendTypeStr")
                        stopSelf()
                        return START_NOT_STICKY
                    }
                }

                AppLogger.d(TAG, "ACTION_START received - starting privileged firewall with backend: $backendType")
                wasExplicitlyStopped = false
                startFirewall(backendType)
                return START_STICKY
            }
            ACTION_STOP -> {
                AppLogger.d(TAG, "ACTION_STOP received - stopping privileged firewall")
                wasExplicitlyStopped = true
                stopFirewall()
                return START_NOT_STICKY
            }
            else -> {
                AppLogger.w(TAG, "Unknown action or null intent - stopping self")
                stopSelf()
                return START_NOT_STICKY
            }
        }
    }

    override fun onBind(intent: Intent?): IBinder? = null

    override fun onDestroy() {
        stopFirewall()
        serviceScope.cancel()

        try {
            unregisterReceiver(rulesChangedReceiver)
        } catch (e: Exception) {
            AppLogger.w(TAG, "Failed to unregister broadcast receiver", e)
        }

        super.onDestroy()
        AppLogger.d(TAG, "Service destroyed")
    }

    private fun createNotificationChannel() {
        val channel = NotificationChannel(
            CHANNEL_ID,
            CHANNEL_NAME,
            NotificationManager.IMPORTANCE_LOW
        ).apply {
            description = "Firewall service notification"
            setShowBadge(false)
        }

        val notificationManager = getSystemService(NotificationManager::class.java)
        notificationManager.createNotificationChannel(channel)
    }

    private fun createNotification(): Notification {
        val intent = Intent(this, MainActivity::class.java)
        val pendingIntent = PendingIntent.getActivity(
            this,
            0,
            intent,
            PendingIntent.FLAG_IMMUTABLE
        )

        val backendName = when (currentBackendType) {
            FirewallBackendType.IPTABLES -> "iptables"
            FirewallBackendType.CONNECTIVITY_MANAGER -> "ConnectivityManager"
            FirewallBackendType.NETWORK_POLICY_MANAGER -> "NetworkPolicyManager"
            else -> "Unknown"
        }

        return NotificationCompat.Builder(this, CHANNEL_ID)
            .setContentTitle(getString(R.string.privileged_firewall_notification_title))
            .setContentText(getString(R.string.privileged_firewall_notification_text, backendName))
            .setSmallIcon(R.drawable.ic_notification_de1984)
            .setContentIntent(pendingIntent)
            .setOngoing(true)
            .build()
    }

    private fun startFirewall(backendType: FirewallBackendType) {
        AppLogger.d(TAG, "startFirewall() called with backend: $backendType")

        serviceScope.launch {
            try {
                // Create backend instance
                val app = application as De1984Application
                val deps = app.dependencies

                val backend = when (backendType) {
                    FirewallBackendType.IPTABLES -> {
                        val b = IptablesFirewallBackend(
                            context = applicationContext,
                            rootManager = deps.rootManager,
                            shizukuManager = deps.shizukuManager,
                            errorHandler = deps.errorHandler
                        )
                        // Call internal start method
                        b.startInternal().getOrElse { error ->
                            AppLogger.e(TAG, "Failed to start iptables backend: ${error.message}")
                            stopSelf()
                            return@launch
                        }
                        b
                    }
                    FirewallBackendType.CONNECTIVITY_MANAGER -> {
                        val b = ConnectivityManagerFirewallBackend(
                            context = applicationContext,
                            shizukuManager = deps.shizukuManager,
                            errorHandler = deps.errorHandler
                        )
                        // Call internal start method
                        b.startInternal().getOrElse { error ->
                            AppLogger.e(TAG, "Failed to start ConnectivityManager backend: ${error.message}")
                            stopSelf()
                            return@launch
                        }
                        b
                    }
                    FirewallBackendType.NETWORK_POLICY_MANAGER -> {
                        val b = NetworkPolicyManagerFirewallBackend(
                            context = applicationContext,
                            shizukuManager = deps.shizukuManager,
                            errorHandler = deps.errorHandler
                        )
                        // Call internal start method
                        b.startInternal().getOrElse { error ->
                            AppLogger.e(TAG, "Failed to start NetworkPolicyManager backend: ${error.message}")
                            stopSelf()
                            return@launch
                        }
                        b
                    }
                    else -> {
                        AppLogger.e(TAG, "Unsupported backend type: $backendType")
                        stopSelf()
                        return@launch
                    }
                }

                currentBackend = backend
                currentBackendType = backendType
                isServiceActive = true

                // Update SharedPreferences to indicate service is running
                val prefs = getSharedPreferences(Constants.Settings.PREFS_NAME, Context.MODE_PRIVATE)
                prefs.edit()
                    .putBoolean(Constants.Settings.KEY_PRIVILEGED_SERVICE_RUNNING, true)
                    .putString(Constants.Settings.KEY_PRIVILEGED_BACKEND_TYPE, backendType.name)
                    .apply()
                AppLogger.d(TAG, "Updated SharedPreferences: PRIVILEGED_SERVICE_RUNNING=true, BACKEND_TYPE=$backendType")

                // Start foreground service
                AppLogger.d(TAG, "Starting foreground service with notification")
                startForeground(NOTIFICATION_ID, createNotification())

                // Apply initial rules
                scheduleRuleApplication("initial")

                // Start monitoring
                startMonitoring()
                startBackendHealthMonitoring()

                AppLogger.d(TAG, "Privileged firewall started successfully with backend: $backendType")
            } catch (e: Exception) {
                AppLogger.e(TAG, "Error in startFirewall", e)
                stopSelf()
            }
        }
    }

    private fun stopFirewall() {
        AppLogger.d(TAG, "stopFirewall() called")

        isServiceActive = false

        // Stop monitoring
        monitoringJob?.cancel()
        monitoringJob = null
        healthMonitoringJob?.cancel()
        healthMonitoringJob = null
        ruleApplicationJob?.cancel()
        ruleApplicationJob = null

        // Reset adaptive health check tracking
        consecutiveSuccessfulHealthChecks = 0
        currentHealthCheckInterval = Constants.HealthCheck.BACKEND_HEALTH_CHECK_INTERVAL_INITIAL_MS

        // Stop backend
        serviceScope.launch {
            val backend = currentBackend
            val backendType = currentBackendType

            if (backend != null && backendType != null) {
                // Call internal stop method based on backend type
                when (backendType) {
                    FirewallBackendType.IPTABLES -> {
                        (backend as? IptablesFirewallBackend)?.stopInternal()?.getOrElse { error ->
                            AppLogger.w(TAG, "Failed to stop iptables backend: ${error.message}")
                        }
                    }
                    FirewallBackendType.CONNECTIVITY_MANAGER -> {
                        (backend as? ConnectivityManagerFirewallBackend)?.stopInternal()?.getOrElse { error ->
                            AppLogger.w(TAG, "Failed to stop ConnectivityManager backend: ${error.message}")
                        }
                    }
                    FirewallBackendType.NETWORK_POLICY_MANAGER -> {
                        (backend as? NetworkPolicyManagerFirewallBackend)?.stopInternal()?.getOrElse { error ->
                            AppLogger.w(TAG, "Failed to stop NetworkPolicyManager backend: ${error.message}")
                        }
                    }
                    else -> {
                        AppLogger.w(TAG, "Unknown backend type: $backendType")
                    }
                }
            }

            currentBackend = null
            currentBackendType = null

            // Update SharedPreferences
            val prefs = getSharedPreferences(Constants.Settings.PREFS_NAME, Context.MODE_PRIVATE)
            prefs.edit()
                .putBoolean(Constants.Settings.KEY_PRIVILEGED_SERVICE_RUNNING, false)
                .remove(Constants.Settings.KEY_PRIVILEGED_BACKEND_TYPE)
                .apply()
            AppLogger.d(TAG, "Updated SharedPreferences: PRIVILEGED_SERVICE_RUNNING=false")

            // Stop foreground and remove notification
            stopForeground(STOP_FOREGROUND_REMOVE)
            stopSelf()
        }
    }

    private fun startMonitoring() {
        AppLogger.d(TAG, "Starting network/screen state monitoring")

        // Monitor network type and screen state changes
        monitoringJob = serviceScope.launch {
            combine(
                networkStateMonitor.observeNetworkType(),
                screenStateMonitor.observeScreenState()
            ) { networkType, screenOn ->
                Pair(networkType, screenOn)
            }.collect { (networkType, screenOn) ->
                currentNetworkType = networkType
                isScreenOn = screenOn

                if (isServiceActive) {
                    AppLogger.d(TAG, "State changed: network=$networkType, screen=$screenOn - scheduling rule application")
                    scheduleRuleApplication("state-change")
                }
            }
        }

        // Monitor rule changes from repository
        serviceScope.launch {
            firewallRepository.getAllRules().collect { _ ->
                if (isServiceActive) {
                    AppLogger.d(TAG, "🔥 [TIMING] Flow EMITTED: timestamp=${System.currentTimeMillis()}")
                    scheduleRuleApplication("flow")
                }
            }
        }
    }

    private fun startBackendHealthMonitoring() {
        // Reset adaptive tracking when starting new monitoring
        consecutiveSuccessfulHealthChecks = 0
        currentHealthCheckInterval = Constants.HealthCheck.BACKEND_HEALTH_CHECK_INTERVAL_INITIAL_MS

        AppLogger.d(TAG, "🔍 STARTING ADAPTIVE SERVICE HEALTH MONITORING | Initial interval: ${currentHealthCheckInterval}ms (30 seconds) | Stable interval: ${Constants.HealthCheck.BACKEND_HEALTH_CHECK_INTERVAL_STABLE_MS}ms (5 minutes) | Threshold: ${Constants.HealthCheck.BACKEND_HEALTH_CHECK_STABLE_THRESHOLD} successful checks | Purpose: Detect permission loss within service")

        healthMonitoringJob = serviceScope.launch {
            while (isServiceActive) {
                delay(currentHealthCheckInterval)

                val backend = currentBackend
                val backendType = currentBackendType

                if (backend == null || backendType == null) {
                    AppLogger.w(TAG, "⚠️  SERVICE HEALTH CHECK: backend is null, stopping monitoring")
                    break
                }

                try {
                    AppLogger.d(TAG, "=== SERVICE HEALTH CHECK: $backendType ===")
                    AppLogger.d(TAG, "Checking if backend still has required permissions... (interval: ${currentHealthCheckInterval}ms, consecutive successes: $consecutiveSuccessfulHealthChecks)")

                    // For root-based backends (iptables) explicitly re-check root status so
                    // we can detect Magisk revocation while the app is in background.
                    if (backendType == FirewallBackendType.IPTABLES) {
                        try {
                            val app = application as De1984Application
                            val deps = app.dependencies
                            AppLogger.d(TAG, "Health check: forcing root status re-check for iptables backend")
                            // CRITICAL: Must await the result so the StateFlow is updated before checkAvailability()
                            deps.rootManager.forceRecheckRootStatus()
                            AppLogger.d(TAG, "Health check: root status re-check complete, new status: ${deps.rootManager.rootStatus.value}")
                        } catch (e: Exception) {
                            AppLogger.w(TAG, "Health check: failed to force root status re-check: ${e.message}")
                        }
                    }

                    // Check if backend is still available (root/Shizuku access, iptables binary, etc.)
                    val availabilityResult = backend.checkAvailability()

                    if (availabilityResult.isFailure) {
                        AppLogger.e(TAG, "❌ SERVICE: BACKEND AVAILABILITY CHECK FAILED | Backend: $backendType | Reason: ${availabilityResult.exceptionOrNull()?.message} | Action: Stopping service to trigger FirewallManager fallback")

                        // Reset adaptive tracking on failure
                        consecutiveSuccessfulHealthChecks = 0
                        currentHealthCheckInterval = Constants.HealthCheck.BACKEND_HEALTH_CHECK_INTERVAL_INITIAL_MS

                        handleBackendFailure(backendType)
                        break
                    }

                    // Note: We don't check backend.isActive() here because it checks if THIS service
                    // is running (circular check). The checkAvailability() above is sufficient to
                    // verify the backend can still function (root/Shizuku access, APIs available, etc.)

                    // Health check passed - increment success counter
                    consecutiveSuccessfulHealthChecks++
                    AppLogger.d(TAG, "✅ SERVICE: Health check passed - $backendType is healthy (consecutive successes: $consecutiveSuccessfulHealthChecks)")

                    // Check if we should increase interval (backend is stable)
                    if (consecutiveSuccessfulHealthChecks >= Constants.HealthCheck.BACKEND_HEALTH_CHECK_STABLE_THRESHOLD &&
                        currentHealthCheckInterval == Constants.HealthCheck.BACKEND_HEALTH_CHECK_INTERVAL_INITIAL_MS) {
                        currentHealthCheckInterval = Constants.HealthCheck.BACKEND_HEALTH_CHECK_INTERVAL_STABLE_MS
                        AppLogger.d(TAG, "⚡ SERVICE: BACKEND STABLE - INCREASING INTERVAL | Backend: $backendType | New interval: ${currentHealthCheckInterval}ms (5 minutes) | Battery savings: ~90% reduction in wake-ups")
                    }

                } catch (e: Exception) {
                    AppLogger.e(TAG, "❌ SERVICE: HEALTH CHECK EXCEPTION | Backend: $backendType | Exception: ${e.message} | Action: Stopping service to trigger FirewallManager fallback")
                    AppLogger.e(TAG, "", e)

                    // Reset adaptive tracking on exception
                    consecutiveSuccessfulHealthChecks = 0
                    currentHealthCheckInterval = Constants.HealthCheck.BACKEND_HEALTH_CHECK_INTERVAL_INITIAL_MS

                    handleBackendFailure(backendType)
                    break
                }
            }
        }
    }

    private fun handleBackendFailure(backendType: FirewallBackendType) {
        AppLogger.e(TAG, "⚠️  BACKEND FAILURE DETECTED IN SERVICE | Backend: $backendType | Action: Notifying FirewallManager and stopping service")

        // Show notification to user
        showFailureNotification(backendType)

        // Notify FirewallManager immediately instead of waiting for health check
        // This makes VPN fallback instant instead of waiting up to 15 seconds
        try {
            val app = application as De1984Application
            val deps = app.dependencies
            AppLogger.e(TAG, "Notifying FirewallManager of backend failure...")
            serviceScope.launch {
                deps.firewallManager.handleBackendFailureFromService(backendType)
            }
        } catch (e: Exception) {
            AppLogger.e(TAG, "Failed to notify FirewallManager of backend failure: ${e.message}")
            // Continue with service stop - health check will detect it eventually
        }

        // Stop the service
        // IMPORTANT: Set wasExplicitlyStopped = true to prevent service from restarting
        // The FirewallManager will start VPN backend instead
        wasExplicitlyStopped = true

        AppLogger.e(TAG, "Stopping service now...")
        stopFirewall()

        AppLogger.e(TAG, "Service stopped. FirewallManager should handle VPN fallback immediately.")
    }

    private fun showFailureNotification(backendType: FirewallBackendType) {
        val backendName = when (backendType) {
            FirewallBackendType.IPTABLES -> "iptables"
            FirewallBackendType.CONNECTIVITY_MANAGER -> "ConnectivityManager"
            FirewallBackendType.NETWORK_POLICY_MANAGER -> "NetworkPolicyManager"
            else -> "Unknown"
        }

        // Open the main UI when the user taps the notification so they can see
        // the current firewall state and manually restart if needed.
        // We intentionally do NOT try to start any backend directly from here;
        // FirewallManager's planner remains the single source of truth.
        val intent = Intent(this, MainActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TOP
        }
        val pendingIntent = PendingIntent.getActivity(
            this,
            0,
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        val notification = NotificationCompat.Builder(this, CHANNEL_ID)
            .setContentTitle(getString(R.string.privileged_firewall_failure_notification_title))
            .setContentText(getString(R.string.privileged_firewall_failure_notification_text, backendName))
            .setSmallIcon(R.drawable.ic_notification_de1984)
            .setContentIntent(pendingIntent)
            .setPriority(NotificationCompat.PRIORITY_HIGH)
            .setAutoCancel(true)
            .build()

        val notificationManager = getSystemService(NotificationManager::class.java)
        notificationManager.notify(NOTIFICATION_ID + 1, notification)
    }

    private var ruleApplicationStartTime: Long = 0L

    private fun scheduleRuleApplication(source: String) {
        val previousJob = ruleApplicationJob
        ruleApplicationJob?.cancel()
        ruleApplicationStartTime = System.currentTimeMillis()

        AppLogger.d(TAG, "🔥 [TIMING] scheduleRuleApplication($source): previousJobActive=${previousJob?.isActive}, timestamp=$ruleApplicationStartTime")

        ruleApplicationJob = serviceScope.launch {
            AppLogger.d(TAG, "🔥 [TIMING] Debounce START (300ms): source=$source")
            delay(300)  // Debounce
            AppLogger.d(TAG, "🔥 [TIMING] Debounce END: +${System.currentTimeMillis() - ruleApplicationStartTime}ms")

            if (!isServiceActive) {
                AppLogger.d(TAG, "Service not active, skipping rule application")
                return@launch
            }

            val backend = currentBackend
            if (backend == null) {
                AppLogger.w(TAG, "Backend is null, cannot apply rules")
                return@launch
            }

            try {
                AppLogger.d(TAG, "🔥 [TIMING] Fetching rules from DB: +${System.currentTimeMillis() - ruleApplicationStartTime}ms")
                val rules = firewallRepository.getAllRules().first()
                AppLogger.d(TAG, "🔥 [TIMING] Rules fetched (${rules.size} rules): +${System.currentTimeMillis() - ruleApplicationStartTime}ms")

                AppLogger.d(TAG, "🔥 [TIMING] Applying rules to backend: network=$currentNetworkType, screen=$isScreenOn")
                val applyStartTime = System.currentTimeMillis()
                backend.applyRules(rules, currentNetworkType, isScreenOn).getOrElse { error ->
                    AppLogger.e(TAG, "🔥 [TIMING] Backend applyRules FAILED: +${System.currentTimeMillis() - ruleApplicationStartTime}ms, error=${error.message}")
                    return@launch
                }

                AppLogger.d(TAG, "🔥 [TIMING] Backend applyRules SUCCESS: backend took ${System.currentTimeMillis() - applyStartTime}ms, total +${System.currentTimeMillis() - ruleApplicationStartTime}ms")
            } catch (e: Exception) {
                AppLogger.e(TAG, "🔥 [TIMING] Exception while applying rules: +${System.currentTimeMillis() - ruleApplicationStartTime}ms", e)
            }
        }
    }
}

