//=========================================================================
//
//  Part of units package -- a Java version of GNU Units program.
//
//  Units is a program for unit conversion originally written in C
//  by Adrian Mariano (adrian@cam.cornell.edu.).
//  Copyright (C) 1996, 1997, 1999, 2000, 2001, 2002, 2003, 2004,
//  2005, 2006, 2007 by Free Software Foundation, Inc.
//
//  Java version Copyright (C) 2003, 2004, 2005, 2006, 2007, 2008,
//  2009 by Roman R Redziejowski (roman.redz@tele2.se).
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program. If not, see <http://www.gnu.org/licenses/>.
//
//-------------------------------------------------------------------------
//
//  Change log
//
//    050203 Created for version 1.84.J05.
//    050226 Version 1.84.J06.
//           Help text examples expanded and moved to 'Env'.
//    050315 Version 1.84.J07. Changed package name to "units".
//    061230 Version 1.86.J01.
//           Changed 'Reduce' button to 'Definition' and rearranged buttons.
//           Show reduced value when 'Convert' pressed with empty 'You want'.
//    061231 Pressing 'Definition' shows the definition only if 'You have'
//           is a function name or a possibly prefixed unit name or a prefix.
//           Modified the HELP text accordingly.
//           Moved table statistics to the end of 'About' text.
//    070102 Suppress printing "\tDefinition :" and tabs for compact output.
//    070103 Use 'Env.showAbout' to show info about units.
//           If 'Source' pressed for prefix+unit, show source for unit.
//    091024 Version 1.87.J01. Added serialVersionUID.
//    091028 Changed font in windows to Verdana.
//    091029 Specified margins and text wrapping for 'outArea'
//    091101 Added 'Search' button.
//
//=========================================================================

package units;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;


//HHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHH
//
//  Class GUI.
//
//HHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHH
/**
 *  Common graphic interface for applet and interactive invocation.
 */


public class GUI extends Container
implements ActionListener
{
  //=====================================================================
  //  Help text.
  //=====================================================================
  static final String HELP =
    " Units converts between different measuring systems.\n\n"
     + " Type the measure you want to convert in 'You have:' field.\n"
     + " Type the desired units in 'You want:' field.\n"
     + " Press 'Convert' or hit the return key.\n"
     + " The result will appear in the field below.\n"
     + " If you leave 'You want:' field empty, you obtain\n"
     + " your measure reduced to primitive units.\n"
     + " Press 'Conformable units' for a list of units to which\n"
     + " you can convert your measure.\n\n"
     + " Enter a unit name in 'You have:' field and press 'Definition'\n"
     + " to see the definition of that unit.\n"
     + " Press 'Source' to explore units database\n"
     + " around the definition of that unit.\n"
     + " Press 'Search' to find units whose names\n"
     + " contain the 'You have' string.\n\n";


  //=====================================================================
  //  Elements of GUI.
  //=====================================================================
  JTextField haveField = new JTextField();
  JTextField wantField = new JTextField();

  JTextArea outArea  = new JTextArea(10,10);
  JScrollPane scroll = new JScrollPane(outArea,
                           JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                           JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);

  JButton aboutKey   = new JButton("About Units");
  JButton clearKey   = new JButton("Clear");
  JButton conformKey = new JButton("Conformable units");
  JButton convertKey = new JButton("Convert");
  JButton helpKey    = new JButton("Help");
  JButton defKey     = new JButton("Definition");
  JButton searchKey  = new JButton("Search");
  JButton sourceKey  = new JButton("Source");


  //=====================================================================
  //  Writer.
  //=====================================================================
  class myOut extends Env.Writer
  {
    void print(final String s)   {outArea.append(s);}
    void println(final String s) {outArea.append(s + "\n");}
  }


  //=====================================================================
  //  Constructor
  //=====================================================================
  GUI(JRootPane rootPane)
    {
      setLayout(new BorderLayout());

      JPanel dialog = new JPanel(new GridLayout(2,0));
      add(dialog,BorderLayout.NORTH);

        JPanel havePart = new JPanel(new BorderLayout());
        dialog.add(havePart);
          JPanel haveLine = new JPanel(new BorderLayout());
          havePart.add(haveLine,BorderLayout.NORTH);
            haveLine.add(new JLabel("  "),BorderLayout.NORTH);
            haveLine.add(new JLabel(" "),BorderLayout.EAST);
            haveLine.add(new JLabel("  You have:  "),BorderLayout.WEST);
            haveLine.add(haveField,BorderLayout.CENTER);
          JPanel haveKeys = new JPanel(new FlowLayout(FlowLayout.RIGHT));
          havePart.add(haveKeys,BorderLayout.SOUTH);
            haveKeys.add(searchKey);
            haveKeys.add(conformKey);
            haveKeys.add(defKey);
            haveKeys.add(sourceKey);

        JPanel wantPart = new JPanel(new BorderLayout());
        dialog.add(wantPart);
          JPanel wantLine = new JPanel(new BorderLayout());
          wantPart.add(wantLine,BorderLayout.NORTH);
            wantLine.add(new JLabel(" "),BorderLayout.NORTH);
            wantLine.add(new JLabel(""),BorderLayout.EAST);
            wantLine.add(new JLabel("  You want:  "),BorderLayout.WEST);
            wantLine.add(wantField,BorderLayout.CENTER);
            wantLine.add(new JLabel(" "),BorderLayout.EAST);
          JPanel wantKeys = new JPanel(new FlowLayout(FlowLayout.RIGHT));
          wantPart.add(wantKeys,BorderLayout.SOUTH);
            wantKeys.add(convertKey);

      add(scroll,BorderLayout.CENTER);

      add(new JLabel(" "),BorderLayout.EAST);
      add(new JLabel(" "),BorderLayout.WEST);

      JPanel control = new JPanel(new FlowLayout(FlowLayout.RIGHT));
      add(control,BorderLayout.SOUTH);
        control.add(aboutKey);
        control.add(clearKey);
        control.add(helpKey);

      aboutKey.addActionListener(this);
      clearKey.addActionListener(this);
      conformKey.addActionListener(this);
      convertKey.addActionListener(this);
      defKey.addActionListener(this);
      helpKey.addActionListener(this);
      searchKey.addActionListener(this);
      sourceKey.addActionListener(this);

      rootPane.setDefaultButton(convertKey);

      haveField.setFont(new Font("Verdana",Font.PLAIN,12));
      wantField.setFont(new Font("Verdana",Font.PLAIN,12));
      outArea.setFont(new Font("Verdana",Font.PLAIN,12));
      outArea.setEditable(false);
      outArea.setMargin(new Insets(0,8,0,8));
      outArea.setLineWrap(true);
      outArea.setTabSize(3);

      //---------------------------------------------------------------
      //  Direct writers to window.
      //---------------------------------------------------------------
      Env.out = new myOut();
      Env.err = Env.out;

      //---------------------------------------------------------------
      //  Register GUI in Env.
      //---------------------------------------------------------------
      Env.gui = this;
    }


  //=====================================================================
  //  Get default button
  //=====================================================================
  JButton defaultButton()
    { return convertKey; }

  //=====================================================================
  //
  //   ActionListener interface
  //
  //=====================================================================

  public void actionPerformed(ActionEvent e)
    {
      String arg = e.getActionCommand();

      //===============================================================
      //  "Search" pressed
      //===============================================================
      if (arg.equals("Search"))
      {
        outArea.setText(null);

        String havestr = haveField.getText().trim();
        if (havestr.length()==0)
        {
          Env.out.println("Enter search string in 'You have' and try again.");
          return;
        }

        Tables.showMatching(havestr);
      }

      //===============================================================
      //  "Conformable units" pressed
      //===============================================================
      if (arg.equals("Conformable units"))
      {
        outArea.setText(null);

        String havestr = haveField.getText().trim();
        if (havestr.length()==0)
        {
          Env.out.println("Enter 'You have' expression and try again.");
          return;
        }

        Function func = (Function)DefinedFunction.table.get(havestr);
        if (func!=null)
        {
          Env.out.println("Add function argument and try again.");
          return;
        }

        Value have = Value.fromString(havestr);
        if (have==null)
        {
          Env.out.println("Correct 'You have' expression and try again.");
          return;
        }

        Tables.showConformable(have,havestr);
      }

      //===============================================================
      //  "Definition" pressed
      //===============================================================
      else if (arg.equals("Definition"))
      {
        // Clear output area.
        outArea.setText(null);

        // Get 'have' string or prompt for it if none.
        String havestr = haveField.getText().trim();
        if (havestr.length()==0)
        {
          Env.out.println("Enter 'You have' expression and try again.");
          return;
        }

        // If 'have' is function name, show function definition.
        Function func = (Function)DefinedFunction.table.get(havestr);
        if (func!=null)
        {
          Env.verbose = 0;
          Env.out.println(func.showdef());
          Env.verbose = 2;
          return;
        }

        // Check if 'have' is a possibly prefixed unit name.
        // write a message if not.
        Factor[] pu = Factor.split(havestr);
        if (pu==null)
        {
          Env.out.println("Unit '" + havestr + "' is unknown.");
          Env.out.println("Correct 'You have' expression and try again.");

          return;
        }

        // If a primitive unit, refer to units data base.
        if (pu[0]==null && pu[1].isPrimitive)
        {
          Env.out.println("This is a primitive unit. Press 'Source' to learn more.");
          return;
        }

        // Show definition
        Value have = Value.fromString(havestr);
        if (have==null)
        {
          Env.out.println("Correct 'You have' expression and try again.");
          return;
        }
        Factor.showdef(havestr);
        have.show();
      }

      //===============================================================
      //  "Source" pressed
      //===============================================================
      else if (arg.equals("Source"))
      {
        // Clear output area.
        outArea.setText(null);

        // Get 'have' string or prompt for it if none.
        String havestr = haveField.getText().trim();
        if (havestr.length()==0)
        {
          Env.out.println("Enter 'You have' expression and try again.");
          return;
        }

        // If 'have' is function name, show its source.
        Function func = (Function)DefinedFunction.table.get(havestr);
        if (func!=null)
        {
          Tables.showSource(havestr);
          return;
        }

        // Check if 'have' is a possibly prefixed unit name.
        // write a message if not.
        Factor[] pu = Factor.split(havestr);
        if (pu==null)
        {
          Env.out.println("Unit '" + havestr + "' is unknown.");
          Env.out.println("Correct 'You have' expression and try again.");

          return;
        }

        // If there is a unit, show its source.
        if (pu[1]!=null)
        {
          Tables.showSource(pu[1].name);
          return;
        }

        // Otherwise show prefix.
        Tables.showSource(pu[0].name);
      }

      //===============================================================
      //  "Conformable units" pressed
      //===============================================================
      else if (arg.equals("Conformable units"))
      {
        outArea.setText(null);

        String havestr = haveField.getText().trim();
        if (havestr.length()==0)
        {
          Env.out.println("Enter 'You have' expression and try again.");
          return;
        }

        Function func = (Function)DefinedFunction.table.get(havestr);
        if (func!=null)
        {
          Env.out.println("Add function argument and try again.");
          return;
        }

        Value have = Value.fromString(havestr);
        if (have==null)
        {
          Env.out.println("Correct 'You have' expression and try again.");
          return;
        }

        Tables.showConformable(have,havestr);
      }

      //===============================================================
      //  "Convert" pressed
      //===============================================================
      else if (arg.equals("Convert"))
      {
        outArea.setText(null);

        String havestr = haveField.getText().trim();
        if (havestr.length()==0)
        {
          Env.out.println("Enter 'You have' expression and try again.");
          return;
        }

        Value have = Value.fromString(havestr);
        if (have==null)
        {
          Env.out.println("Correct 'You have' expression and try again.");
          return;
        }

        String wantstr = wantField.getText().trim();
        if (wantstr.length()==0)
        {
          have.show();
          return;
        }

        Function func = (Function)DefinedFunction.table.get(wantstr);
        if (func!=null)
        {
          Value.convert(havestr,have,func);
          return;
        }

        Value want = Value.fromString(wantstr);
        if (want==null)
        {
          Env.out.println("Correct 'You want' expression and try again.");
          return;
        }

        Value.convert(havestr,have,wantstr,want);
      }

      //===============================================================
      //  "Clear" pressed
      //===============================================================
      else if (arg.equals("Clear"))
      {
        haveField.setText(null);
        wantField.setText(null);
        outArea.setText(null);
      }

      //===============================================================
      //  "About Units" pressed
      //===============================================================
      else if (arg.equals("About Units"))
      {
        outArea.setText("");
        Env.showAbout();
        scroll.getViewport().setViewPosition(new Point(0,0));
        outArea.select(0,0);
      }

      //===============================================================
      //  "Help" pressed
      //===============================================================
      else if (arg.equals("Help"))
        Browser.show("Units help", HELP+Env.EXAMPLES,0,0,false);

    }


  //=====================================================================
  //  Serial version UID. Unused - defined to eliminate compiler warning
  //=====================================================================
  static final long serialVersionUID = 4711L;
}


