package pageroute

import (
	"fmt"
	"regexp"
	"strings"
	"time"

	"gioui.org/layout"
	"gioui.org/unit"
	"gioui.org/widget/material"
	"git.sr.ht/~mil/transito/src/components/divider"
	"git.sr.ht/~mil/transito/src/util/utiltheme"
)

// Used to match a valid (possibly partial) time.
var NUMBER *regexp.Regexp = regexp.MustCompile(`\d`)

// Widgets to control route parameters.
func (page *PageRoute) widgetParams() []layout.Widget {
	widgetCheckboxesTransfers := func(gtx layout.Context) layout.Dimensions {
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(40))
		gtx.Constraints.Max.Y = gtx.Constraints.Min.Y
		return layout.Flex{}.Layout(gtx, layout.Rigid(dimensionsToWidget(layout.Flex{}.Layout(
			gtx,
			layout.Flexed(0.2, material.CheckBox(page.GlobalState.Theme, &page.checkboxTransfersFeed, "Feed\nTransfers").Layout),
			layout.Flexed(0.01, layout.Spacer{Width: unit.Dp(2)}.Layout),
			layout.Flexed(0.2, material.CheckBox(page.GlobalState.Theme, &page.checkboxTransfersImplicit, "Implicit\nTransfers").Layout),
			layout.Flexed(0.01, layout.Spacer{Width: unit.Dp(2)}.Layout),
			layout.Flexed(0.2, material.CheckBox(page.GlobalState.Theme, &page.checkboxTransfersGenerated, "Generated\nTransfers").Layout),
		))))
	}

	// Route params
	widgetTextfieldMaxWalkMinutes := func(gtx layout.Context) layout.Dimensions {
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(15))
		page.textfieldMaxWalkMinutes.SingleLine = true
		return page.textfieldMaxWalkMinutes.Layout(gtx, page.GlobalState.Theme, "Max Wm")
	}
	widgetTextfieldMaxTripMinutes := func(gtx layout.Context) layout.Dimensions {
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(15))
		page.textfieldMaxTripMinutes.SingleLine = true
		return page.textfieldMaxTripMinutes.Layout(gtx, page.GlobalState.Theme, "Max Tm")
	}
	widgetTextfieldMinTransferMinutes := func(gtx layout.Context) layout.Dimensions {
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(15))
		page.textfieldMinTransferMinutes.SingleLine = true
		return page.textfieldMinTransferMinutes.Layout(gtx, page.GlobalState.Theme, "Min Cm")
	}
	widgetTextfieldMaxTransferMinutes := func(gtx layout.Context) layout.Dimensions {
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(15))
		page.textfieldMaxTransferMinutes.SingleLine = true
		return page.textfieldMaxTransferMinutes.Layout(gtx, page.GlobalState.Theme, "Max Cm")
	}
	widgetTextfieldWalkspeedKmHr := func(gtx layout.Context) layout.Dimensions {
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(15))
		page.textfieldWalkspeedKmHr.SingleLine = true
		return page.textfieldWalkspeedKmHr.Layout(gtx, page.GlobalState.Theme, "Wkmhr")
	}

	widgetRouteNumbersRow := func(gtx layout.Context) layout.Dimensions {
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(40))
		gtx.Constraints.Max.Y = gtx.Constraints.Min.Y
		return layout.Flex{}.Layout(gtx, layout.Rigid(dimensionsToWidget(layout.Flex{}.Layout(
			gtx,
			layout.Flexed(0.2, widgetTextfieldMaxWalkMinutes),
			layout.Flexed(0.2, widgetTextfieldMaxTripMinutes),
			layout.Flexed(0.2, widgetTextfieldMinTransferMinutes),
			layout.Flexed(0.2, widgetTextfieldMaxTransferMinutes),
			layout.Flexed(0.2, widgetTextfieldWalkspeedKmHr),
		))))
	}

	widgetScheduleTime := func(gtx layout.Context) layout.Dimensions {
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(40))
		gtx.Constraints.Max.Y = gtx.Constraints.Min.Y

		widgets := []layout.FlexChild{
			layout.Flexed(0.5, page.widgetDepartureNow),
			layout.Flexed(0.5, page.widgetDepartureAt),
		}
		if page.scheduleType.Value == SCHEDULE_DEPART_AT {
			// E.g. show time if not departing at specific time
			widgets = append(widgets, layout.Flexed(0.5, page.widgetDepartureTime))
		} else {
			widgets = append(widgets, layout.Flexed(0.5, blankWidget))
		}

		return layout.Flex{}.Layout(gtx,
			layout.Rigid(dimensionsToWidget(layout.Flex{}.Layout(gtx, widgets...))),
		)
	}

	// E.g. show day if not departing at specific time
	widgetScheduleDay := blankWidget
	if page.scheduleType.Value == SCHEDULE_DEPART_AT {
		widgetScheduleDay = func(gtx layout.Context) layout.Dimensions {
			gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(40))
			gtx.Constraints.Max.Y = gtx.Constraints.Min.Y
			return layout.Flex{}.Layout(gtx, layout.Rigid(dimensionsToWidget(layout.Flex{
				Alignment: layout.Middle,
			}.Layout(
				gtx,
				layout.Flexed(0.3, page.widgetBtnDayPrev),
				layout.Flexed(0.5, page.widgetDay),
				layout.Flexed(0.3, page.widgetBtnDayNext),
			))))
		}
	}

	return []layout.Widget{
		divider.Divider(page.GlobalState.Theme).Layout,
		material.Body1(page.GlobalState.Theme, "Parameters").Layout,
		widgetRouteNumbersRow,
		widgetCheckboxesTransfers,
		widgetScheduleTime,
		widgetScheduleDay,
	}
}

func (page *PageRoute) widgetDepartureNow(gtx layout.Context) layout.Dimensions {
	var color = utiltheme.ThemecolorDefault
	return material.RadioButton(
		utiltheme.MaterialTheme(*page.GlobalState.Theme, color, 15),
		&page.scheduleType,
		SCHEDULE_DEPART_NOW,
		"Departing\nNow",
	).Layout(gtx)
}

func (page *PageRoute) widgetDepartureAt(gtx layout.Context) layout.Dimensions {
	var color = utiltheme.ThemecolorDefault
	return material.RadioButton(
		utiltheme.MaterialTheme(*page.GlobalState.Theme, color, 15),
		&page.scheduleType,
		SCHEDULE_DEPART_AT,
		"Departing\nAt…",
	).Layout(gtx)
}

func (page *PageRoute) widgetDepartureTime(gtx layout.Context) layout.Dimensions {
	color := utiltheme.ThemecolorDefault
	if page.scheduleType.Value == SCHEDULE_DEPART_NOW {
		page.textfieldDepartureTime.SetText(time.Now().Format("15:04"))
		page.textfieldDepartureTime.ClearError()
		page.textfieldDepartureTime.ReadOnly = true
		color = utiltheme.ThemecolorDisabled
	} else if page.textfieldDepartureTime.Text() == "" {
		page.textfieldDepartureTime.ReadOnly = false
		page.textfieldDepartureTime.ClearError()
	} else if page.textfieldDepartureTime.IsActive() {
		page.textfieldDepartureTime.ReadOnly = false
		// Put the colon in the right place, for it to look like a valid
		// HH:MM or H:MM. In ambiguous cases, assume incomplete input.

		orig := page.textfieldDepartureTime.Text()
		matches := strings.Join(NUMBER.FindAllString(orig, 4), "")
		var text string
		length := len(matches)
		switch {
		case length >= 4:
			text = fmt.Sprintf("%s:%s", matches[:2], matches[2:4])
		case length == 3:
			text = fmt.Sprintf("%s:%s", matches[:1], matches[1:])
		case length == 2:
			text = matches[:2]
		case length == 1:
			text = matches[0:1]
		default:
			text = ""

		}

		if text != orig {
			// If we're shifting content, move caret accordingly.
			offset := len(text) - len(orig)

			caretStart, caretEnd := page.textfieldDepartureTime.Selection()
			page.textfieldDepartureTime.SetText(text)
			page.textfieldDepartureTime.SetCaret(caretStart, caretEnd)
			if offset != 0 {
				page.textfieldDepartureTime.MoveCaret(offset, offset)
			}

			_, err := time.Parse("15:04", page.textfieldDepartureTime.Text())
			if err != nil {
				page.textfieldDepartureTime.SetError(err.Error())
			} else {
				page.textfieldDepartureTime.ClearError()
			}
		}
	}

	gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(15))
	page.textfieldDepartureTime.SingleLine = true
	return page.textfieldDepartureTime.Layout(gtx, utiltheme.MaterialTheme(*page.GlobalState.Theme, color, 15), "Departure")
}

func (page *PageRoute) widgetBtnDayPrev(gtx layout.Context) layout.Dimensions {
	if page.clickableDayPrev.Clicked(gtx) {
		page.scheduleDay = page.scheduleDay.Add(-24 * time.Hour)
	}

	var label = "« Previous"
	var color = utiltheme.ThemecolorFgBlackBgGray

	return material.Button(
		utiltheme.MaterialTheme(*page.GlobalState.Theme, color, 15),
		&page.clickableDayPrev,
		label,
	).Layout(gtx)
}

func (page *PageRoute) widgetDay(gtx layout.Context) layout.Dimensions {
	var label string

	// TODO: should use current locale
	todayStr := time.Now().Format("Mon 2006-01-02")
	departureDayStr := page.scheduleDay.Format("Mon 2006-01-02")
	if todayStr == departureDayStr {
		label = "Today"
	} else {
		label = departureDayStr

	}

	var color = utiltheme.ThemecolorFgBlackBgGray

	labelWidget := material.Label(
		utiltheme.MaterialTheme(*page.GlobalState.Theme, color, 15),
		page.GlobalState.Theme.TextSize,
		label,
	)
	return layout.Center.Layout(gtx, labelWidget.Layout)
}

func (page *PageRoute) widgetBtnDayNext(gtx layout.Context) layout.Dimensions {
	if page.clickableDayNext.Clicked(gtx) {
		page.scheduleDay = page.scheduleDay.Add(24 * time.Hour)
	}

	var label = "Next »"
	var color = utiltheme.ThemecolorFgBlackBgGray

	return material.Button(
		utiltheme.MaterialTheme(*page.GlobalState.Theme, color, 15),
		&page.clickableDayNext,
		label,
	).Layout(gtx)
}
